/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.adaeditor;

import java.util.HashMap;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.TextSelection;
import org.eclipse.jface.text.TextViewer;
import org.eclipse.ui.texteditor.TextEditorAction;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstruct;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstructCutFilter;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstructFilter;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedLocation;
import com.adacore.gnatbench.core.internal.analyzer.IAdaConstructFilterProvider;
import com.adacore.gnatbench.core.internal.codingstyle.AdaIndentBuffer;
import com.adacore.gnatbench.library.Language.Language_Category;

public class AdaActionInsertEnd extends TextEditorAction {

	private AdaEditor editor;
	final protected Map<Integer, Description> blockDescriptors = setBlockDescriptions();
    private AdaEditorTextUtils textManager;


 	public AdaActionInsertEnd(ResourceBundle bundle, String prefix, AdaEditor editor) {
		super(bundle, prefix, editor);
		this.editor = editor;
	} // ctor


	public void run() {
		final GeneralizedLocation currentLocation = editor.getCurrentLocation();

		final int currentLine = currentLocation.getLine() - 1;

		// force analysis so that latest changes are guaranteed to be recognized
		editor.getAnalyzer().analyze();

		AdaConstruct enclosingConstruct = editor.getAnalyzer().getConstructAt(
				currentLocation,
				new ClosableConstructs());

		if (enclosingConstruct == null) {
			return;
		} // if

	    final Integer blockType = new Integer(enclosingConstruct.getCategory());

	    if (!blockDescriptors.containsKey(blockType)) {
			return;
		} // if

		try {
			textManager = new AdaEditorTextUtils(editor);

		    Description descriptor = (Description) blockDescriptors.get(blockType);

		    final String closingLine = getClosingLine(descriptor, enclosingConstruct);

		    textManager.insertText(closingLine);

		    indentLine(currentLine, closingLine.length());

		    // we go to the end of this line in case they want to insert a comment or
		    // anything else, for that matter.
		    textManager.gotoEOL(currentLine);

		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} // try
	} // run



	protected String getClosingLine
		(Description descriptor, final AdaConstruct construct)
		throws BadLocationException
	{
		String result = descriptor.terminal;

        String line = textManager.getLine(construct.getLineBegin());

        if (!descriptor.pattern.equals("")) {
        	// a name or label may be present

    		Pattern re_pattern = Pattern.compile(descriptor.pattern,
    				Pattern.CASE_INSENSITIVE | Pattern.DOTALL);
    		Matcher match = re_pattern.matcher(line);

    		if (match.matches()) {
    			result = match.replaceFirst(descriptor.terminal);
    		} else {
    			// The pattern does not match the content.
    			// This is possible with loops and block stmnts without a label
    			// on the same line, so we look on the line above for a label.
    			// There's no guarantee it is one line above either, but that's
    			// as far as we're willing to search.
    			line = textManager.getLine(construct.getLineBegin() - 1);
    			final String label = "\\s*([^ ]+)\\s*:.*";
    			re_pattern = Pattern.compile(label, Pattern.DOTALL);
    			match = re_pattern.matcher(line);

    			if (match.matches()) {
    				result = match.replaceFirst(descriptor.terminal);
    			} else {
    				// give up searching for the label and remove the tags
    				result = result.replaceFirst (" \\$1", "");
    				result = result.replaceFirst ("\\$1", "");
    				result = result.replaceFirst (" \\$2", "");
    				result = result.replaceFirst ("\\$2", "");
    			} // if
    		} // if
		} // if

		return result;
	} // getClosingLine


	protected void indentLine(final int lineNum, final int length) throws BadLocationException {
		final IDocument doc = editor.getDocument();

		final TextSelection selection = new TextSelection(
				doc,
				doc.getLineOffset(lineNum),
				length);

		AdaIndentBuffer indentAction = new AdaIndentBuffer(editor.getDocumentBuffer());

		TextSelection newSelection = indentAction.format(selection, false);

		((TextViewer) editor.getSourceViewerPublic()).setSelectedRange(
				newSelection.getOffset(), newSelection.getLength());

	} // indentLine


	protected class ClosableConstructs implements IAdaConstructFilterProvider {

		public boolean isClosable(AdaConstruct element) {
			int category = element.getCategory();
			return category == Language_Category.Cat_Package
			    || category == Language_Category.Cat_If_Statement
			    || category == Language_Category.Cat_Task
			    || category == Language_Category.Cat_Protected
			    || category == Language_Category.Cat_Procedure
			    || category == Language_Category.Cat_Function
			    || category == Language_Category.Cat_Structure
			    || category == Language_Category.Cat_Class
		  	    || category == Language_Category.Cat_Loop_Statement
			    || category == Language_Category.Cat_Case_Statement
			    || category == Language_Category.Cat_Select_Statement
			    || category == Language_Category.Cat_Accept_Statement
			    || category == Language_Category.Cat_Declare_Block
			    || category == Language_Category.Cat_Simple_Block;
		} // isClosable

		public AdaConstructFilter getFilter() {
			return new AdaConstructCutFilter () {
				public boolean simpleFilter(AdaConstruct construct) {
					return isClosable (construct);
				}
			};
		} // getFilter

	} // ClosableConstructs


	protected class Description {

		public String terminal;
		public String pattern;

		public Description(final String term, final String pattern) {
			this.pattern = pattern;
			this.terminal = term;
		} // ctor

	} // Description


	protected Map<Integer, Description> setBlockDescriptions() {
		HashMap<Integer, Description> result = new HashMap<Integer, Description>();
		result.put(
				new Integer(Language_Category.Cat_If_Statement),
				new Description("end if;", ""));
		result.put(
				new Integer(Language_Category.Cat_Case_Statement),
				new Description("end case;", ""));
		result.put(
				new Integer(Language_Category.Cat_Structure),
				new Description("end record;", ""));
		result.put(
				new Integer(Language_Category.Cat_Class),
				new Description("end record;", ""));
		result.put(
				new Integer(Language_Category.Cat_Select_Statement),
				new Description("end select;", ""));
		result.put(
				new Integer(Language_Category.Cat_Package),
				new Description("end $2;", "\\s*package\\s+(body\\s+)?([^ \\n]+).*"));
		result.put(
				new Integer(Language_Category.Cat_Loop_Statement),
				new Description("end loop $1;", "\\s*([^ ]+)\\s*:(.*)?\\s*loop.*"));
		result.put(
				new Integer(Language_Category.Cat_Procedure),
				new Description("end $1;", "\\s*procedure\\s+([^ \\n]+).*"));
		result.put(
				new Integer(Language_Category.Cat_Function),
				new Description("end $1;", "\\s*function\\s+([^ \\n]+).*"));
		result.put(
				new Integer(Language_Category.Cat_Task),
				new Description("end $2;", "\\s*task\\s+(body\\s+)?([^ \\n]+).*"));
		result.put(
				new Integer(Language_Category.Cat_Protected),
				new Description("end $2;", "\\s*protected\\s+(body\\s+)?([^ \\n]+).*"));
		result.put(
				new Integer(Language_Category.Cat_Declare_Block),
				new Description("end $1;", "\\s*([^ ]+)\\s*:\\s*declare.*"));
		result.put(
				new Integer(Language_Category.Cat_Simple_Block),
				new Description("end $1;", "\\s*([^ ]+)\\s*:\\s*begin.*"));
		result.put(
				new Integer(Language_Category.Cat_Accept_Statement),
				new Description("end $1;", "\\s*accept\\s+([^ \\n\\(]+).*"));

		return result;
	} // setBlockDescriptions


} // AdaActionInsertEnd