/*******************************************************************************
 * Copyright (C) 2006-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.adaeditor;

import java.io.File;
import java.util.ResourceBundle;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.ui.progress.UIJob;
import org.eclipse.ui.texteditor.TextEditorAction;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedFile;
import com.adacore.gnatbench.core.internal.utils.ProjectUtils;
import com.adacore.gnatbench.ui.internal.GNATbenchUIPlugin;
import com.adacore.gnatbench.ui.internal.jobs.GNATstubJob;

public class AdaActionGenerateBody extends TextEditorAction {

	protected AdaEditor fEditor;

	public AdaActionGenerateBody(ResourceBundle bundle, String prefix,
			AdaEditor editor) {
		super(bundle, prefix, editor);

		fEditor = editor;
	} // ctor

	public void run() {
		final IProject project = fEditor.getFile().getProject();

		final String inputFileName = fEditor.getFile().getFile().getName();

		String location = fEditor.getFile().getFile().getLocation().toString();
		location = location.substring(0, location.lastIndexOf("/"));

		final String specSuffix = ProjectUtils.specSuffixAda(project);
		final String bodySuffix = ProjectUtils.bodySuffixAda(project);

		final String bodyFileName = basename(inputFileName) + bodySuffix;

		if (!inputFileName.endsWith(specSuffix)) {
			MessageDialog
				.openError(
					null,
					"Generate Package Body",
					"The file name extension of '"
					+ inputFileName
					+ "' is not consistent with a file containing a package declaration.");
			return;
		} // if

		final File file = new File(location + "/" + bodyFileName);

		if (file.exists()) {
			boolean overwrite = MessageDialog
				.openConfirm(
					null,
					"Generate Package Body",
					"File '"
					+ bodyFileName
					+ "' already exists.\nIs it OK to overwrite it with a new body?");
			// yes, we could just test the result of the messageDialog but
			// this is clearer.
			if (!overwrite) {
				return;
			} // if
		} // if

		if (fEditor.isDirty()) {
			fEditor.doSave(null);
		} // if

		GNATstubJob job = new GNATstubJob(location, inputFileName, bodyFileName, project);
		job.schedule();

		// if gnatstub succeeds, open the new file using a listener
		job.addJobChangeListener(new NewFileJobAdapter(location, bodyFileName, project));

	} // run

	// This class listens for the job to complete and then opens the
	// newly-created file in the editor.
	public class NewFileJobAdapter extends JobChangeAdapter {

		private String newFile;
		private String location;
		private IProject project;

		public NewFileJobAdapter(String location, String newFile, IProject project) {
			super();
			this.newFile = newFile;
			this.location = location;
			this.project = project;
		} // ctor

		public void done(IJobChangeEvent event) {
			final File file = new File(location + "/" + newFile);
			if (!file.exists()) {
				return;
			} // if

			// tell the workspace the new file is here
			try {
				project.refreshLocal(IResource.DEPTH_INFINITE, null);
			} catch (CoreException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
				return;
			} // try

			GeneralizedFile bodyFile = GeneralizedFile.fromOSPath(null,
					location + "/" + newFile);

			new OpenEditorJob(bodyFile).schedule();
		} // done

	} // NewFileJobAdapter

	// Open the newly-created package body file for editing. This has to be done
	// in a UIJob because otherwise the call to openEditor will fail because
	// it will not get be able to get the active display value and will return
	// null instead, leading to a nullpointer exception.
	public class OpenEditorJob extends UIJob {

		protected GeneralizedFile file;

		public OpenEditorJob(GeneralizedFile file) {
			super("Opening New Ada File");
			this.file = file;
		} // ctor

		public IStatus runInUIThread(IProgressMonitor monitor) {
			try {
				GNATbenchUIPlugin.getDefault().openEditor(file);
			} catch (Throwable t) {
				return new Status(IStatus.ERROR, GNATbenchUIPlugin.getId(),
						IStatus.OK, "Command failed due to an exception", null);
			} // try

			return new Status(IStatus.OK, GNATbenchUIPlugin.getId(),
					IStatus.OK, "", null);
		} // runInUIThread

	} // OpenEditorJob

	protected String basename(String input) {
		int dot = input.indexOf('.');
		int sep = input.lastIndexOf('/');
		return input.substring(sep + 1, dot);
	} // basename

} // AdaActionGenerateBody

