/*******************************************************************************
 * Copyright (C) 2005-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.LinkedList;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorDescriptor;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.GNATbenchProjectProperties;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedFile;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedLocation;
import com.adacore.gnatbench.core.internal.filesystem.FileStoreReference;
import com.adacore.gnatbench.ui.internal.adaeditor.AdaEditor;
import com.adacore.gnatbench.ui.internal.adaeditor.AdaSyntaxHighlighting;
import com.adacore.gnatbench.ui.internal.codingstyle.AdaCodingStylePreferencesProvider;
import com.adacore.gnatbench.ui.internal.filesystem.FileStoreWorkbenchAdapterFactory;
import com.adacore.gnatbench.ui.internal.quickfix.ErrorMarkerSpy;


/**
 * The main plugin class to be used in the desktop.
 */
public class GNATbenchUIPlugin
	extends AbstractUIPlugin
	implements IPropertyChangeListener {

	//The shared instance.
	private static GNATbenchUIPlugin plugin;

	//Resource bundle.
	private ResourceBundle resourceBundle;

	private AdaSyntaxHighlighting codeScanner;

	private Shell fHiddenShell;

	public static final String GENERIC_ERROR_ICON = "generic_error_message";

	public static final String ADACORE_LOGO = "adacore_logo";

	final static public String ICON_FOLDER = "icon_folder";
	final static public String ICON_ADA_FILE = "icon_ada_file";
	final static public String ICON_FILE = "icon_file";
	final static public String ICON_MATCH = "icon_search";
	final static public String ICON_PROJECT = "icon_project";
	final static public String ICON_SOURCE_FOLDER = "icon_source_folder";
	final static public String ICON_LIBRARY = "icon_library";
	final static public String ICON_FOREIGN_PRJ = "icon_foreign_prj";
	final static public String ICON_BINARY_FOLDER = "icon_binary_folder";


	/**
	 * This variable is used for debug purpose, in order to store multiple time
	 * stamps.
	 */
	private LinkedList <Long> fTimes = new LinkedList <Long> ();

	/**
	 * The constructor.
	 */
	public GNATbenchUIPlugin() {
		super();

		plugin = this;
		try {
			resourceBundle = ResourceBundle
					.getBundle("com.adacore.gnatbench.ui.internal.adaeditor.AdaEditorMessages");
		} catch (MissingResourceException x) {
			GNATbenchCorePlugin.getDefault().logError("", x);
			resourceBundle = null;
		}

		AdaCodingStylePreferencesProvider
				.refreshIndentParameters(GNATbenchCorePlugin.getDefault());

		//  To improve somehow...
		ResourcesPlugin.getWorkspace().addResourceChangeListener(
				new ErrorMarkerSpy ());

		Platform.getAdapterManager()
				.registerAdapters(new FileStoreWorkbenchAdapterFactory(),
						FileStoreReference.class);
	}

	/**
	 * Adds an image to the image registery. The image is then stored during the
	 * whole life of the plugin. Never create any image directly (will cost a
	 * lot of memory
	 *
	 * @param key
	 * @param basePath
	 * @param name
	 */
	public synchronized void addImage(String key, String basePath, String name) {
		getImageRegistry ().put(key, createImage(basePath, name));
	}

	/**
	 * Returns an image that has been set by addImage.
	 *
	 * @param key
	 * @return
	 */
	public synchronized Image getImage(String key) {
		return getImageRegistry ().get(key);
	}

	public synchronized ImageDescriptor getImageDescriptor(String key) {
		return getImageRegistry ().getDescriptor(key);
	}

	/**
	 * Returns a descriptor based on a path and a name.
	 *
	 * @param basePath
	 * @param name
	 * @return
	 */
	public synchronized ImageDescriptor getImageDescriptor(String basePath, String name) {
		URL baseURL;

		try {
			baseURL = new URL(getBundle().getEntry(basePath), name);
		} catch (MalformedURLException e) {
			e.printStackTrace();
			return null;
		}

		return ImageDescriptor.createFromURL(baseURL);
	}

	/**
	 * Create an image based on a path and a name. If such an image cannot be
	 * found, then returns null.
	 *
	 * @param basePath
	 * @param name
	 * @return
	 */
	private Image createImage(String basePath, String name) {
		return getImageDescriptor(basePath, name).createImage();
	}

	/**
	 * This method is called upon plug-in activation
	 */
	@Override
	public void start(BundleContext context) throws Exception {
		super.start(context);

		addImage(ADACORE_LOGO, "img/", "adacore_logo.gif");
		addImage(ICON_FOLDER, "/img/", "fldr_obj.gif");
		addImage(ICON_ADA_FILE, "/img/", "ada_file_obj.gif");
		addImage(ICON_FILE, "/img/", "file_obj.gif");
		addImage(ICON_MATCH, "/img/", "searchm_obj.gif");
		addImage(ICON_PROJECT, "/img/", "prj_obj.gif");
		addImage(ICON_SOURCE_FOLDER, "/img/", "packagefolder_obj.gif");
		addImage(ICON_LIBRARY, "/img/", "lib_obj.gif");
		addImage(ICON_FOREIGN_PRJ, "/img/", "foreign_dep_obj.gif");
		addImage(ICON_BINARY_FOLDER, "/img/", "binary_folder_obj.gif");

		Display.getDefault().syncExec(new Runnable () {

			public void run() {
				PlatformUI.getWorkbench().getThemeManager().getCurrentTheme()
				.addPropertyChangeListener(GNATbenchUIPlugin.this);
			}
		});

		new Job ("Initialize GNATbench plugin") {

			@Override
			protected IStatus run(IProgressMonitor monitor) {
				//  Forces the load of xrefs database
				IProject[] projects = ResourcesPlugin.getWorkspace().getRoot()
						.getProjects();

				for (int j = 0; j < projects.length; ++j) {
					if (projects [j].isOpen()) {
						GNATbenchProjectProperties props = GNATbenchProjectProperties
						.getPropertiesFor(projects[j]);

						try {
							if (props.isRootProject()) {
								// Force project loading

								GNATbenchSession.getDefault().getOrLoadRegistry(projects[j]);
							}
						} catch (Throwable t) {
							createProjectLoadMarker(projects[j]);
							GNATbenchCorePlugin.getDefault().logError(
									"Failure to load kernel for project '" +
									projects[j].getName() + "'",
									t);
						}
					}
				}

				return GNATbenchCorePlugin.OK_STATUS;
			}
		}.schedule ();
	}

	protected void createProjectLoadMarker(final IProject project) {
		final IMarker marker;
		try {
			marker = project.createMarker(GNATbenchCorePlugin.GPR_ERROR_MARKER);
			marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
			marker.setAttribute(IMarker.TRANSIENT, true);
			marker.setAttribute(IMarker.MESSAGE,
					"The project \"" + project.getName() + "\" cannot be loaded");
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}
	}

	/**
	 * Returns the shared instance.
	 */
	public static GNATbenchUIPlugin getDefault() {
		return plugin;
	}

	/**
	 * Returns the string from the plugin's resource bundle, or 'key' if not
	 * found.
	 */
	public static String getResourceString(String key) {
		ResourceBundle bundle = GNATbenchUIPlugin.getDefault().getResourceBundle();
		try {
			return (bundle != null) ? bundle.getString(key) : key;
		} catch (MissingResourceException e) {
			return key;
		}
	}

	/**
	 * Returns the plugin's resource bundle,
	 */
	public ResourceBundle getResourceBundle() {
		return resourceBundle;
	}

	/**
	 *
	 * @return the code scanner shared by all editors.
	 */
	public AdaSyntaxHighlighting getCodeScanner() {
		if (codeScanner == null) {
			codeScanner = new AdaSyntaxHighlighting();
		}

		return codeScanner;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.jface.util.IPropertyChangeListener#propertyChange(org.eclipse.jface.util.PropertyChangeEvent)
	 */
	public void propertyChange(PropertyChangeEvent event) {
		if (event.getProperty().indexOf("com.adacore") != -1) {
			if (codeScanner != null) {
				codeScanner.updateFromPreferences();
			}
		}
	}

	/**
	 * Opens an editor, according to a precise location. If such an editor is
	 * already opened, it will just move the cursor to the corresponding
	 * location.
	 *
	 * @param location
	 * @param from
	 *            the Ada editor that originally made this open to appen, null
	 *            if none.
	 * @return
	 * @throws PartInitException
	 */
	public IEditorPart openEditor(GeneralizedLocation location)
			throws PartInitException {

		IEditorPart newEditor = openEditor(location.getFile());

		if (newEditor instanceof AdaEditor) {
			((AdaEditor) newEditor).focus(location);
		}

		return newEditor;
	}

	/**
	 * Same as before, but doesn't do focus.
	 *
	 * @param file
	 * @return
	 * @throws PartInitException
	 */
	public IEditorPart openEditor(GeneralizedFile file)
			throws PartInitException {
		if (file == null)
			return null;

		Path thePath = new Path(file.getOSPath());

		IFile theFile = file.getFile();

		IEditorPart newEditor = null;

		IEditorDescriptor editorDesc = getWorkbench().getEditorRegistry()
				.getDefaultEditor(file.getOSPath());

		if (theFile == null || !theFile.exists()) {
			java.io.File f = new java.io.File (thePath.toOSString());

			if (!f.exists() || !f.isFile()) {
				return null;
			}

			newEditor = PlatformUI.getWorkbench().getActiveWorkbenchWindow()
					.getActivePage().openEditor(file.getInput(),
							editorDesc.getId());
		} else if (editorDesc != null){
			newEditor = IDE.openEditor(PlatformUI.getWorkbench()
					.getActiveWorkbenchWindow().getActivePage(), theFile,
					editorDesc.getId());
		} else {
			newEditor = IDE.openEditor(PlatformUI.getWorkbench()
					.getActiveWorkbenchWindow().getActivePage(), theFile);
		}

		return newEditor;
	}

	/**
	 * This function forces all open ada editors to be refreshed, updating the
	 * latest display preferences.
	 *
	 */
	public void refreshEditors() {
		IWorkbenchPage page = PlatformUI.getWorkbench()
				.getActiveWorkbenchWindow().getActivePage();

		if (page == null) {
			return;
		}

		IEditorReference[] editors = page.getEditorReferences();

		for (int i = 0; i < editors.length; i++) {
			IEditorPart editor = editors[i].getEditor(false);
			if (!(editor instanceof AdaEditor))
				continue;

			((AdaEditor) editor).refreshWidget();
		}
	}

	/**
	 * This functions is used on debug purpose. It store a time stamp.
	 */
	public void pushTime() {
		fTimes.add(new Long((new java.util.Date()).getTime()));
	}

	/**
	 * This functions is used on debug purpose. It get the last time stamp and
	 * display the difference with the current date.
	 */
	public void popTime(String label) {
		Long lastTime = fTimes.getLast();
		fTimes.removeLast();
		long timeSpent = (new java.util.Date()).getTime()
				- lastTime.longValue();

		System.out.println(label + ": " + timeSpent);
	}

	/**
	 *
	 * @return The current id of the plugin.
	 */
	static public String getId() {
		return "com.adacore.gnatbench.ui";
	}

	public AdaEditor getActiveEditor () {
		IWorkbenchWindow workbenchWindow = PlatformUI.getWorkbench().getActiveWorkbenchWindow();

		if (workbenchWindow == null) return null;

		IWorkbenchPage page = workbenchWindow.getActivePage();

		if (page == null) return null;

		IEditorPart editor = page.getActiveEditor();

		if (editor == null || !(editor instanceof AdaEditor)) return null;

		return (AdaEditor) editor;
	}

	public void hideEclipse() {
		fHiddenShell = PlatformUI.getWorkbench().getDisplay().getActiveShell();
		fHiddenShell.setVisible(false);
	}

	public void showEclipse() {
		fHiddenShell.setVisible(true);
	}

	public String getPluginPath() {
		try {
			String path = FileLocator.toFileURL(getBundle().getEntry("/"))
					.getPath();

			if (Platform.getOS().equals("win32")) {
				// Transforms path like /c:/ into c:/
				return path.substring(1);
			} else {
				return path;
			}
		} catch (IOException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return "";
		}
	}

}
