/*******************************************************************************
 * Copyright (C) 2008-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.spark.tools;

import java.util.LinkedList;
import java.util.TreeMap;
import java.util.Map.Entry;

import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;

import com.adacore.gnatbench.core.GNATbenchCoreException;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.filesystem.FileStoreRegistry;
import com.adacore.gnatbench.core.internal.gpswrappers.GPRProject;
import com.adacore.gnatbench.core.internal.jobs.CommandFilesJob;
import com.adacore.gnatbench.core.projects.IGNATProjectView;
import com.adacore.gnatbench.spark.SPARKToolChain;
import com.adacore.gnatbench.spark.SPARKToolMessage;

/**
 * This class manages the commands for SPARKmake
 */
public class POGSJob extends CommandFilesJob {

	final private static String name = "POGS";
	private String directory = null;
	private String sumPath = null;

	public POGSJob(IFileStore resource) {
		super(name, SPARKToolMessage.class, resource, true);
		IProject project = FileStoreRegistry.getDefaultProject(resource);
		POGSDirectory(project);
		getSumPath(project);
	} // ctor

	/**
	 * Retrieve the path of the sum file
	 * if not specified in the spark POGS switches,
	 * it will be in the project output folder
	 * @param project
	 */
	private void getSumPath(IProject project){
		final String[] switches = SPARKToolChain.getSPARKAttribute(project, "Default_Switches", "IDE", "pogs");
		for (int j=0; j < switches.length; j++){
			if (switches[j].startsWith("o=", 1)){
				// if the path is absolute, we keep it as is
				if ((switches[j].substring(3, 4).equals(System.getProperty("file.separator")))){
					sumPath = new String(switches[j].substring(3));
					return;
				} else
				// if the path contains a separator, we add the project's path
				if (switches[j].contains(System.getProperty("file.separator"))){
					sumPath = project.getLocation().toOSString() + System.getProperty("file.separator") + switches[j].substring(3);
					return;
				} else{
					// if not we send it to the output folder
					IFileStore folder = SPARKToolChain.getProjectOutputDirectory(project);
					if (folder != null){
						sumPath = URIUtil.toPath(folder.toURI()).toOSString()
								+ System.getProperty("file.separator")
								+ switches[j].substring(3);
					}
					return;
				}
			}
		}

		// by default the sum file will be in the output directory
		if (sumPath == null){
			IFileStore folder = SPARKToolChain.getProjectOutputDirectory(project);
			// retrieve the path and file name of the file
			sumPath = URIUtil.toPath(folder.toURI()).toOSString()
					+ System.getProperty("file.separator") + folder.getName()
					+ ".sum";
		}

	}

	/**
	 * Retrieve the directory containing the vcg's files
	 * It can be either
	 * - the output_directory specified in the gpr file
	 * - the first common parent directory of sources specified in the gpr file
	 * - the project folder
	 * @param project
	 */
	private void POGSDirectory(IProject project){
		// Get the output_directory value
		final String[] switches = SPARKToolChain.getSPARKAttribute(project, "Default_Switches", "IDE", "Examiner");

		for (int j=0; j < switches.length; j++){
			if (switches[j].startsWith("output_directory=", 1)){
				directory = new String(switches[j].substring(18));
			}
		}

		// get the common directory for all sources
		if (directory == null){
			final String separator = System.getProperty("file.separator");
			LinkedList<String> gprDirs = SPARKToolChain.getSourceDirectories(project);

			if (gprDirs != null && !gprDirs.isEmpty()){
				String tmp = gprDirs.getFirst();

				for (int j = 1; j < gprDirs.size(); j++){
					int idx1 = tmp.indexOf(separator);
					int idx2 = gprDirs.get(j).indexOf(separator);
					while (idx1 != -1 && idx2 != -1 && gprDirs.get(j).substring(0,idx2).equals(tmp.substring(0,idx1))){
						directory = tmp.substring(0,idx2);

						idx1 = tmp.indexOf(separator, idx1+1);
						idx2 = gprDirs.get(j).indexOf(separator, idx2+1);
					}

				}

			}
		}

		//else it will be at the project level
		if (directory == null){
			directory = project.getLocation().toOSString();
		}
	}

	/*
	 *  Open the sum file after POGS has been executed
	 */
	protected void afterCommand(){
    	try {
			final IFileStore sumFileStore = GNATbenchCorePlugin.getDefault()
					.getEFSRegistry().getUniqueStore(
							URIUtil.toURI(new Path(sumPath)));
			PlatformUI.getWorkbench().getDisplay().asyncExec(new Runnable() {

				public void run() {
					try {
						IDE.openEditorOnFileStore(PlatformUI.getWorkbench()
								.getActiveWorkbenchWindow().getActivePage(),
								sumFileStore);

					} catch (PartInitException e) {
						GNATbenchCorePlugin.getDefault().logError(null, e);
					}
				}
			});
		} catch (GNATbenchCoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}
    }

	@Override
	public void run(TreeMap<IGNATProjectView, LinkedList<IFileStore>> files,
			IProgressMonitor monitor) {

		final String command = "pogs";

		for (Entry<IGNATProjectView, LinkedList<IFileStore>> entry : files
				.entrySet()) {

			LinkedList <String> prjCommand = new LinkedList<String> ();
			prjCommand.add(command);

			POGSDirectory (entry.getKey().getEclipseProject());
			boolean directorySet = false;

			final String[] switches = SPARKToolChain.getSPARKAttribute(entry.getKey().getEclipseProject(),
					"Default_Switches", "IDE", "pogs");
			if (switches == null){
				continue;
			}

			// add the switches if not -o, we'll set it afterwards
			for (int j= 0; j < switches.length; j++) {
				if (! switches[j].startsWith("-o")){
					if (switches[j].startsWith("-d")){
						directorySet = true;
					}
					prjCommand.add(switches [j]);
				}
			}

			// if the -d option has not been specified in the spark switches,
			// we set it
			if (! directorySet){
				prjCommand.add("-d=" + directory);
			}

			// set the path and name of the output sum file
			prjCommand.add("-o=" + sumPath);

			//we do not add files as sparkmake apply to a project

			try {
				runProcessInEnv((GPRProject)entry.getKey(), "SPARK POGS",
						prjCommand.toArray(new String[prjCommand.size()]), entry
								.getKey().getProjectFile().getParent(), monitor);
				//TODO test if this is done after the command
				afterCommand();
			} catch (GNATbenchCoreException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		}
	}

} // POGSJob
