/*****************************************************************************
 * Copyright (c) 2007, 2008 AdaCore.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *****************************************************************************/

/**
 * GPS is a single threaded application. Therefore, it should not be accessed
 * by two threads in parallel. Before calling any of the GPS-binding function,
 * the user must protect the calls by a call to startGPSWork. This call will
 * be blocking if another thread is doing GPS queries. This function returns a
 * number that has to be given to stopGPSWork, in order to release the handle.
 * stopGPSWork has to be called in a finally handle, so that it's processed
 * even if exception occures. Multiple calls on startGPSWork on the same 
 * thread will not block, and blocked threads will be released when the 
 * outermost handle has been released.
 */

package com.adacore.gnatbench.library;

import org.eclipse.core.runtime.jobs.Job;

import com.adacore.ajis.NativeException;

public class LibrarySemaphore {
	
	static public class LibraryLoc implements com.adacore.ajis.ILock {
		org.eclipse.core.runtime.jobs.ILock lock = Job.getJobManager()
		.newLock();

		Thread currentThread;

		public void lock() {
			lock.acquire();

			synchronized (this) {
				currentThread = Thread.currentThread();
			}
		}

		public void unlock() {
			lock.release();

			synchronized (this) {
				if (lock.getDepth() == 0 && isHeldByCurrentThread()) {
					//  If we're still on the current thread (e.g. no other
					//  thread has already taken the lock) then reset the 
					//  currentThread value.

					currentThread = null;
				}
			}
		}

		public boolean isHeldByCurrentThread() {
			synchronized (this) {
				return currentThread == Thread.currentThread();
			}
		}
	}

	public static LibraryLoc fgLock = new LibraryLoc ();
	
	/**
	 * This has to be called before using any of the bound functions. No other
	 * thread will access to the bound functions until stopGPSWork have been
	 * called with the resulting value of this function. If there's already a
	 * thread working, this function will be blocking.
	 * 
	 * @return
	 */
	public static LibraryMonitor startGPSWork () {
		fgLock.lock ();
		
		LibraryMonitor m = new LibraryMonitor ();
		m.depth = fgLock.lock.getDepth();
		
		return m;
	}
	
	public static void stopGPSWork (LibraryMonitor monitor) {
		int depth = fgLock.lock.getDepth();
		
		if (monitor.depth != depth) {
			throw new NativeException("stop doesn't match a start, depth is "
					+ depth + " and should be " + monitor.depth);
		}
		
		fgLock.unlock();
	}
	
	/**
	 * Return true if the current call is protected by the library semaphore,
	 * false otherwise.
	 * 
	 * @return
	 */
	public static boolean isProtected () {
		return fgLock.isHeldByCurrentThread();
	}
	
	public static void checkIsProtected () {
		if (!isProtected()) {
			throw new NativeException ("Code is not protected by the lock");
		}
	}
}
