/*****************************************************************************
 * Copyright (c) 2008 AdaCore.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *****************************************************************************/

package com.adacore.gnatbench.core.projects;

import java.util.LinkedList;

import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.resources.IProject;

import com.adacore.gnatbench.core.GNATbenchCoreException;

/**
 * This class implements a view of a GNATProject. This view has a limited 
 * lifetime - it should last until the project is recomputed. For safety,
 * operations may be called using a GNATbenchKernelJob, so that updates of
 * the project are prevented during the call. 
 * 
 * If the project is out of date, then its methods will throw an exeption.
 * 
 * This interface is not intended to be implemented by clients.
 * 
 * @since 2.2
 */
public interface IGNATProjectView {

	/**
	 * Return the name of this GNAT project
	 */
	public String getName() throws GNATbenchCoreException;

	/** 
	 * Return the corresponding project file.
	 */
	public IFileStore getProjectFile () throws GNATbenchCoreException;

	/**
	 * Return the list of object directories.
	 * 
	 * @param recursive when true, includes object directories coming from the
	 * dependencies.
	 */
	public LinkedList<IFileStore> getObjectDirectories(boolean recursive)
			throws GNATbenchCoreException;
	
	/**
	 * Return the list of source directories.
	 * 
	 * @param recursive when true, includes source directories coming from the
	 * dependencies.
	 */
	public LinkedList<IFileStore> getSourceDirectories(boolean recursive)
			throws GNATbenchCoreException;
	
	/**
	 * Return the list of source files.
	 * 
	 * @param recursive when true, include source files coming from the 
	 * dependencies.
	 */
	public LinkedList<IFileStore> getSourceFiles(boolean recursive)
			throws GNATbenchCoreException;
	
	/**
	 * Return the list of dependencies.
	 * @param recursive when true, include indirect dependencies recursively
	 */
	public LinkedList<IGNATProjectView> getDependencies(boolean recursive)
			throws GNATbenchCoreException;
	
	
	/**
	 * Return the registry associated to this project view.
	 */
	public IGNATProjectRegistry getRegistry ();
	
	/**
	 * Return the eclipse project associated with this GNAT project view if
	 * any, null otherwise.
	 */
	public IProject getEclipseProject ();
	
	/**
	 * Return the list of scenario variables confirgured in this project and
	 * its dependencies.
	 */
	public LinkedList<IScenarioVariable> getScenarioVariables()
			throws GNATbenchCoreException;
}
