/*****************************************************************************
 * Copyright (c) 2008 AdaCore.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *****************************************************************************/

package com.adacore.gnatbench.core.projects;

import java.util.LinkedList;

import org.eclipse.core.runtime.IPath;

/**
 * A GNATbench project registry is a cross-project class holding information 
 * about a whole GNATbench hierarchy of projects. Modules that work on a 
 * project hierarchy can add assistants to the kernel in order to synchonize 
 * their internal data.
 * 
 * There can be several registries loaded in GNATbench at a time - each
 * of them representing a distinct GNAT project hierarchy.
 * 
 * GNAT projects referenced in the registry may or may not be associated with
 * actual Eclipse projects.
 * 
 * This interface is not intended to be implemented by clients.
 * 
 * @since 2.2
 */
public interface IGNATProjectRegistry {

	/**
	 * Return the root project of the project hierarchy contained in this
	 * kernel.
	 */
	public IGNATProjectView getRootProject ();

	/**
	 * Return a project view based on its name.
	 */
	public IGNATProjectView getProjectFromName(String name);	

	/**
	 * Return a project view based on its system location
	 */
	public IGNATProjectView getProjectFromLocation(IPath path);

	/**
	 * Return the assistant of the given name.
	 */
	public IGNATProjectRegistryAssistant getAssistant (String id);
	
	
	/**
	 * Recomputes the content of the project hierarchy based on the scenario
	 * variable values and the contents on the disk - all pending kernel
	 * jobs will be cancelled and waited for before the computation is actually
	 * done.
	 */
	public void recompute ();
	
	/**
	 * Adds a listener to the list of listeners for this registry - does 
	 * nothing if the listener is already in place.
	 */
	public void addListener (IGNATProjectRegistryListener listener);
	
	/**
	 * Removes a listener to the list of listeners for this registry - does 
	 * nothing if the listener is not already in place.
	 */
	public void removeListener (IGNATProjectRegistryListener listener);

	/**
	 * Return the list of messages parsed after the last project load.
	 */
	public LinkedList <IGNATProjectMessage> getMessages ();
	
	/**
	 * Unload the project from the memory. This should be called manually for
	 * anonymous registries. Registries loaded with an eclipse project will be
	 * unloaded automatically. Failing to call this on an anonymous registry
	 * will result in a memory leak. 
	 */
	public void unload ();
	
	/**
	 * Return true if the project has been sucessfully loaded, false if not or
	 * if it has been unloaded.
	 */
	public boolean isValid ();
	
	/**
	 * Executes the runnable given in parameter with the garantee that no 
	 * recomputation will occur until the runnable is finished.
	 */
	public void run (Runnable runnable);
}
