/*****************************************************************************
 * Copyright (C) 2007-2009 AdaCore.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *****************************************************************************/

package com.adacore.gnatbench.core.internal.utils;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;

import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedFile;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.Projects.Registry.Registry_Package;
import com.adacore.gnatbench.library.GNATCOLL.Filesystem.Filesystem_String;
import com.adacore.gnatbench.library.GNATCOLL.VFS.Virtual_File;

public class FileUtils {

	/**
	 * Finds all files in the root of the specified project having
	 * the specified file extension.
	 */
	public static IResource[] rootFilesWithExtension(final String extension, final IProject project) {
		IResource[] result = null;
		IResource[] candidates = null;
		int count = 0;

		try {
			candidates = project.members(IContainer.EXCLUDE_DERIVED);
		} catch (CoreException e) {
		} // try

		for (int k = 0; k < candidates.length; k++) {
			final String ext = candidates[k].getFileExtension();
			if ((ext != null) && ext.equalsIgnoreCase(extension)) {
				++count;
			} // if
		} // for

		result = new IResource[count]; // OK if count is zero

		if (count > 0) {
			count = 0;
			for (int k = 0; k < candidates.length; k++) {
				final String ext = candidates[k].getFileExtension();
				if ((ext != null) && ext.equalsIgnoreCase(extension)) {
					result[count++] = candidates[k];
				} // if
			}
		} // if

		return result;
	} // rootFilesWithExtension


	/**
	 * Search the entire project, recursively, including dependent
	 * projects, for the file named FileName.
	 *
	 * Note this would not be an Ada file because otherwise you could
	 * just use a call to function File.fromGPSName(project, fileName) instead.
	 *
	 */
	public static IResource findGlobally(final IProject project, final String fileName) {
		IResource result = null;

		IPath path = new Path(fileName);
		if (path.segmentCount() > 1) {
			// not just a base name so use the path it specifies; might return null.
			return GeneralizedFile.fromOSPath(project, path.toString()).getFile();
		} // if

		// Look in this project, recursing down into the containers therein.
		// We use this approach first since accessing the registry is comparatively
		// more expensive.
		FileNameMatcher locator = new FileNameMatcher(project, fileName);
		try {
			if (locator.search()) {
				return locator.result();
			} // if
			// it isn't in the directory tree of this project
			// so we have to look for it elsewhere
			LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();
			try {
				GNATProjectRegistry registry = (GNATProjectRegistry) GNATbenchSession
						.getDefault().getOrLoadRegistry(project);
				Virtual_File vfile = Registry_Package.Create(
						new Filesystem_String(fileName), registry
								.getProjectRegistry(), true, true);
				result = GeneralizedFile.fromVirtualFile(project, vfile)
						.getFile();
			} finally {
				LibrarySemaphore.stopGPSWork(libMonitor);
			} // try
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
			// result stays at current null value
		} // try

		return result;
	} // findGlobally


	/**
	 * A resource tree walker that searches the specified project
	 * looking for a file with the specified name.
	 *
	 */
	public static class FileNameMatcher extends FileFinder {

		protected final String name;

		public FileNameMatcher(IProject prj, final String fileName) {
			super(prj);
			name = fileName;
		} // FileNameMatcher

		@Override
		public boolean userOp(IFile resource) {
			return resource.getName().equals(name);
		} // userOp

	} // FileNameMatcher


	/**
	 * A resource tree walker that finds a file by applying userOp.
	 * Traversal starts at the root of the specified project and
	 * continues down to all folders unless userOp indicates that
	 * further searching is not necessary.
	 *
	 * In contrast to a resource visitor, traversal only continues
	 * until userOp() returns true.
	 *
	 */
	public abstract static class FileFinder {

		protected IResource result = null;
		protected final IProject project;

		public FileFinder(final IProject prj) {
			project = prj;
		} // FileFinder

		public IResource result() {
			return result;
		} // result

		public abstract boolean userOp(IFile resource);

		public boolean search() throws CoreException {
			return search(project);
		} // search

		protected boolean search(IContainer container) throws CoreException {
			final IResource [] resources = container.members(IProject.EXCLUDE_DERIVED);
			for (int k = 0; k < resources.length; k++) {
				if (resources[k].getType() == IResource.FILE) {
					if (userOp((IFile) resources[k])) {
						result = resources[k];
						return true;
					} // if
				} else if (resources[k].getType() == IResource.FOLDER) {
					if (search((IContainer) resources[k])) {
						return true;
					} // if
				} // if
			} // for
			return false;
		} // search

	} // FileFinder


} // FileUtils
