/*****************************************************************************
 * Copyright (C) 2008-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *****************************************************************************/

package com.adacore.gnatbench.core.internal.projects;

import java.net.URI;
import java.util.LinkedList;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;

import com.adacore.gnatbench.core.GNATbenchCoreException;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.GNATbenchProjectProperties;
import com.adacore.gnatbench.core.internal.builder.GNATbenchProjectNature;
import com.adacore.gnatbench.core.internal.gpswrappers.GPRProject;
import com.adacore.gnatbench.core.projects.IGNATProjectRegistry;
import com.adacore.gnatbench.core.projects.IGNATProjectRegistryFactory;
import com.adacore.gnatbench.core.projects.IGNATProjectRegistryFactoryListener;
import com.adacore.gnatbench.core.projects.IGNATProjectView;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;

public class GNATProjectRegistryFactory implements IGNATProjectRegistryFactory {

	private Map <String, GNATProjectRegistry> fRegsByProject =
		new TreeMap<String, GNATProjectRegistry> ();
	private Map <URI, GNATProjectRegistry> fRegsByURI =
		new TreeMap<URI, GNATProjectRegistry> ();
	private Set <String> fRegsInDestruction =
		new TreeSet<String> ();

	private LinkedList <IGNATProjectRegistryFactoryListener> listeners = new
	LinkedList <IGNATProjectRegistryFactoryListener> ();
	// TODO: connect notifications to this listener.

	public IGNATProjectRegistry createAnonymousRegistry(IFileStore file) {
		GNATProjectRegistry result = new GNATProjectRegistry(this, file, true);

		result.initialize();

		return result;
	}

	/**
	 * Return a project registry corresponding to the project given in
	 * parameter. If the project is a root project, then the associated
	 * registry  will hold its project hierarchy. If not, the returned registry
	 * is a registry containing a project hierarchy including this project.
	 */
	public IGNATProjectRegistry getOrLoadRegistry (IProject project) {
		try {
			if (project == null || !project.isOpen()
					|| !project.hasNature(GNATbenchProjectNature.NATURE_ID)
					|| fRegsInDestruction.contains(project.getName())) {
				return null;
			}
		} catch (CoreException e1) {
			GNATbenchCorePlugin.getDefault().logError(null, e1);

			return null;
		}

		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			if (fRegsByProject.containsKey(project.getName())) {
				//  This is an optimisation - all the computation underneath is
				//  very expensive, so if we've already got the kernel for this
				//  project, use it.

				return fRegsByProject.get(project.getName());
			}

			GNATProjectRegistry registry;
			GNATbenchProjectProperties props = GNATbenchProjectProperties
					.getPropertiesFor(project);
			String projectPath = props.getGPROSPath();

			if (projectPath == null) {
				return null;
			}

			IFileStore prjStore = GNATbenchCorePlugin.getDefault()
					.getEFSRegistry()
					.getUniqueStore(URIUtil.toURI(projectPath));

			if (fRegsByURI.containsKey(prjStore.toURI())) {
				return fRegsByURI.get(prjStore.toURI());
			}

			//  If we are on a root project, then create a new project
			//  association

			if (props.isRootProject()) {
				registry = new GNATProjectRegistry (this, prjStore, false);

				fRegsByURI.put(prjStore.toURI(), registry);

				registry.initialize();

				return registry;
			}

			//  Otherwise, look from all root projects, if we find one
			//  importing that one.

			IProject[] projects = ResourcesPlugin.getWorkspace().getRoot()
					.getProjects();

			for (int j = 0; j < projects.length; ++j) {
				props = GNATbenchProjectProperties
						.getPropertiesFor(projects[j]);

				if (props.isRootProject()) {
					registry = (GNATProjectRegistry) getOrLoadRegistry(projects [j]);

					LinkedList <IGNATProjectView> deps =
						registry.getRootProject().getDependencies(true);

					for (IGNATProjectView prj : deps) {
						if (prj.getProjectFile().equals(prjStore)) {
							fRegsByProject.put(project.getName(), registry);

							return registry;
						}
					}
				}
			}

		} catch (GNATbenchCoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}

		return null;
	}



	/**
	 * If the project given in parameter is already loaded in the memory,
	 * returns the corresponding kernel.
	 *
	 * @param path
	 * @return
	 */
	public GPRProject getGPRForPath(IPath path) {
		GNATProjectRegistry reg = getRegistryForPath(path);

		if (reg != null) {
			return reg.getProjectFromLocation(path);
		} else {
			return null;
		}
	}

	/**
	 * Return the registry that contains the project referenced by the path
	 * given in parameter.
	 *
	 * @param path
	 * @return
	 */
	public GNATProjectRegistry getRegistryForPath(IPath path) {
		URI uri = URIUtil.toURI(path);

		if (fRegsByURI.containsKey(uri)) {
			return fRegsByURI.get(uri);
		}

		IProject[] projects = ResourcesPlugin.getWorkspace().getRoot()
				.getProjects();

		for (int j = 0; j < projects.length; ++j) {
			GNATProjectRegistry kernel = (GNATProjectRegistry) getOrLoadRegistry(projects[j]);

			if (kernel != null) {
				GPRProject prj = kernel.getProjectFromLocation(path);

				fRegsByURI.put(uri, kernel);

				if (prj != null) {
					return kernel;
				}
			}
		}

		return null;
	}

	public void unloadRegistry (GNATProjectRegistry registry) {
		String kernelName =  null;

		if (!registry.isAnonymous()) {
			//  During the process of kernel unloading, there may be process
			//  trying to get the kernel on this project. We want to avoid
			//  that, since the kernel should not be reachable anymore, that's
			//  why we add it to a list of projects that can't be associated
			//  with a kernel.

			kernelName = registry.getRootProject().getEclipseProject()
					.getName();
			fRegsInDestruction.add(kernelName);

			fRegsByURI.remove(registry.getRootProjectFile().toURI());

			LinkedList <String> disProject = new LinkedList <String> ();

			for (Map.Entry<String, GNATProjectRegistry> entry : fRegsByProject
					.entrySet()) {
				if (entry.getValue().equals(this)) {
					disProject.add(entry.getKey());
				}
			}

			for (String projectName : disProject) {
				fRegsByProject.remove(projectName);
			}
		}

		registry.unload();

		if (kernelName != null) {
			fRegsInDestruction.remove(kernelName);
		}
	}

	public void recomputeAllKernels () {
		for (Map.Entry <URI, GNATProjectRegistry> entry : fRegsByURI.entrySet()) {
			entry.getValue().recompute();
		}
	}

	/**
	 * Get a project from the stored kernels.
	 *
	 * @param name
	 * @return
	 */
	public GPRProject getGlobalProjectFromName(String name) {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			for (Map.Entry<URI, GNATProjectRegistry> entry : fRegsByURI
					.entrySet()) {
				GPRProject prj = entry.getValue().getProjectFromName(name);

				if (prj != null) {
					return prj;
				}
			}
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}

		return null;
	}

	public void addListener (IGNATProjectRegistryFactoryListener listener) {
		listeners.add(listener);
	}

	public void removeListener (IGNATProjectRegistryFactoryListener listener) {
		listeners.remove(listener);
	}
}
