/*****************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *****************************************************************************/

package com.adacore.gnatbench.core.internal.projects;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Map;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.InvalidRegistryObjectException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.IJobChangeListener;
import org.eclipse.core.runtime.jobs.ILock;
import org.eclipse.core.runtime.jobs.ISchedulingRule;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.core.runtime.jobs.MultiRule;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchListener;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.osgi.framework.Bundle;

import com.adacore.ajis.BooleanRef;
import com.adacore.ajis.IProxy;
import com.adacore.ajis.NativeException;
import com.adacore.ajis.IProxy.Owner;
import com.adacore.gnatbench.core.GNATbenchCoreException;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.GNATbenchKernelRule;
import com.adacore.gnatbench.core.internal.GNATbenchProjectProperties;
import com.adacore.gnatbench.core.internal.GPRMessage;
import com.adacore.gnatbench.core.internal.adaeditor.AbstractAdaEditor;
import com.adacore.gnatbench.core.internal.adaeditor.AdaEditorToolPreferencesProvider;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedFile;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedLocation;
import com.adacore.gnatbench.core.internal.filesystem.FileStoreRegistry;
import com.adacore.gnatbench.core.internal.filesystem.ResourceComparator;
import com.adacore.gnatbench.core.internal.gpswrappers.GPRProject;
import com.adacore.gnatbench.core.internal.toolchain.ToolChain;
import com.adacore.gnatbench.core.internal.toolchain.ToolChain.NoExtensionFound;
import com.adacore.gnatbench.core.internal.utils.PluginUtils;
import com.adacore.gnatbench.core.internal.utils.ViewUtils;
import com.adacore.gnatbench.core.projects.IGNATProjectMessage;
import com.adacore.gnatbench.core.projects.IGNATProjectRegistry;
import com.adacore.gnatbench.core.projects.IGNATProjectRegistryAssistant;
import com.adacore.gnatbench.core.projects.IGNATProjectRegistryListener;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.ALI_Parser.ALI_Parser_Package;
import com.adacore.gnatbench.library.Ada_Semantic_Tree.Assistants.Assistants_Package;
import com.adacore.gnatbench.library.Entities.Entities_Database;
import com.adacore.gnatbench.library.Entities.Entities_Package;
import com.adacore.gnatbench.library.Entities.LI_Handler_Record;
import com.adacore.gnatbench.library.GNATCOLL.Filesystem.Filesystem_String;
import com.adacore.gnatbench.library.GNATCOLL.VFS.File_Array;
import com.adacore.gnatbench.library.GNATCOLL.VFS.VFS_Package;
import com.adacore.gnatbench.library.GNATCOLL.VFS.Virtual_File;
import com.adacore.gnatbench.library.Language.Language_Root;
import com.adacore.gnatbench.library.Language.Ada.Ada_Package;
import com.adacore.gnatbench.library.Language.Tree.Tree_Language;
import com.adacore.gnatbench.library.Language.Tree.Database.Construct_Database;
import com.adacore.gnatbench.library.Language.Tree.Database.File_Buffer_Provider;
import com.adacore.gnatbench.library.Language.Tree.Database.Structured_File;
import com.adacore.gnatbench.library.Language_Handlers.Language_Handler_Record;
import com.adacore.gnatbench.library.Language_Handlers.Language_Handlers_Package;
import com.adacore.gnatbench.library.Projects.Error_Report;
import com.adacore.gnatbench.library.Projects.Project_Status;
import com.adacore.gnatbench.library.Projects.Project_Type;
import com.adacore.gnatbench.library.Projects.Projects_Package;
import com.adacore.gnatbench.library.Projects.Editor.Editor_Package;
import com.adacore.gnatbench.library.Projects.Registry.Project_Registry;
import com.adacore.gnatbench.library.Projects.Registry.Registry_Package;
import com.adacore.gnatbench.library.Standard.AdaString;

/**
 * This class stores a GNATbench kernel - each project hierarchy is assigned
 * to a distinct kernel, holding distinct cross references & construct
 * database.
 *
 */
final public class GNATProjectRegistry implements IGNATProjectRegistry, IJobChangeListener {

	// XXX: This should be private
	Project_Registry fProjectRegistry;
	private Entities_Database fEntitiesDatabase;
	public Language_Handler_Record fLanguageHandler;
	private Job fConstructLoadJob;
	public Construct_Database fConstructDatabase;

	private final boolean fIsAnonymous;

	private GPRProject fRootProject;

	private static final boolean isWRSVersion = PluginUtils.isWRSPluginPresent();

	private int fTimestamp = 0;

	private HashSet <IGNATProjectRegistryListener> fGPRListeners =
		new HashSet <IGNATProjectRegistryListener> ();

	private LinkedList <Job> fJobs = new LinkedList <Job> ();

	private IFileStore fProjectFile;

	private boolean fValid = true;

	private LinkedList<IGNATProjectMessage> fMessages = new LinkedList<IGNATProjectMessage>();

	private TreeMap<String, IGNATProjectRegistryAssistant> fAssistants = new TreeMap<String, IGNATProjectRegistryAssistant>();

	private String fVersion;

	public static QualifiedName managedSourceFile = new QualifiedName(
			"com.adacore.core", "managedSourceFile");
	public static QualifiedName managedSourceFolder = new QualifiedName(
			"com.adacore.core", "managedSourceFolder");
	public static QualifiedName managedBinaryFolder = new QualifiedName(
			"com.adacore.core", "managedBinaryFolder");
	public static QualifiedName correspondingKernel = new QualifiedName(
			"com.adacore.com", "correspondingKernel");

	public GNATProjectRegistryFactory fFactory;

	public boolean fIsUnloaded = false;

	private ILock registryLock = Job.getJobManager()
	.newLock();

	private LinkedList <IMarker> fLastBuildMarkers = new LinkedList<IMarker> ();

	private TreeSet<IResource> fFilesWithMarkers = new TreeSet<IResource>(
			new ResourceComparator ());

	/**
	 * Creates a registry. When created, kernel jobs are locked until the
	 * caller calls unlockNewJobs - this helps fixing synchronization problems.
	 *
	 * @param projectPath
	 */
	public GNATProjectRegistry(GNATProjectRegistryFactory factory,
			IFileStore gprFile, boolean isAnonymous) {
		fFactory = factory;
		fProjectFile = gprFile;
		fIsAnonymous = isAnonymous;
	}

	static private class GNATbenchBufferProvider extends File_Buffer_Provider {
		public AdaString Get_Buffer(Virtual_File File) {
			try {
				GeneralizedLocation loc = GeneralizedLocation
						.fromAbsoluteLocation(GeneralizedFile.fromVirtualFile(
								null, File), 1, 0);

				AbstractAdaEditor editor = GNATbenchCorePlugin.getDefault()
						.getEditor(loc.getFile());

				if (editor != null && editor.getDocument() != null) {
					// Copy the string and let the caller manage the returned
					// value, as requested by the specification.
					AdaString copy = new AdaString(editor.getDocument().get());
					((IProxy) copy).setOwner (Owner.NATIVE);

					return copy;
				} else {
					return super.Get_Buffer(File);
				}
			} catch (Throwable e) {
				GNATbenchCorePlugin.getDefault().logError("", e);
				return null;
			}
		}
	}

	/**
	 * Initialize the contents of the kernel
	 *
	 */
	public void initialize () {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			fProjectRegistry = new Project_Registry ();
			((IProxy) fProjectRegistry).setOwner(Owner.NATIVE);
			//  FIXME: There is likely to be a reference to the project
			//  registry from all over the place. See how / when to free this
			//  registry.

			fConstructDatabase = new Construct_Database ();
			((IProxy)fConstructDatabase).setOwner(Owner.NATIVE);
			//  FIXME: This construct database will be referenced from various
			//  locations - it shouldn't be owned by Java proxy. We should
			//  have a way to free it.

			GNATbenchBufferProvider provider = new GNATbenchBufferProvider ();
			((IProxy) provider).setOwner(Owner.NATIVE);
			//  FIXME: This states that the destruction of the value passed to
			//  initialize is supposed to be freed by the construct database.
			//  To be checked.

			fConstructDatabase.Initialize(provider);
			Assistants_Package.Register_Ada_Assistants(fConstructDatabase);

			fEntitiesDatabase = Entities_Package.Create(fProjectRegistry);

			Language_Handler_Record.Ref langRef = new Language_Handler_Record.Ref();
			Language_Handlers_Package.Create_Handler(langRef);
			fLanguageHandler = langRef.ref;

			fEntitiesDatabase.Register_Language_Handler(fLanguageHandler);
			fLanguageHandler.Set_Registry(fProjectRegistry);

			loadProjectFromPath ();

			String gnatls = getGNATLS();
			if (gnatls.equals("")) {
				gnatls = "gnatls";
				// default to native for attempting computePredefinedPaths
			}
			computePredefinedPaths(gnatls);

			LI_Handler_Record li_handler = ALI_Parser_Package
					.Create_ALI_Handler(fEntitiesDatabase, fProjectRegistry);
			fLanguageHandler.Register_Language(
					com.adacore.gnatbench.library.Language.Ada.Ada_Package.Ada_Lang(),
					com.adacore.gnatbench.library.Language.Tree.Ada.Ada_Package
							.Ada_Tree_Lang(), li_handler);
			fProjectRegistry.Register_Default_Language_Extension(new AdaString(
					"Ada"), new AdaString(".ads"), new AdaString(".adb"));

			recompute();

			configureAssistants();
		}finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	private void loadProjectFromPath () {
		BooleanRef status = new BooleanRef ();
		BooleanRef newProjectLoaded = new BooleanRef ();

		fProjectRegistry.Load(VFS_Package.Create(new Filesystem_String(URIUtil
				.toPath(fProjectFile.toURI()).toOSString())),
				new Error_Report() {

			@Override
			public void Error_Report_Body(AdaString Msg) {
				fMessages.add(new GPRMessage(Msg.toString(),
						GNATProjectRegistry.this));
			}
		}, newProjectLoaded, status);

		if (fProjectRegistry.Get_Root_Project().Status() == Project_Status.Empty) {
			fValid = false;
		} else {
			fValid = true;
		}
	}

	public synchronized void reloadConstructFile (Virtual_File file) {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			Structured_File structuredFile = fConstructDatabase.Get_Or_Create(
					file, com.adacore.gnatbench.library.Language.Tree.Ada.Ada_Package
							.Ada_Tree_Lang());

			structuredFile.Update_Contents();
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Interrupts the current construct database load.
	 *
	 */
	public synchronized void interruptConstructDatabaseLoad () {
		if (fConstructLoadJob != null) {
			fConstructLoadJob.cancel();
		}
	}


	private class ReloadConstructDatabaseJob extends Job implements IWorkbenchListener {

		private Job fPreviousJob = null;

		private boolean fIsShutdown = false;

		public ReloadConstructDatabaseJob (String name, Job previousJob) {
			super ("loading construct database for " + name);

			setPriority(LONG);
		}

		private void loadFiles (File_Array files, IProgressMonitor monitor) throws CoreException {
			LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

			int i;

			try {
				if (files.Length() == 0) {
					return;
				}

				i = files.First();
			} finally {
				LibrarySemaphore.stopGPSWork(libMonitor);
			}

			while (true) {
				if (monitor.isCanceled()) {
					break;
				}

				boolean isAdaTree = false;
				Virtual_File file = null;

				libMonitor = LibrarySemaphore.startGPSWork();

				try {
					if (fIsShutdown) {
						return;
					}

					if (i > files.Last()) {
						break;
					}

					if (monitor.isCanceled()) {
						break;
					}

					file = files.Get_Element_At(i);

					monitor.subTask(file.Full_Name(false).toString());
					Language_Root lang =
						fLanguageHandler.Get_Language_From_File_Language (file, false);

					isAdaTree = Arrays.equals(lang.getAccess(),
							(Ada_Package.Ada_Lang().getAccess()));


					if (isAdaTree) {
						Tree_Language tree_lang = fLanguageHandler
								.Get_Tree_Language_From_File(file, false);
						fConstructDatabase.Get_Or_Create(file, tree_lang);
					}
				} finally {
					LibrarySemaphore.stopGPSWork(libMonitor);
				}

				i = i + 1;

				monitor.worked(1);
			}
		}

		File_Array prjFiles;
		File_Array stdFiles;

		protected IStatus run(IProgressMonitor monitor) {
			if (fPreviousJob != null) {
				try {
					fPreviousJob.join();
				} catch (InterruptedException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				}
			}

			if (monitor.isCanceled()) {
				return GNATbenchCorePlugin.OK_STATUS;
			}

			try {
				PlatformUI.getWorkbench().addWorkbenchListener(this);

				LibraryMonitor libMonitor = LibrarySemaphore
				.startGPSWork();

				try {
					if (fIsShutdown) {
						return GNATbenchCorePlugin.OK_STATUS;
					}

					fConstructDatabase.Clear();

					prjFiles = fProjectRegistry.Get_Root_Project()
					.Get_Source_Files(true);
					stdFiles = fProjectRegistry
					.Get_Predefined_Source_Files();

					monitor.beginTask("Parse Ada Files", prjFiles
							.Length()
							+ stdFiles.Length());
				} catch (NativeException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);

					throw e;
				} finally {
					LibrarySemaphore.stopGPSWork(libMonitor);
				}


				if (!monitor.isCanceled()) {
					loadFiles(prjFiles, monitor);
				}

				if (!monitor.isCanceled()) {
					loadFiles(stdFiles, monitor);
				}

				libMonitor = LibrarySemaphore.startGPSWork();

				try {
					VFS_Package.Unchecked_Free((File_Array.Ref) prjFiles
							.NewProxyRef());
					VFS_Package.Unchecked_Free((File_Array.Ref) stdFiles
							.NewProxyRef());
				} finally {
					LibrarySemaphore.stopGPSWork(libMonitor);
				}

				monitor.done();
			} catch (CoreException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} finally {
				PlatformUI.getWorkbench().removeWorkbenchListener(this);
			}

			return GNATbenchCorePlugin.OK_STATUS;
		}

		public void postShutdown(IWorkbench workbench) {
			// TODO Auto-generated method stub

		}

		public boolean preShutdown(IWorkbench workbench, boolean forced) {
			LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

			try {
				//  This is done withing the protection of the library
				//  semaphore so that no protected section is started with
				//  incorrect state.
				fIsShutdown = true;
			} finally {
				LibrarySemaphore.stopGPSWork(libMonitor);
			}

			return true;
		}};

	/**
	 * Reloads the construct database.
	 *
	 */
	public synchronized void reloadConstructDatabase () {
		if (fConstructLoadJob != null) {
			fConstructLoadJob.cancel();
		}

		if (isAnonymous ()) {
			//  Do not currently load construct db for anonymous registries.

			return;
		}

		if (!GNATbenchCorePlugin.getDefault().getPreferenceStore().getBoolean(
				AdaEditorToolPreferencesProvider.PREF_TOOLS_CODE_ASSIST)) {
			return;
		}

		String name = "";

		try {
			name = this.fRootProject.getName();
		} catch (GNATbenchCoreException e1) {
			GNATbenchCorePlugin.getDefault().logError(null, e1);
		}

		fConstructLoadJob = new ReloadConstructDatabaseJob(name,
				fConstructLoadJob);
		scheduleJob(fConstructLoadJob);
	}

	/**
	 * Return the project registry for this kernel.
	 *
	 * @return
	 */
	public Project_Registry getProjectRegistry () {
		return fProjectRegistry;
	}

	public Construct_Database getConstructDatabase () {
		return fConstructDatabase;
	}

	/**
	 * Return the entity database for this kernel.
	 */
	public Entities_Database getEntitiesDatabase () {
		return fEntitiesDatabase;
	}

	/**
	 * Return the full path of the source name given in parameter, provided
	 * that this source is contained in the registry, null otherwise.
	 */
	public IPath getSourcePath (String sourceName) {
		if (sourceName == null) {
			return null;
		}

		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
            Virtual_File file = new Virtual_File ();

            getProjectRegistry().Get_Full_Path_From_File(
					new Filesystem_String(sourceName), true, false,
					Projects_Package.No_Project(), true, file);


			return new Path (file.Full_Name(true).toString());
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Return the workspace project imported all the projects of this kernel.
	 */
	public GPRProject getRootProject () {
		return fRootProject;
	}

	public IFileStore getRootProjectFile () {
		return fProjectFile;
	}

	/**
	 * Query the predefined paths using gnatls and then reloads the construct
	 * database if needed. This is done in a separate job so that it's not
	 * blocking.
	 */
	public void recomputePredefinedPaths () {
		//  This has to be done before launching the job - this way if the
		//  kernel gets updated during the operation we won't have any gpr
		//  reading to do and won't have timestamps issues.
		final String gnatls = getGNATLS();

		if (gnatls.equals("")) {
			return;
		}

		String name = "";

		try {
			name = this.fRootProject.getName();
		} catch (GNATbenchCoreException e1) {
			GNATbenchCorePlugin.getDefault().logError(null, e1);
		}

		final Job loadPathJob = new Job("Load Ada Predefined Paths for "
				+ name) {

			protected IStatus run(IProgressMonitor monitor) {
				// This has to be done within a workbench job, so that the
				// preference is set when loaded.

				computePredefinedPaths(gnatls);
				reloadConstructDatabase();

				return new Status(IStatus.OK, GNATbenchCorePlugin.getId(),
						IStatus.OK, "", null);
			}

		};

		loadPathJob.setRule(getSecuredRule ());
		scheduleJob(loadPathJob);
	}


	private class SearchPaths {
		public final Filesystem_String objectPath;
		public final Filesystem_String sourcePath;
		public final Filesystem_String projectPath;
		public final String gnatVersion;

		public SearchPaths(final Filesystem_String object,
				final Filesystem_String source,
				final Filesystem_String project,
				final String version) {
			objectPath = object;
			sourcePath = source;
			projectPath = project;
			gnatVersion = version;
		}
	}


	/**
	 * The mapping of "gnatls" commands to the installed compilers'
	 * source and object paths.
	 */
	private static Map<String, SearchPaths> predefinedPathsCache =
		new HashMap<String, SearchPaths>();

	/**
	 * Call gnatls and set the predefined path.
	 */
	private void computePredefinedPaths (String gnatls) {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			if (predefinedPathsCache.containsKey(gnatls)) {
				SearchPaths paths = predefinedPathsCache.get(gnatls);

				fProjectRegistry.Set_Predefined_Source_Path(paths.sourcePath);
				fProjectRegistry.Set_Predefined_Object_Path(paths.objectPath);
				fProjectRegistry.Set_Predefined_Project_Path(paths.projectPath);
				fVersion = paths.gnatVersion;

				return;
			}
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}

		Process GNATLSProcess;

		try {
			GNATLSProcess = Runtime.getRuntime().exec(gnatls + " -v");
		} catch (IOException e) {
			try {
				createGNATLSMarker(fRootProject, gnatls);
			} catch (NullPointerException e1) {
				GNATbenchCorePlugin.getDefault().logError(null, e1);
			} catch (CoreException e2) {
				GNATbenchCorePlugin.getDefault().logError(null, e2);
			} catch (GNATbenchCoreException e3) {
				GNATbenchCorePlugin.getDefault().logError(null, e3);
			}
			return;
		}

		StringBuffer buffer;

		try {
			buffer = new StringBuffer ();
			GNATbenchCorePlugin.getDefault().timedProcess(GNATLSProcess, 5000);

			while (GNATLSProcess.getErrorStream().available() > 0) {
				buffer.append ((char) GNATLSProcess.getErrorStream().read());
			}

			while (GNATLSProcess.getInputStream().available() > 0) {
				buffer.append ((char) GNATLSProcess.getInputStream().read());
			}
		} catch (IOException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return;
		}

		String lines [] = buffer.toString().split("\\n");

		Pattern p_version = Pattern.compile("^GNATLS (.*)$");

		for (int i = 0; i < lines.length; ++i) {
			lines [i] = lines [i].replaceAll("\\n", "").replaceAll("\\r", "");
		}

		int lineNumber = 0;

		Filesystem_String sourceSearchPath;
		Filesystem_String objectSearchPath;
		Filesystem_String projectSearchPath;

		libMonitor = LibrarySemaphore.startGPSWork();

		try {
			sourceSearchPath = new Filesystem_String ("");
			objectSearchPath = new Filesystem_String ("");
			projectSearchPath = new Filesystem_String ("");
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}

		while (lineNumber < lines.length) {
			Matcher m_version = p_version.matcher(lines [lineNumber]);

			if (m_version.matches()) {
				fVersion = m_version.group(1);

				++lineNumber;
				break;
			}

			++lineNumber;
		}

		while (lineNumber < lines.length) {
			if (lines [lineNumber].equals ("Source Search Path:")) {
				++lineNumber;
				break;
			}

			++lineNumber;
		}

		// the lines of interest as the output of the call to gnatls is parsed
		LinkedList <String> lineList = new LinkedList <String> ();

		// parse the source search path entries
		while (lineNumber < lines.length) {
			if (lines[lineNumber].equals("Object Search Path:")) {
				// we are past the source search path entries
				sourceSearchPath = asDelimitedPath(lineList);
				lineList.clear();
				++lineNumber;
				break;
			} else if (lines[lineNumber].replaceAll(" ", "").equals("<Current_Directory>")) {
				// null
			} else if (!lines[lineNumber].equals("")) {
				// add to current list
				String path = lines[lineNumber].replaceAll("^ *", "");
				lineList.add(path);
			}
			++lineNumber;
		}

		// parse the object search path entries
		while (lineNumber < lines.length) {
			if (lines[lineNumber].equals("Project Search Path:")) {
				// we are past the object search path entries
				objectSearchPath = asDelimitedPath(lineList);
				lineList.clear();
				++lineNumber;
				break;
			} else if (lines[lineNumber].replaceAll(" ", "").equals("<Current_Directory>")) {
				// null
			} else if (!lines[lineNumber].equals("")) {
				// add to current list
				String path = lines[lineNumber].replaceAll("^ *", "");
				lineList.add(path);
			}
			++lineNumber;
		}

		// parse the project search path entries
		while (lineNumber < lines.length) {
			if (lines[lineNumber].replaceAll(" ", "").equals("<Current_Directory>")) {
				// null
			} else if (!lines[lineNumber].equals("")) {
				String path = lines[lineNumber].replaceAll("^ *", "");
				lineList.add(path);
			}
			++lineNumber;
		}
		projectSearchPath = asDelimitedPath(lineList);

		libMonitor = LibrarySemaphore.startGPSWork();

		try {
			fProjectRegistry.Set_Predefined_Source_Path(sourceSearchPath);
			fProjectRegistry.Set_Predefined_Object_Path(objectSearchPath);
			fProjectRegistry.Set_Predefined_Project_Path(projectSearchPath);

			predefinedPathsCache.put(
					gnatls,
					new SearchPaths(
							objectSearchPath,
							sourceSearchPath,
							projectSearchPath,
							fVersion));
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}

		//  In case the process is not yet destroyed.
		GNATLSProcess.destroy();
	}

	protected Filesystem_String asDelimitedPath(final LinkedList<String> lines) {
		StringBuffer buffer = new StringBuffer();
		for (String path : lines) {
			buffer.append(path + System.getProperty("path.separator"));
		} // for

		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			return new Filesystem_String(buffer.toString());
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	protected void createGNATLSMarker(final GPRProject GPR, final String gnatls)
		throws GNATbenchCoreException, CoreException
	{
		final IMarker marker;
		IResource resource;
		final IProject project = GPR.getEclipseProject();

		if (isWRSVersion) {
			// we put the marker on the project rather than the .gpr file because
			// the .gpr file does not control that setting in the WRS Workbench
			// version of GNATbench and it would be confusing to indicate otherwise.
			resource = project;
		} else {
			final String gprFileName = GPR.getFile().getOSPath();
			final GeneralizedFile gprFile = GeneralizedFile.fromOSPath(project,gprFileName);
			resource = gprFile.getFile();
			if (resource == null) {
				resource = project;
			}
		} // if

		marker = resource.createMarker(GNATbenchCorePlugin.GPR_ERROR_MARKER);
		marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
		marker.setAttribute(IMarker.TRANSIENT, true);
		if (gnatls == null || gnatls.equals("")) {
			marker.setAttribute(IMarker.MESSAGE,
					"The 'gnatls' command could not be determined");
		} else {
			marker.setAttribute(IMarker.MESSAGE,
					"The command \"" + gnatls + "\" cannot be launched");
		}
	}

	/**
	 * Perform an entire recomputation of the project hierarchy.
	 */
	public void recompute () {
		killJobs();

		registryLock.acquire();
		try {
			try {
				cleanResourcesProperties();
			} catch (GNATbenchCoreException e1) {
				GNATbenchCorePlugin.getDefault().logError(null, e1);
			} catch (CoreException e1) {
				GNATbenchCorePlugin.getDefault().logError(null, e1);
			}

			LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

			try {
				incrementTimestamp();

				BooleanRef reloaded = new BooleanRef();

				if (fValid) {
					fMessages.clear();

					fProjectRegistry.Reload_If_Needed(new Error_Report() {
						public void Error_Report_Body(AdaString Msg) {
							fMessages.add(new GPRMessage(Msg.toString(),
									GNATProjectRegistry.this));
						}
					}, reloaded);

					if (fProjectRegistry.Get_Root_Project().Status() == Project_Status.Empty) {
						fValid = false;
					} else {
						fValid = true;
					}
				} else {
					fMessages.clear();
					loadProjectFromPath();
				}

				if (fValid) {
					fMessages.clear();

					fProjectRegistry.Recompute_View(new Error_Report() {
						public void Error_Report_Body(AdaString Msg) {
							fMessages.add(new GPRMessage(Msg.toString(),
									GNATProjectRegistry.this));
						}
					});
				}

				fRootProject = new GPRProject(this, fProjectRegistry
						.Get_Root_Project());

				analyzeGPRErrors();

				if (fValid) {
					recomputePredefinedPaths();
				}
			} catch (GNATbenchCoreException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			} finally {
				LibrarySemaphore.stopGPSWork(libMonitor);
			}

			try {
				setResourcesProperties();
			} catch (GNATbenchCoreException e1) {
				GNATbenchCorePlugin.getDefault().logError(null, e1);
			} catch (CoreException e1) {
				GNATbenchCorePlugin.getDefault().logError(null, e1);
			}

			callRecompute();
		} finally {
			registryLock.release();
		}
	}

	@SuppressWarnings("unchecked")
	private void callRecompute() {
		HashSet<IGNATProjectRegistryListener> copy =
			(HashSet<IGNATProjectRegistryListener>) fGPRListeners
				.clone();

		for (IGNATProjectRegistryListener element : copy) {
			element.recomputed(this);
		}
	}

	public void addListener (IGNATProjectRegistryListener listener) {
		if (!fGPRListeners.contains(listener)) {
			fGPRListeners.add(listener);
		}
	}

	public void removeListener (IGNATProjectRegistryListener listener) {
		if (fGPRListeners.contains(listener)) {
			fGPRListeners.remove(listener);
		}
	}

	/**
	 * Computes the GNATls command for the given project, stripped of any arguments.
	 *
	 */
	public String getGNATLS () {
		if (fRootProject == null) {
			return "";
		}

		final IProject project = fRootProject.getEclipseProject();

		if (project == null) {
			// it is possible for this to be the case, specifically
			// during import-by-copy operations, but ignoring it is
			// OK because another attempt is made later
			return "";
		}

		try {
			return ToolChain.active().gnatlsCommand(project)[0];
		} catch (NoExtensionFound e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return "";
		}
	}

	/**
	 * Get a project from the kernel given in parameter.
	 */
	public GPRProject getProjectFromName(String name) {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			Project_Type prj = getProjectRegistry().Get_Project_From_Name(
					Editor_Package.Get_String_Name_Id(new AdaString(name)));

			if (!prj.equals(Projects_Package.No_Project())) {
				return new GPRProject(this, prj);
			}

			return null;
		} catch (GNATbenchCoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return null;
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Return a GPR project based on its path.
	 *
	 * @param path
	 * @return
	 */
	public GPRProject getProjectFromLocation(IPath path) {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			String projectName = path.lastSegment();
			projectName = projectName.substring(0, projectName.length() - 4);
			projectName = projectName.replaceAll("-", ".");

			return getProjectFromName(projectName);
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	public GPRProject getProjectFromURI(URI uri) {
		return getProjectFromLocation (URIUtil.toPath(uri));
	}

	/**
	 * Return the GPRProject corresponding to the given eclipse project
	 *
	 * @param project
	 * @return
	 */
	public GPRProject getGPRFor(IProject project) {
		GNATbenchProjectProperties props = GNATbenchProjectProperties
				.getPropertiesFor(project);
		String projectPath = props.getGPROSPath();

		if (projectPath != null && !projectPath.equals("")) {
			return getProjectFromLocation(new Path (projectPath));
		} else {
			return null;
		}
	}

	/**
	 * Return the current timestamp. This subprogram has to be called with the
	 * protection of the library semaphore. If not, GNATbenchCoreException
	 * will be thrown.
	 *
	 *
	 * @return
	 * @throws GNATbenchCoreException
	 */
	public int getTimestamp() throws GNATbenchCoreException {
		if (!LibrarySemaphore.isProtected()) {
			throw new GNATbenchCoreException("Call is not protected.");
		}

		return fTimestamp;
	}

	/**
	 * Increment the global gpr timestamp. Each time this timestamp is
	 * incremented, all the GPRProject are invalidated, and calling any of
	 * their procedure requiring to make JNI queries will throw a
	 * GNATbenchCoreException. This subprogram has to be called with the
	 * protection of the library semaphore. If not, GNATbenchCoreException
	 * will be thrown.
	 *
	 * @throws GNATbenchCoreException
	 */
	public void incrementTimestamp() throws GNATbenchCoreException {
		if (!LibrarySemaphore.isProtected()) {
			throw new GNATbenchCoreException("Call is not protected.");
		}

		fTimestamp += 1;
	}

	/**
	 * Return a rule that won't run during a workspace operation or another
	 * operation on this kernel. On some cases, a GNATProjectRegistry rule is
	 * enough - but this give a higher level on conflict information.
	 *
	 * @return
	 */
	public ISchedulingRule getSecuredRule () {
		return new MultiRule(new ISchedulingRule[] {
				ResourcesPlugin.getWorkspace().getRoot(),
				new GNATbenchKernelRule(this)});
	}

	/**
	 * Schedule a job that will works on the current state of the kernel. Such
	 * jobs are meant to be tight to this kernel lifetime - if a recompute is
	 * done, the job will be cancelled, and the actual recompute won't be
	 * performed until all jobs are finished.
	 *
	 * @param job
	 */
	public void scheduleJob (Job job) {
		synchronized (fJobs) {
			fJobs.add(job);
			job.schedule();
			job.addJobChangeListener(this);
		}
	}

	/**
	 * Cancels all jobs that have been scheduled through scheduleJob. Will
	 * wait until all jobs have been effectively killed.
	 *
	 */
	public void killJobs () {
		LinkedList <Job> jobsCopy;

		synchronized (fJobs) {
			jobsCopy = new LinkedList <Job> (fJobs);
			fJobs.clear ();
		}

		for (Job j : jobsCopy) {
			j.cancel();
		}

		for (Job j : jobsCopy) {
			try {
				if (j.getState() != Job.NONE) {
					j.join();
				}
			} catch (InterruptedException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		}
	}

	/**
	 * Return true if the kernel has been correctly loaded, false otherwise.
	 * Incorrect load may be due to errors in the GNAT project file. Project
	 * based operations are likely to fail on incorrect projects, so users
	 * should check it the kernel is valid before doing any operation on it.
	 *
	 * @return
	 */
	public boolean isValid () {
		return fValid;
	}

	/**
	 * Return true if this project is anonymous - i.e. not linked to an eclipse
	 * project.
	 */
	public boolean isAnonymous () {
		return fIsAnonymous;
	}

	/**
	 * Remove the previous error markers if needed, and add the one found in
	 * GPRErrors
	 *
	 */
	private void analyzeGPRErrors () {
		final LinkedList<IGNATProjectMessage> messages = new LinkedList<IGNATProjectMessage>(
				fMessages);

		Job gprMessage = new Job("Analyzing GNAT Project File") {

			protected IStatus run(IProgressMonitor monitor) {
				WorkspaceModifyOperation op = new WorkspaceModifyOperation(
						GNATProjectRegistry.this.getSecuredRule()) {

					protected void execute(IProgressMonitor monitor)
					throws CoreException,
					InvocationTargetException, InterruptedException {

						try {
							updateGPRErrorMarkers(messages);
						} catch (GNATbenchCoreException e) {
							GNATbenchCorePlugin.getDefault().logError(null,
									e);
						}
					}
				};

				try {
					op.run(null);
				} catch (InvocationTargetException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				} catch (InterruptedException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				}

				return GNATbenchCorePlugin.OK_STATUS;
			}
		};

		gprMessage.setRule(getSecuredRule());
		scheduleJob(gprMessage);
	}

	private void updateGPRErrorMarkers(LinkedList <IGNATProjectMessage> messages)
			throws GNATbenchCoreException
	{
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			if (fIsUnloaded) {
				return;
			}

			fRootProject.checkTimestamp();

			for (IResource file : fFilesWithMarkers) {
				IMarker[] markers;

				markers = file.findMarkers(
						GNATbenchCorePlugin.GPR_ERROR_MARKER, true,
						// include subclasses of the indicated marker
						// class.
						IResource.DEPTH_INFINITE);

				for (int i = 0; i < markers.length; ++i) {
					markers[i].delete();
				}
			}

			fFilesWithMarkers.clear();

			boolean shouldShowProblemsView = false;

			for (IGNATProjectMessage message : messages) {
				if (!isAnonymous()) {
					shouldShowProblemsView = true;
					fFilesWithMarkers.addAll(((GPRMessage) message)
							.createMarker());
				}
			}

			if (shouldShowProblemsView) {
				ViewUtils.showProblemsView();
				shouldShowProblemsView = false;
			}
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Wait for all the jobs currently launched from the kernel to finish.
	 */
	public void waitForAllJobs () {
		LinkedList <Job> jobsCopy;

		synchronized (fJobs) {
			jobsCopy = new LinkedList <Job> (fJobs);
		}

		for (Job j : jobsCopy) {
			try {
				if (j.getState() != Job.NONE) {
					j.join();
				}
			} catch (InterruptedException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		}
	}

	/**
	 * Return the assistant registered with the corresponding id.
	 *
	 * @param id
	 * @return
	 */
	public IGNATProjectRegistryAssistant getAssistant (String id) {
		return fAssistants.get(id);
	}

	/**
	 * Creates all the instances of the assistants registeres through the
	 * extension point and add them to the assistant list.
	 */
	private void configureAssistants () {
		IExtensionRegistry er = Platform.getExtensionRegistry();
		IExtensionPoint ep = er.getExtensionPoint(GNATbenchCorePlugin.getId(),
				"kernelAssistant");

		IExtension[] extensions = ep.getExtensions();

		if (extensions.length == 0) {
			return;
		}


		for (IExtension extension : extensions) {
			IConfigurationElement[] elements = extension
					.getConfigurationElements();

			for (IConfigurationElement conf : elements) {
				if (conf.getName().equals("assistant")) {
					try {
						Bundle bundle = Platform.getBundle(conf
								.getNamespaceIdentifier());

						String id = conf.getAttribute("id");

						IGNATProjectRegistryAssistant assistant =
							(IGNATProjectRegistryAssistant) bundle
							.loadClass(
									conf.getAttribute("class"))
									.newInstance();

						assistant.initialize(this);

						fAssistants.put(id, assistant);
					} catch (InvalidRegistryObjectException e) {
						GNATbenchCorePlugin.getDefault().logError (null, e);
					} catch (InstantiationException e) {
						GNATbenchCorePlugin.getDefault().logError (null, e);
					} catch (IllegalAccessException e) {
						GNATbenchCorePlugin.getDefault().logError (null, e);
					} catch (ClassNotFoundException e) {
						GNATbenchCorePlugin.getDefault().logError (null, e);
					}
				}
			}
		}
	}

	public boolean equals (Object cmp) {
		if (!(cmp instanceof GNATProjectRegistry)) {
			return false;
		}

		GNATProjectRegistry cmpKernel = (GNATProjectRegistry) cmp;

		return fProjectFile.equals (cmpKernel.fProjectFile);
	}

	public String getVersion () {
		return fVersion;
	}

	private void setResourcesProperties() throws GNATbenchCoreException,
			CoreException {
		switchResourceProperties(true);
	}

	private void cleanResourcesProperties() throws GNATbenchCoreException,
			CoreException {
		switchResourceProperties(false);
	}

	private void setProperty(IFileStore resource, QualifiedName property,
			boolean value) throws CoreException {

		GNATbenchCorePlugin.getDefault().getEFSRegistry().setSessionProperty(
				resource, property, value);
	}

	private void switchResourceProperties(boolean value)
			throws GNATbenchCoreException, CoreException {
		if (!isValid() || this.getRootProject() == null) {
			return;
		}

		LinkedList<IFileStore> sourcesFiles = this.getRootProject()
				.getSourceFiles(true);
		LinkedList<IFileStore> sourcesFolders = this.getRootProject()
				.getSourceDirectories(true);
		LinkedList<IFileStore> objectsFolders = this.getRootProject()
				.getObjectDirectories(true);

		for (IFileStore file : sourcesFiles) {
			setProperty(file, managedSourceFile, value);
		}

		for (IFileStore objectFolder : objectsFolders) {
			setProperty(objectFolder, managedBinaryFolder, value);
		}

		for (IFileStore sourceFolder : sourcesFolders) {
			setProperty(sourceFolder, managedSourceFolder, value);
		}

		IFileStore execDir = this.getRootProject().getExecDirectory();

		if (execDir != null) {
			setProperty(execDir, managedBinaryFolder, value);
		}
	}

	public static boolean isManagedResource(IResource res)
			throws GNATbenchCoreException {
		return isManagedResource(GNATbenchCorePlugin.getDefault()
				.getEFSRegistry().getUniqueStore(res.getLocation()));
	}

	public static boolean isManagedSourceFile(IResource res)
			throws GNATbenchCoreException {
		return isManagedSourceFile(GNATbenchCorePlugin.getDefault()
				.getEFSRegistry().getUniqueStore(res.getLocation()));
	}

	public static boolean isManagedSourceFolder(IResource res)
			throws GNATbenchCoreException {
		return isManagedSourceFolder(GNATbenchCorePlugin.getDefault()
				.getEFSRegistry().getUniqueStore(res.getLocation()));
	}

	public static boolean isManagedBinaryFolder(IResource res)
			throws GNATbenchCoreException {
		return isManagedBinaryFolder(GNATbenchCorePlugin.getDefault()
				.getEFSRegistry().getUniqueStore(res.getLocation()));
	}

	public static boolean isManagedResource (IFileStore res) {
		FileStoreRegistry reg = GNATbenchCorePlugin.getDefault().getEFSRegistry();

		Boolean p1 = (Boolean) reg.getSessionProperty(res, managedSourceFile);
		Boolean p2 = (Boolean) reg.getSessionProperty(res, managedSourceFolder);
		Boolean p3 = (Boolean) reg.getSessionProperty(res, managedBinaryFolder);

		return (p1 != null && p1.booleanValue())
			|| (p2 != null && p2.booleanValue())
			|| (p3 != null && p3.booleanValue());
	}

	public static boolean isManagedSourceFile(IFileStore res) {
		FileStoreRegistry reg = GNATbenchCorePlugin.getDefault().getEFSRegistry();

		Boolean p = (Boolean) reg.getSessionProperty(res, managedSourceFile);

		return p != null && p.booleanValue();
	}

	public static boolean isManagedSourceFolder(IFileStore res) {
		FileStoreRegistry reg = GNATbenchCorePlugin.getDefault().getEFSRegistry();

		Boolean p = (Boolean) reg.getSessionProperty(res, managedSourceFolder);

		return p != null && p.booleanValue();
	}

	public static boolean isManagedBinaryFolder(IFileStore res) {
		FileStoreRegistry reg = GNATbenchCorePlugin.getDefault().getEFSRegistry();

		Boolean p = (Boolean) reg.getSessionProperty(res, managedBinaryFolder);

		return p != null && p.booleanValue();
	}

	public void unload () {
		fValid = false;

		killJobs();

		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			fConstructDatabase.Destroy();
			fEntitiesDatabase.Destroy();
			fProjectRegistry.Destroy();

			fConstructDatabase.deallocateNativeObject();

			fConstructDatabase = null;
			fProjectRegistry = null;
			fEntitiesDatabase = null;
			fIsUnloaded = true;
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}

		callRecompute();
	}

	static {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			Registry_Package.Initialize();
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	public LinkedList<IGNATProjectMessage> getMessages() {
		return fMessages;
	}

	public void run (Runnable runnable) {
		registryLock.acquire();

		try {
			runnable.run();
		} finally {
			registryLock.release();
		}
	}

	public void aboutToRun(IJobChangeEvent event) {
		// TODO Auto-generated method stub

	}

	public void awake(IJobChangeEvent event) {
		// TODO Auto-generated method stub

	}

	public void done(IJobChangeEvent event) {
		fJobs.remove(event.getJob());
	}

	public void running(IJobChangeEvent event) {
		// TODO Auto-generated method stub

	}

	public void scheduled(IJobChangeEvent event) {
		// TODO Auto-generated method stub

	}

	public void sleeping(IJobChangeEvent event) {
		// TODO Auto-generated method stub

	}

	public void addMarkerFromBuild (IMarker marker) {
		fLastBuildMarkers.add(marker);
	}

	public void removeLastBuildMarkers () {
		for (IMarker marker : fLastBuildMarkers) {
			try {
				if (marker != null && marker.exists()) {
					marker.delete();
				}
			} catch (CoreException e) {
				//  Do not analyse exceptions here - if the marker is not
				//  here anymore, that's fine, just go to the next one.
			}
		}

		fLastBuildMarkers.clear();
	}
}
