/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal.jobs;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.text.BadLocationException;

import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.adaeditor.AdaDocumentBuffer;
import com.adacore.gnatbench.core.internal.analyzer.AdaEntityReference;
import com.adacore.gnatbench.core.internal.analyzer.Column;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedFile;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedLocation;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.core.internal.utils.FileUtils;
import com.adacore.gnatbench.core.internal.utils.ProjectUtils;

/**
 * This class handles a single message from the builder having at least a line
 * number but also an optional column number in the message.  Such messages
 * typically come from the compiler but can come from other tools too, such as
 * the GNAT project manager.
 * Subclasses must have a regex to match the message and set the different values
 * via getMessageInfo() which has to be overridden
 *
 */
public abstract class ToolMessage {

    private String fMessage;
    private IProject fProject;


    public ToolMessage(String message, IProject project){
    	fMessage = message;
    	fProject = project;
    }

    public final class ParseResult {
    	public int line;
    	public Column column;
    	public String caption;
    	public int severety = IMarker.SEVERITY_ERROR;
    	public String markerType;
    	public String fileName;
    }

    /**
     * Retrieve the relevant data from the message
     * Is suppose to set members
     * line
	 * column
	 * containsWarning
	 * caption
     * @param match
     */
	public abstract ParseResult parse();

    /**
     * Create the marker associated with the message
     * @param prj
     * @param match
     * @param num
     */
    final public IMarker createMarker() {
		ParseResult loc = parse ();

		if (loc == null) {
			return null;
		}

		IMarker marker = null;

	    try {
			if (loc.fileName != null
					&& ProjectUtils.isAdaSourceFileName(loc.fileName, fProject)) {
				GNATProjectRegistry registry = (GNATProjectRegistry) GNATbenchSession
						.getDefault().getOrLoadRegistry(fProject);

			    GeneralizedLocation fLocation = GeneralizedLocation.fromLineColumn(GeneralizedFile
						.fromOSPath(fProject, registry.getSourcePath(loc.fileName)
								.toOSString()), loc.line, loc.column);

				IFile resource = fLocation.getFile().getFile();

				if (resource == null) {
					return null;
				}

				marker = resource.createMarker(loc.markerType);

				AdaDocumentBuffer buffer = GNATbenchCorePlugin.getDefault()
					.getDocumentBuffer(fLocation.getFile());

				AdaEntityReference entity = null;

				try {
					entity = buffer.getEntity(fLocation);
				} catch (BadLocationException e) {
					//  given the fact that we get information from error
					//  messages, having a bad loc exception here is possible
					//  and not worth notifying the user.
				}

				if (entity != null) {
					marker.setAttribute(IMarker.CHAR_START, entity.getOffset());
					marker.setAttribute(IMarker.CHAR_END, entity.getOffset()
							+ entity.getName().length());
					marker.setAttribute(IMarker.LINE_NUMBER, entity.getLocation().getLine());
					marker.setAttribute(GNATbenchCorePlugin.LOCATION_MARKER_TEXT,
							entity.getName());
				} else {
					marker.setAttribute(IMarker.CHAR_START, 0);
					marker.setAttribute(IMarker.CHAR_END, 0);
					marker.setAttribute(GNATbenchCorePlugin.LOCATION_MARKER_TEXT, "");
					marker.setAttribute(IMarker.LINE_NUMBER, loc.line);
				} // if

			} else { // not an Ada file
				IResource resource = null;

				if (loc.fileName != null) {
					resource = FileUtils.findGlobally(fProject, loc.fileName);
				}

				if (resource == null) {
					resource = fProject;
				} // if

				marker = resource.createMarker(loc.markerType);
				marker.setAttribute(IMarker.CHAR_START, 0);
				marker.setAttribute(IMarker.CHAR_END, 0);
				marker.setAttribute(GNATbenchCorePlugin.LOCATION_MARKER_TEXT, "");
				marker.setAttribute(IMarker.LINE_NUMBER, loc.line);
			} // if

			marker.setAttribute(IMarker.SEVERITY, loc.severety);
			marker.setAttribute(IMarker.MESSAGE, loc.caption);
			marker.setAttribute(IMarker.TRANSIENT, true);
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		return marker;
	} // createMarker

    public String getMessage () {
    	return fMessage;
    }

    public IProject getProject () {
    	return fProject;
    }
} // ToolMessage
