/*******************************************************************************
 * Copyright (c) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal.jobs;

import java.util.regex.Pattern;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.analyzer.Column;
import com.adacore.gnatbench.core.internal.utils.FileUtils;
import com.adacore.gnatbench.core.internal.utils.ProjectUtils;


/**
 * Messages that do not include a line or column number, but
 * do include a filename.  Note that gcc tool messages that
 * _do_ include a line and optional column number are handled
 * elsewhere.
 *
 * For example:
 *     g-io.adb:(.text+0xc): undefined reference to `putchar'
 *     /bin/ld.exe: warning: cannot find entry symbol _start; defaulting to 01800074
 *
 */
public class GeneralToolMessage extends DefaultToolMessage {

	static private final String MESSAGE_FORMAT = "((\\b[a-z]:)?[^:*?\"<>|\r\n]*):(.*): (warning: )?([^:]+)$";

	static final Pattern MESSAGE_PATTERN  =
		Pattern.compile(MESSAGE_FORMAT, Pattern.CASE_INSENSITIVE);

	public GeneralToolMessage (String message, IProject project) {
		super (message, project);
	} // GeneralToolMessage

	@Override
	public ParseResult parse() {
		String message = getMessage();
		IProject project = getProject();

		ParseResult result = new ParseResult ();

	    result.markerType = GNATbenchCorePlugin.ADA_ERROR_MARKER;
	    result.column = new Column (0, Column.DEFAULT_GNAT_TAB_WIDTH);
	    result.line = 0;
	    result.severety = IMarker.SEVERITY_ERROR;

		final String [] msgParts = message.split(":");

		if (msgParts.length < 3) {
			return null;
		} // if

		String next;
		int last = msgParts.length - 1;

		result.caption = msgParts[last].trim();
		--last;

		next = msgParts[last].trim();
		if (next.equalsIgnoreCase("warning")) {
			result.severety = IMarker.SEVERITY_WARNING;
			--last;
		} else if (next.startsWith("(")) {
			// Eg: g-io.adb:(.text+0xc): undefined reference to `putchar'
			--last;
		} // if

		result.fileName = msgParts[last];
		--last;

		if (last >= 0 && (msgParts[last].length() == 1)) {
			// Windows drive letter
			result.fileName = msgParts[last] + ":" + result.fileName;
		} // if

	    IResource resource = null;

	    final boolean isAdaFile = ProjectUtils.isAdaSourceFileName(result.fileName, project);

	    if (isAdaFile) {
	    	resource = FileUtils.findGlobally(project, result.fileName);
	    } // if
	    if (resource == null) {
	    	// we did not find it (eg a file from adainclude) so put the
	    	// marker on the project itself
			resource = project;
		} // if

	    if (message.contains("(style)")) {
	    	result.severety = IMarker.SEVERITY_WARNING;
	    }

	    if (isAdaFile && resource == project) {
	    	// we have a useful name but cannot use it as a
	    	// Problems view location so we put in in the caption
	    	result.caption = result.caption + result.fileName + ": " + result.caption;
	    } // if

		return result;
	}

} // GeneralToolMessage
