/*****************************************************************************
 * Copyright (C) 2008-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *****************************************************************************/

package com.adacore.gnatbench.core.internal.jobs;

import java.io.IOException;
import java.text.DateFormat;
import java.util.LinkedList;
import java.util.Map;
import java.util.TreeMap;

import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.resources.WorkspaceJob;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.InvalidRegistryObjectException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.ui.PlatformUI;
import org.osgi.framework.Bundle;

import com.adacore.gnatbench.core.GNATbenchCoreException;
import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedFile;
import com.adacore.gnatbench.core.internal.builder.AdaBuilderPreferencesProvider;
import com.adacore.gnatbench.core.internal.builder.BuildConsole;
import com.adacore.gnatbench.core.internal.builder.BuildConsoleManager;
import com.adacore.gnatbench.core.internal.builder.CompilationEvent;
import com.adacore.gnatbench.core.internal.builder.ConsolePrinter;
import com.adacore.gnatbench.core.internal.builder.IGNATbenchBuildListener;
import com.adacore.gnatbench.core.internal.filesystem.FileStoreRegistry;
import com.adacore.gnatbench.core.internal.gpswrappers.GPRProject;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistryFactory;
import com.adacore.gnatbench.core.internal.utils.ProjectUtils;
import com.adacore.gnatbench.core.internal.utils.ViewUtils;
import com.adacore.gnatbench.core.projects.IGNATProjectView;
import com.adacore.gnatbench.core.projects.IScenarioVariable;

/**
 * This class will be extended to get user selection
 * and apply command on the selected files
 * The selection have to be managed to be either
 * Files or IFileStore
 */
abstract public class CommandFilesJob extends WorkspaceJob {

	/** The resources to apply the commands */
	private IFileStore [] validResources = null;

	/**
	 * The StandardMessage associated to the command
	 */
	private final Class <? extends ToolMessage> fMessageClass;

	/**
	 * When this flag is set to true, various views (e.g. problems / console)
	 * may be open when the build is done - otherwise they aren't
	 */
	private boolean fRaiseViews = true;

	/**
	 * @param doAutoSave should the constructor issue an auto-save (interactive
	 * or automatic based on the
	 * @AdaBuilderPreferencesProvider#ADA_SAVE_DIRTY_ON_BUILD value.
	 * @param commandName a name for the command (ie: Ada build)
	 * @param resources the resources to apply the command
	 * if the resource is a project, the project file will be added
	 */
	public CommandFilesJob(final String name,
			Class<? extends ToolMessage> messageClass, IFileStore[] resources,
			boolean doAutoSave) {
		super(name);

		fMessageClass = messageClass;

		setResources(resources);

		if (doAutoSave) {
			final boolean autoSavePrefEnabled = GNATbenchCorePlugin.getDefault()
			.getPreferenceStore().getBoolean(
					AdaBuilderPreferencesProvider.ADA_SAVE_DIRTY_ON_BUILD);

			PlatformUI.getWorkbench().saveAllEditors(!autoSavePrefEnabled);
		}

	} // ctor

	public abstract void run(
			TreeMap<IGNATProjectView, LinkedList<IFileStore>> files,
			IProgressMonitor monitor);

	/**
	 * Create a process with the environment set up taking into account
	 * environment variables extracted from the GNAT project.
	 * @throws GNATbenchCoreException
	 */
	public void runProcessInEnv(GPRProject project, String commandName,
			String[] command, IFileStore workingDir, IProgressMonitor monitor)
			throws GNATbenchCoreException {

		final String [] env;

		try {
			// Load environment vars (to get any scenario vars but also the
			// path, etc) and also load those scenario vars set by the user
			// within Workbench.
			env = environment(project);
		} catch (GNATbenchCoreException e) {
			throw new GNATbenchCoreException("Could not extract environment");
		}

		BuildConsole console = BuildConsoleManager.findConsole(commandName
				+ " [" + project.getName() + "]");
		// we don't use the command in the console name because that
		// would create too many consoles.

		if (fRaiseViews) {
			ViewUtils.showConsoleView();
		}

		console.clearConsole();

		// display the command in the console
		for (String arg : command) {
			console.print(arg + " ");
		}

		console.print ("\n");

		Process proc;

		try {
			proc = Runtime.getRuntime()
					.exec(
							command,
							env,
							workingDir.toLocalFile(EFS.NONE,
									new NullProgressMonitor()));
		} catch (IOException e) {
			throw new GNATbenchCoreException("Could not execute command '"
					+ commandName + "'");
		} catch (CoreException e) {
			throw new GNATbenchCoreException("Could not execute command '"
					+ commandName + "'");
		}

		ConsolePrinter errorMessages = new ConsolePrinter(
				proc.getErrorStream(), monitor, console);
		ConsolePrinter normalMessages = new ConsolePrinter(proc
				.getInputStream(), monitor, console);

		errorMessages.start();
		normalMessages.start();

		try {
			proc.waitFor();
		} catch (InterruptedException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		final DateFormat df = DateFormat.getDateTimeInstance(DateFormat.MEDIUM,
				DateFormat.FULL);
		final String now = df.format(new java.util.Date());
		console.print(commandName + " for [" + project.getName()
				+ "] completed " + now + ".");

		// we need to refresh the object directories since there may be new files
		// there that are referenced in the error/warning messages (eg binder-
		// generated Ada files).  this must be done before we signal the completion
		// of the compilation since that is when the error messages are processed.
		refreshObjectDirs(project, monitor);

		String [] errorMessagesStr = errorMessages.getMessages();
		String [] normalMessagesStr = normalMessages.getMessages();

		String[] messages = new String[errorMessagesStr.length
				+ normalMessagesStr.length];

		for (int j = 0; j < errorMessagesStr.length; ++j) {
			messages [j] = errorMessagesStr [j];
		}

		for (int j = errorMessagesStr.length; j < messages.length; ++j) {
			messages [j] = normalMessagesStr [j - errorMessagesStr.length];
		}

		signalCompilationCompleted(project, messages, monitor);
	}


	/**
	 * @param commandsName
	 * @param resource
	 */
	public CommandFilesJob(final String name,
			Class<? extends ToolMessage> messageClass, IFileStore resource,
			boolean doAutoSave) {
		this(name, messageClass, new IFileStore[] { resource }, doAutoSave);
	} // ctor


	/**
	 * add non-null resources to validResources and, for those,
	 * remove the annotations in the corresponding project
	 * @param resources
	 */
	private void setResources(IFileStore[] resources){
		LinkedList<IFileStore> list = new LinkedList<IFileStore>();

		for (int j = 0; j < resources.length; j++) {
			if (resources[j] != null){
				list.add(resources[j]);
			} // if
		}//for

		validResources = list.toArray(new IFileStore[0]);
	}

	@Override
	final public IStatus runInWorkspace(IProgressMonitor monitor) {
		TreeMap<IGNATProjectView, LinkedList<IFileStore>> files = groupFiles();

		run(files, monitor);

		return GNATbenchCorePlugin.OK_STATUS;
	}

	public TreeMap<IGNATProjectView, LinkedList<IFileStore>> groupFiles() {
		TreeMap<IGNATProjectView, LinkedList<IFileStore>> res = new TreeMap<IGNATProjectView, LinkedList<IFileStore>>();

		for (int i = 0; i < validResources.length; i++) {
			final IFileStore element = validResources[i];

			final IProject project = FileStoreRegistry
					.getDefaultProject(element);

			// the command is aborted if the project file contains error(s)
			if (ProjectUtils.GPRHasErrors(project)) {
				GNATbenchCorePlugin.getDefault().logError(
						"Errors are found in the project file", null);
				// we display the problem view so the user directly know what is
				// wrong

				if (fRaiseViews) {
					ViewUtils.showProblemsView();
				}

				continue;
			}

			GNATProjectRegistry reg = (GNATProjectRegistry)
			((GNATProjectRegistryFactory) GNATbenchSession
					.getDefault().getProjectRegistryFactory())
					.getOrLoadRegistry(project);

			if (reg == null) {
				continue;
			}

			GPRProject gpr = reg.getGPRFor(project);

			if (gpr == null) {
				continue;
			}

			LinkedList <IFileStore> fileList;

			if (!res.containsKey(gpr)) {
				res.put(gpr, new LinkedList <IFileStore> ());
			}

			fileList = res.get(gpr);
			fileList.add(element);
		}

		return res;
	}

	/**
	 * Create an array of IFileStore from an array of Object
	 * Only instances of GeneralizedFile, IFileStore or IResources
	 * will be managed
	 * @param selection
	 */
	public static IFileStore[] selectionToIFileStore(Object[] selection){
		LinkedList<IFileStore> list = new LinkedList<IFileStore>();

		for (int j = 0; j < selection.length; j++){
			if (selection[j] instanceof IResource) {
				try {
					list.add(GNATbenchCorePlugin.getDefault().getEFSRegistry()
							.getUniqueStore(
									((IResource) selection[j]).getLocation()));
				} catch (GNATbenchCoreException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				}
			} else if (selection[j] instanceof GeneralizedFile) {
				list.add(((GeneralizedFile) selection[j]).getFileStore());
			} else if (selection[j] instanceof IFileStore) {
				list.add((IFileStore) selection[j]);
			}
		}

		return list.toArray(new IFileStore[0]);
	}//selectionToIFileStore

	public void setRaiseViews (boolean raiseViews) {
		fRaiseViews = raiseViews;
	}

	/**
	 * Returns the environment variables defined by the user process and
	 * those that are scenario variables defined by the fProject.  Any
	 * scenario vars with the same names as environment variables overwrite
	 * those environment variables so that they take precedence.
	 */
	private String[] environment(GPRProject project)
			throws GNATbenchCoreException {

		String [] result = null;
		Map <String, String> allVars = new TreeMap <String, String> ();

		// put the process's environment variables into the map.
		// allVars will not be null on return but may be empty.
		try {
			allVars.putAll(System.getenv());
		} catch (Throwable e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return null;
		} // try

		// now put the fProject's scenario variables in, if any are defined,
		// overwriting any values for corresponding environment variables
		project.loadScenarioVariables();
		LinkedList <IScenarioVariable> scenarioVars = project.getScenarioVariables();

		for (IScenarioVariable var : scenarioVars) {
			allVars.put(var.getExternalName(), var.getValue());
		} // for

		// put all into the result, each in the form key=value
		int mapsize = allVars.size();
		if (mapsize != 0) {
			result = new String[mapsize];
		} // if

		int k = 0;
		for (Map.Entry <String, String> entry : allVars.entrySet()) {
			String key = entry.getKey();
			String value = entry.getValue();
			result[k] = key + "=" + value;
			k = k + 1;
		} // for

		return result;
	} // environment

	// refresh all the object directories (ie including the dependencies)
	protected void refreshObjectDirs(final GPRProject project,
			IProgressMonitor monitor) {
		try {
			final LinkedList<IFileStore> objDirs = project.getObjectDirectories(true);
			final IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
			for (IFileStore dir : objDirs) {
				monitor.setTaskName("Refreshing " + dir.toURI());

				final IContainer[] containers = root
						.findContainersForLocationURI(dir.toURI());

				for (IContainer cont : containers) {
					cont.refreshLocal(IResource.DEPTH_ONE, monitor);
				}
			} // for
		} catch (GNATbenchCoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} // try
	} // refreshObjectDirs

	/**
	 * Notifies listeners to buildListener extension point that the build is
	 * finished.
	 *
	 * @param compiledFiles
	 * @param errors
	 * @param monitor
	 */
	protected void signalCompilationCompleted
	   (final GPRProject project, final String[] errors, final IProgressMonitor monitor)
	{
		if (errors == null) {
			GNATbenchCorePlugin.getDefault()
				.logError("Null errors passed to signalCompilationCompleted", null);
			return;
		} // if

		IExtensionRegistry er = Platform.getExtensionRegistry();
		IExtensionPoint ep = er.getExtensionPoint(GNATbenchCorePlugin.getId(),
				"buildListener");

		IExtension[] extensions = ep.getExtensions();

		if (extensions.length == 0) {
			return;
		} // if

		monitor.setTaskName("Analyzing builder messages");

		CompilationEvent event = new CompilationEvent ();

		event.messages = errors;
		event.project = project.getEclipseProject();
		event.rootMessageClass = fMessageClass;
		event.raiseViews = fRaiseViews;

		for (int extInd = 0; extInd < extensions.length; ++extInd) {

			IConfigurationElement[] elements = extensions[extInd].getConfigurationElements();

			for (int i = 0; i < elements.length; ++i) {
				if (elements[i].getName().equals("listener")) {
					try {
						Bundle bundle = Platform.getBundle(extensions[extInd]
								.getNamespaceIdentifier());

						final IGNATbenchBuildListener builderListener =
							(IGNATbenchBuildListener) bundle
								.loadClass(elements[i].getAttribute("class"))
								.newInstance();

						builderListener.compilationCompleted(event, monitor);

					} catch (InvalidRegistryObjectException e) {
						GNATbenchCorePlugin.getDefault().logError(null, e);
					} catch (InstantiationException e) {
						GNATbenchCorePlugin.getDefault().logError(null, e);
					} catch (IllegalAccessException e) {
						GNATbenchCorePlugin.getDefault().logError(null, e);
					} catch (ClassNotFoundException e) {
						GNATbenchCorePlugin.getDefault().logError(null, e);
					} // try
				} // if
			} // for
		} // for

		for (IGNATbenchBuildListener listener : GNATbenchCorePlugin
				.getDefault().getBuildListeners()) {
			listener.compilationCompleted(event, monitor);
		}
	} // signalCompilationCompleted

} // CommandFilesJob
