/*****************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *****************************************************************************/

package com.adacore.gnatbench.core.internal.gpswrappers;

import java.util.LinkedList;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.QualifiedName;

import com.adacore.gnatbench.core.GNATbenchCoreException;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.projects.IScenarioVariable;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.Projects.Scenario_Variable_Array;
import com.adacore.gnatbench.library.Standard.AdaString;

/**
 * This class represent a scenario variable.
 */
public class ScenarioVariable implements IScenarioVariable {

	private String fName;
	private String fValue;
	private LinkedList <String> fPossibleValues;
	private GPRProject fProject;

	final static String valueBaseName = "_scenario_var_";

	/**
	 * Creates a scenario variable, and updates the state of GPS from the
	 * information stored in the project if needed.
	 *
	 * @param project This project must be already bounded with an eclipse
	 *  project
	 * @param name
	 * @param value
	 * @param possibleValues
	 */
	public ScenarioVariable(GPRProject project, String name, String value,
			LinkedList <String> possibleValues) {
		fName = name;
		fValue = value;
		fPossibleValues = possibleValues;
		fProject = project;
	}

	/**
	 * Return the name of the variable.
	 *
	 * @return
	 */
	public String getExternalName () {
		return fName;
	}

	/**
	 * Return the current value of the variable.
	 *
	 * @return
	 */
	public String getValue () {
		return fValue;
	}

	/**
	 * Return the GPR project of the variable.
	 * @return
	 */
	public GPRProject getGPRProject() {
		return fProject;
	}


	/**
	 * Return all the values that the variable can take.
	 *
	 * @return
	 */
	public LinkedList <String> getPossibleValues () {
		return fPossibleValues;
	}

	/**
	 * Defines the value of the variable. If it's different from the current
	 *  value, then both the eclipse project and the running GPS will be
	 *  updated.
	 *
	 * @param value
	 */
	public void setValue (String value) throws GNATbenchCoreException {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			Scenario_Variable_Array vars = fProject.getGPRProject()
					.Find_Scenario_Variables(true);

			for (int k = vars.First(); k <= vars.Last(); ++k) {
				if (vars.Get_Element_At(k).External_Reference_Of().toString()
						.equals(fName)) {
					fValue = value;

					vars.Get_Element_At(k).Set_Value(new AdaString (value));

					break;
				}
			}

			QualifiedName qName = fProject.getQualifiedName(getExternalName());

			ResourcesPlugin.getWorkspace().getRoot().setPersistentProperty(
					qName, value);
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}
}
