/*****************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *****************************************************************************/

package com.adacore.gnatbench.core.internal.gpswrappers;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.LinkedList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.QualifiedName;

import com.adacore.gnatbench.core.GNATbenchCoreException;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.GNATbenchProjectProperties;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedFile;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.core.projects.IGNATProjectView;
import com.adacore.gnatbench.core.projects.IScenarioVariable;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.Prj.PP.Write_Char_Ap;
import com.adacore.gnatbench.library.Prj.PP.Write_Eol_Ap;
import com.adacore.gnatbench.library.Prj.PP.Write_Str_Ap;
import com.adacore.gnatbench.library.Projects.Imported_Project_Iterator;
import com.adacore.gnatbench.library.Projects.Project_Type;
import com.adacore.gnatbench.library.Projects.Projects_Package;
import com.adacore.gnatbench.library.Projects.Scenario_Variable_Array;
import com.adacore.gnatbench.library.Projects.Editor.Editor_Package;
import com.adacore.gnatbench.library.Projects.Registry.Registry_Package;
import com.adacore.gnatbench.library.System.Strings.String_List;
import com.adacore.gnatbench.library.GNATCOLL.VFS.File_Array;
import com.adacore.gnatbench.library.GNATCOLL.VFS.VFS_Package;
import com.adacore.gnatbench.library.GNATCOLL.VFS.Virtual_File;
import com.adacore.ajis.IProxy;
import com.adacore.ajis.IProxy.Owner;
import com.adacore.gnatbench.library.Standard.AdaString;
import com.adacore.gnatbench.library.Utils.Utils_Package;

public class GPRProject implements IGNATProjectView, Comparable <GPRProject> {

	private LinkedList <IScenarioVariable> fScenarioVariables;

	private Project_Type fGPR;
	IFileStore fProject;

	private int fTimestamp;

	private GNATProjectRegistry fRegistry;

	private GNATbenchProjectProperties fProperties;

	/**
	 * Check that the current GPR project is still up to date regarding the
	 * project registry. Will throw a GNATbenchCoreException otherwise. This
	 * subprogram has to be called with the protection of the library
	 * semaphore. If not, GNATbenchCoreException will be thrown.
	 * @throws GNATbenchCoreException
	 */
	public synchronized void checkTimestamp() throws GNATbenchCoreException {
		if (!LibrarySemaphore.isProtected()) {
			throw new GNATbenchCoreException("Call is not protected.");
		}

		if (fTimestamp != fRegistry.getTimestamp()) {
			throw new GNATbenchCoreException(
					"GPR reference is older than the project registry.");
		}
	}

	public GPRProject(GNATProjectRegistry registry, Project_Type gpr)
			throws GNATbenchCoreException {

		fRegistry = registry;

		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			fTimestamp = fRegistry.getTimestamp();

			fGPR = gpr;

		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Returns the project interface associated to this project.
	 * @return
	 */
	public IFileStore getProjectFile () {
		if (fProject == null) {
			LibraryMonitor monitor = LibrarySemaphore.startGPSWork();

			try {
				checkTimestamp();

				fProject = GNATbenchCorePlugin.getDefault().getEFSRegistry()
						.getUniqueStore(
								URIUtil.toURI(fGPR
										.Project_Path(new AdaString(""))
										.Full_Name(true).toString()));
			} catch (GNATbenchCoreException e) {
				//  XXX: This exception should be raised to the caller.
				GNATbenchCorePlugin.getDefault().logError(null, e);

				return null;
			} finally {
				LibrarySemaphore.stopGPSWork(monitor);
			}
		}

		return fProject;
	}

	/**
	 * Return the eclipse project associated to this GNAT view - null if no
	 * eclipse project matches this definition.
	 */
	public IProject getEclipseProject () {
		return GNATbenchProjectProperties
				.getProjectForURI(getProjectFile().toURI());
	}

	/**
	 * Returns the sources directories associated to this project.
	 * @return
	 */
	public LinkedList <IFileStore> getSourceDirectories(boolean recursive)
			throws GNATbenchCoreException {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			checkTimestamp();

			LinkedList<IFileStore> result = new LinkedList<IFileStore>();

			String_List directories = fGPR.Source_Dirs (recursive, false);

			if (directories.Length() > 0) {
				for (int i = directories.First(); i <= directories.Last(); ++i) {
					Virtual_File vFile = VFS_Package
							.Create(Utils_Package
									.To_Filesystem_String(directories
											.Get_Element_At(i)));

					IPath path = new Path (vFile
							.Full_Name(true).toString());

					if (GNATbenchCorePlugin.getDefault().getEFSRegistry()
							.exists(path)) {
						result.add(GNATbenchCorePlugin.getDefault()
								.getEFSRegistry().getUniqueStore(path));
					}
				}
			}

			return result;
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Return the list of the sources files for the given project
	 *
	 * @param recursive
	 * @return
	 */
	public LinkedList <IFileStore> getSourceFiles(boolean recursive)
			throws GNATbenchCoreException {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			checkTimestamp();

			File_Array prjFiles = fGPR.Get_Source_Files(recursive);
			LinkedList <IFileStore> result = new LinkedList <IFileStore> ();

			if (prjFiles.Length() > 0) {
				for (int i = prjFiles.First(); i <= prjFiles.Last(); ++i) {
					Virtual_File vFile = prjFiles.Get_Element_At(i);

					IPath path = new Path (vFile.Full_Name(
							true).toString());

					if (GNATbenchCorePlugin.getDefault().getEFSRegistry()
							.exists(path)) {
						result.add(GNATbenchCorePlugin.getDefault()
								.getEFSRegistry().getUniqueStore(path));
					}
				}
			}

			VFS_Package
					.Unchecked_Free((File_Array.Ref) prjFiles.NewProxyRef());

			return result;
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Returns the object directories associated to this project.
	 *
	 * @return
	 */
	public LinkedList<IFileStore> getObjectDirectories(boolean recursive)
			throws GNATbenchCoreException {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			checkTimestamp();

			LinkedList<IFileStore> result = new LinkedList<IFileStore>();

			String directory = fGPR.Object_Path(recursive, false,
					false).toString();

			final String separator = System.getProperty("path.separator");

	        final String [] parts = directory.split(separator);
	        for (int k = 0; k < parts.length; k++) {

	        	if (!parts [k].equals("")) {
	        		//  It's important to explicitely avoid using empty parts,
	        		//  otherwise some random root directory will get used,
	        		//  and slow down processes of e.g. importing or updating
	        		//  the object directory.

		        	IPath path = new Path (parts [k]);

		        	if (GNATbenchCorePlugin.getDefault().getEFSRegistry().exists(
							path)) {
						result.add(GNATbenchCorePlugin.getDefault()
								.getEFSRegistry().getUniqueStore(path));
					}
	        	}
			}

			return result;
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	public IFileStore getExecDirectory () throws GNATbenchCoreException {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			IPath fullPath = new Path (getAttributeValue("Exec_Dir", "", ""));

			if (fullPath.isEmpty()) {
				return null;
			} else {
				if (!fullPath.isAbsolute()) {
					fullPath = getFile().getLocation().removeLastSegments(1)
							.append(fullPath);
				}

				if (GNATbenchCorePlugin.getDefault().getEFSRegistry().exists(
						fullPath)) {
					return GNATbenchCorePlugin.getDefault().getEFSRegistry()
							.getUniqueStore(fullPath);
				} else {
					return null;
				}
			}
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Return the project file corresponding to this project
	 *
	 * @return
	 */
	public GeneralizedFile getFile() throws GNATbenchCoreException {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			checkTimestamp();

			IProject prj = getEclipseProject ();

			return GeneralizedFile.fromVirtualFile(prj, fGPR
					.Project_Path(new AdaString("")));
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Return this project name
	 *
	 * @return
	 */
	public String getName() throws GNATbenchCoreException {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			checkTimestamp();

			return fGPR.Project_Name_String().toString();
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Return all the scenario variables defined in the project
	 *
	 * @return
	 */
	public LinkedList <IScenarioVariable> getScenarioVariables()
			throws GNATbenchCoreException {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			checkTimestamp();

			return fScenarioVariables;
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Loads all the scenario variables for the given project, according to the
	 * information stored in a specified .gpr project file and .gb_project file.
	 * The .gpr file defines their existence and characteristics, and the
	 * .gb_project file determines their current values
	 *
	 * @return
	 */
	public void loadScenarioVariables() throws GNATbenchCoreException {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			checkTimestamp();

			Scenario_Variable_Array vars = fGPR.Find_Scenario_Variables(true);

			LinkedList <IScenarioVariable> variables = new LinkedList <IScenarioVariable> ();

			for (int i = vars.First(); i <= vars.Last(); ++i) {
				String_List values = vars.Get_Element_At(i)
					.Enum_Values_Of(fRegistry.getProjectRegistry());

				LinkedList <String> possibleValues = new LinkedList <String> ();

				// TODO: Free values !

				for (int j = values.First(); j <= values.Last(); ++j) {
					possibleValues.add (values.Get_Element_At(j).toString());
				}

				final String name = vars.Get_Element_At(i).External_Reference_Of().toString();
				final String initialValue = vars.Get_Element_At(i).Value_Of().toString();

				variables.add(new ScenarioVariable(this, name, initialValue,
						possibleValues));
			}

			// set the value of any of these variables based on user settings found
			// in the properties
			for (IScenarioVariable variable : variables) {
				final String name = variable.getExternalName();

				String val = ResourcesPlugin.getWorkspace().getRoot()
						.getPersistentProperty(getQualifiedName(name));

				if (val != null) {
					variable.setValue(val);
				}
			}

			fScenarioVariables = variables;
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Return a list of dependencies on the given project
	 *
	 * @return
	 */
	public LinkedList <IGNATProjectView> getDependencies(boolean recursive)
			throws GNATbenchCoreException {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			checkTimestamp();

			Imported_Project_Iterator it = fGPR.Start(true, !recursive, true);

			it = fGPR.Start(true, !recursive, true);

			LinkedList <IGNATProjectView> result = new LinkedList <IGNATProjectView> ();

			int i = 0;

			while (true) {
				Project_Type prj = it.Current();

				if (prj.equals(Projects_Package.No_Project ())) {
					break;
				} else if (prj.equals(fGPR)) {
					it.Next();
					continue;
				}

				result.add(new GPRProject (getRegistry(), prj));
				it.Next();
				i = i + 1;
			}

			return result;
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * If any Eclipse project is bounded with the given gpr project path, then
	 * return it, otherwise return false.
	 *
	 * @param gprName
	 * @return
	 */
	public static IProject getEclipseProjectForGPRPath(String gprPath) throws CoreException {
		IProject [] projects  =
			ResourcesPlugin.getWorkspace().getRoot().getProjects();

		for (int i = 0; i < projects.length; ++i) {
			GNATbenchProjectProperties props = GNATbenchProjectProperties
					.getPropertiesFor(projects[i]);
			String propertyPath = props.getGPROSPath();

			if (gprPath.equals(propertyPath)) {
				return projects [i];
			}
		}

		return null;
	}

	/**
	 * If any Eclipse project is bounded with the given gpr project name, then
	 * return it, otherwise return false.
	 *
	 * @param gprName
	 * @return
	 */
	public static IProject getEclipseProjectForGPRName(String gprName) throws CoreException {
		IProject [] projects  =
			ResourcesPlugin.getWorkspace().getRoot().getProjects();

		for (int i = 0; i < projects.length; ++i) {
			GNATbenchProjectProperties props = GNATbenchProjectProperties
					.getPropertiesFor(projects[i]);
			String propertyPath = props.getGPROSPath();

			if (propertyPath != null
					&& extractNameFromPath(gprName).toLowerCase().equals(
							extractNameFromPath(propertyPath).toLowerCase())) {
				return projects [i];
			}
		}

		return null;
	}

	/**
	 * Remove a dependency from a GPR project.
	 *
	 * @param path The complete OS path of the dependency to remove.
	 */
	public void removeDependency(String path) throws GNATbenchCoreException {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			checkTimestamp();

			GPRProject removedProject = fRegistry
					.getProjectFromLocation(new Path(path));

			Editor_Package.Remove_Imported_Project(fGPR, removedProject
					.getGPRProject());
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	public void removeDependency(Project_Type project)
			throws GNATbenchCoreException {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			checkTimestamp();

			Editor_Package.Remove_Imported_Project(fGPR, project);
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Set whether the project should be active in the GNATbench project
	 * registry or not.
	 *
	 * @param project
	 * @param path
	 */
	public static void setActive (IProject project, boolean active) {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			QualifiedName name = new QualifiedName(
					GNATbenchCorePlugin.getId(), "isActive");

			try {
				project.setPersistentProperty(name, Boolean.valueOf(active)
						.toString());
			} catch (CoreException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}


	/**
	 * Return whether the project should be active in the GNATbench project
	 * registry or not.
	 *
	 * @param project
	 * @return
	 */
	public static boolean isActive (IProject project) {
		QualifiedName name = new QualifiedName(
				GNATbenchCorePlugin.getId(), "isActive");

		try {
			String prop = project.getPersistentProperty(name);

			if (prop == null) {
				return true;
			}

			return Boolean.valueOf(prop).booleanValue();
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		return true;
	}

	/**
	 * According to a path, guess the most probable name of the project.
	 * @param path
	 * @return
	 */
	static public String extractNameFromPath (String path) {
		Pattern namePattern	= Pattern.compile ("[\\\\/]?([a-zA-Z0-9_]+)\\.gpr$");
		Matcher matcher = namePattern.matcher(path);

		if (matcher.find()) {
			return matcher.group(1);
		} else {
			return path;
		}


	}

	/**
	 * Return an attribute of the current project.
	 *
	 * @param attributeName
	 * @param packageName
	 * @param indexName
	 * @return
	 */
	public String[] getAttributeAsList(String attributeName,
			String packageName, String indexName) throws GNATbenchCoreException {

		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			checkTimestamp();

			String_List list = fGPR.Get_Attribute_Value
			(Projects_Package.Build
					(new AdaString (packageName),
							new AdaString (attributeName)),
							new AdaString (indexName), false);

			String [] result = new String [list.Last() - list.First() + 1];

			for (int i = list.First(); i <= list.Last(); ++i) {
				result [i - list.First()] = list.Get_Element_At(i).toString();
			}

			return result;
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Return the value of an attribute
	 *
	 * @param attributeName
	 * @param packageName
	 * @param indexName
	 */
	public String getAttributeValue(String attributeName, String packageName,
			String indexName) throws GNATbenchCoreException {

		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			checkTimestamp();

			String_List list = fGPR.Get_Attribute_Value
			(Projects_Package.Build
					(new AdaString (packageName),
							new AdaString (attributeName)),
							new AdaString (indexName), false);

			if (list.Length() == 0) {
				return null;
			} else {
				return list.Get_Element_At(list.First()).toString();
			}
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	public Project_Type getGPRProject () throws GNATbenchCoreException {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			checkTimestamp();

			return fGPR;
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	public void setAttributeValue(String attributeName, String packageName,
			String indexName, String value) throws GNATbenchCoreException {

		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			checkTimestamp();

			Editor_Package.Update_Attribute_Value_In_Scenario(fGPR,
					Projects_Package.No_Scenario(), Projects_Package.Build(
							new AdaString(packageName), new AdaString(
									attributeName)), new AdaString(value),
					new AdaString(indexName));
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	public void setAttributeValue(String attributeName, String packageName,
			String indexName, String [] values) throws GNATbenchCoreException  {

		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			checkTimestamp();

			if (values.length == 0) {
				Editor_Package.Delete_Attribute(fGPR, Projects_Package
						.No_Scenario(), Projects_Package.Build(new AdaString(
						packageName), new AdaString(attributeName)),
						new AdaString(indexName));

				return;
			}

			String_List valuesList = new String_List (1, values.length);

			for (int i = 0; i < values.length; ++i) {
				AdaString name = new AdaString (values [i]);
				((IProxy) name).setOwner(Owner.NATIVE);
				valuesList.Set_Element_At(i + 1, name);
			}

			Editor_Package.Update_Attribute_Value_In_Scenario(fGPR,
					Projects_Package.No_Scenario(), Projects_Package.Build(
							new AdaString(packageName), new AdaString(
									attributeName)), valuesList, new AdaString(
							indexName), false);

			// TODO: Free valuesList !
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Save the current project in the file given in parameter. If the file
	 * doesn't exist, it will be created.
	 *
	 * @param file
	 */
	public void save (IFile file) throws GNATbenchCoreException {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		//  Using directly Save_Project from Project_Type doesn't seem to
		//  work since - for unclear reason - Eclipse 3.1 doesn't behave
		//  very well when file are created trough JNI calls. That's why
		//  we have to do the file management manually.
		final StringBuffer buf = new StringBuffer ();

		try {
			checkTimestamp();

			Registry_Package.Pretty_Print(fGPR, 3, false, new Write_Char_Ap() {

				public void Write_Char_Ap_Body(char C) {
					buf.append(C);
				}
			}, new Write_Eol_Ap() {

				public void Write_Eol_Ap_Body() {
					buf.append("\n");

				}
			}, new Write_Str_Ap() {

				public void Write_Str_Ap_Body(AdaString S) {
					buf.append(S.toString());
				}
			});
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}

		try {
			InputStream input = new ByteArrayInputStream(buf.toString()
					.getBytes());
			if (file.exists()) {
				file.setContents(input, IFile.FORCE, null);
			} else {
				file.create(input, true, null);
			}
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError("", e);
		}

	}

	public GNATbenchProjectProperties getProperties () {
		if (fProperties == null) {
			fProperties = GNATbenchProjectProperties
					.getPropertiesFor(getEclipseProject());
		}

		return fProperties;
	}

	public GNATProjectRegistry getRegistry () {
		return fRegistry;
	}

	/**
	 * Return a qualified name based on the root name of this project hierarchy
	 * and the name given in parameter.
	 */
	public QualifiedName getQualifiedName(String name)
			throws GNATbenchCoreException {
		return new QualifiedName(
				"com.adacore.gnatbench.core.internal.scenarioVariable", this
						.getRegistry().getRootProject().getName()
						+ "/" + name);
	}

	public int compareTo(GPRProject o) {
		if (o == null) {
			return 1;
		} else {
			return getProjectFile().toURI().compareTo(
					o.getProjectFile().toURI());
		}
	}
}
