/*****************************************************************************
 * Copyright (c) 2007, 2008 AdaCore.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *****************************************************************************/

package com.adacore.gnatbench.core.internal.gpswrappers;

import org.eclipse.core.resources.IProject;

import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstruct;
import com.adacore.gnatbench.core.internal.analyzer.AdaEntityReference;
import com.adacore.gnatbench.core.internal.analyzer.Column;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedFile;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedLocation;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.Entities.Entity_Information_Record;
import com.adacore.gnatbench.library.Entities.Entity_Reference;
import com.adacore.gnatbench.library.Entities.File_Location;
import com.adacore.gnatbench.library.Entities.Queries.Queries_Package;
import com.adacore.gnatbench.library.GNATCOLL.Filesystem.Filesystem_String;
import com.adacore.gnatbench.library.GNATCOLL.VFS.VFS_Package;
import com.adacore.gnatbench.library.GNATCOLL.VFS.Virtual_File;
import com.adacore.gnatbench.library.Standard.AdaString;
import com.adacore.ajis.IntegerRef;

public class Entity {
	private GeneralizedLocation fSpec = null;
	private GeneralizedLocation fBody = null;
	private IProject fProject = null;
	private boolean fIsValid = false;
	public String fName;

	public Entity_Information_Record fEntityInformation;

	public Entity(IProject project, String fileName, String name, int line,
			Column column) {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			Virtual_File file =
				VFS_Package.Create_From_Base(new Filesystem_String (fileName));

			initialize(project, file, name, line, column);
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	public Entity(AdaEntityReference reference) {
		this(reference.getLocation().getFile().getProject(), reference
				.getLocation().getFile().getOSPath(), reference.getName(),
				reference.getLocation().getLine(), reference.getLocation()
						.getColumn());
	}

	public Entity(IProject project, AdaConstruct construct) {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			Virtual_File file = VFS_Package.Create(new Filesystem_String(
					construct.getLocation().getFile().getOSPath()));

			initialize(project, file, construct.getName(), construct
					.getLineEntity(), construct.getLocation().getColumn());
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	private void initialize
	   (IProject project,
		Virtual_File file,
		String name,
		int line,
		Column column) {

		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			fProject = project;

			Entity_Information_Record.Ref entityRef =
				new Entity_Information_Record.Ref ();
			Entity_Reference closestRef = new Entity_Reference ();
			IntegerRef statusRef = new IntegerRef (0);

			GNATProjectRegistry reg = (GNATProjectRegistry) GNATbenchSession
					.getDefault().getOrLoadRegistry(project);

			Queries_Package.Find_Declaration
			(reg.getEntitiesDatabase(),
					file,
					new AdaString (name),
					line,
					column.getColumnIndex(),
					entityRef,
					closestRef,
					statusRef,
					false);

			if (entityRef.ref == null) {
				fIsValid = false;
				return;
			}

			fIsValid = true;

			fEntityInformation = entityRef.ref;

			fName = name;

			File_Location specLoc = fEntityInformation.Get_Declaration_Of();
			File_Location bodyLoc = new File_Location ();

			Queries_Package.Find_Next_Body
			(fEntityInformation,
					specLoc,
					bodyLoc,
					false);

			fSpec = GeneralizedLocation
					.fromLineColumn(GeneralizedFile.fromOSPath(this.fProject,
							specLoc.File().Get_Filename().Full_Name(false)
									.toString()), specLoc.Line(), new Column(
							specLoc.Column(), Column.DEFAULT_GNAT_TAB_WIDTH));

			if (bodyLoc.Column() != 0) {
				fBody = GeneralizedLocation.fromLineColumn(GeneralizedFile
						.fromOSPath(this.fProject, bodyLoc.File()
								.Get_Filename().Full_Name(false).toString()),
						bodyLoc.Line(), new Column(bodyLoc.Column(),
								Column.DEFAULT_GNAT_TAB_WIDTH));
			}

			if (fSpec != null) {
				fSpec.setLength(fName.length());
			}

			if (fBody != null) {
				fBody.setLength(fName.length());
			}
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 *
	 * @return the spec location of the current entity, null if none.
	 * @throws GPSError
	 */
	public GeneralizedLocation getSpec() {
		return fSpec;
	}

	/**
	 *
	 * @return the body location of the current entity, null if none.
	 * @throws GPSError
	 */
	public GeneralizedLocation getBody() {
		return fBody;
	}

	/**
	 * If body is not null, then returns the body of the entity, otherwise
	 * returns the spec.
	 *
	 * @return
	 * @throws GPSError
	 */
	public GeneralizedLocation getBodyOrElseSpec() {
		if (fBody != null) return fBody;
		else return fSpec;
	}

	/**
	 * If spec is not null, then returns the spec of the entity, otherwise
	 * returns the body.
	 *
	 * @return
	 * @throws GPSError
	 */
	public GeneralizedLocation getSpecOrElseBody() {
		if (fSpec != null) return fSpec;
		else return fBody;
	}

	public String getName () {
		return fName;
	}

	public boolean isValid () {
		return fIsValid;
	}
}
