/*****************************************************************************
 * Copyright (c) 2008 AdaCore.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *****************************************************************************/

package com.adacore.gnatbench.core.internal.filesystem;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.TreeMap;

import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.QualifiedName;

import com.adacore.gnatbench.core.GNATbenchCoreException;

public class FileStoreRegistry {

	public TreeMap<URI, IFileStore> fProjects = new TreeMap<URI, IFileStore>();
	public TreeMap<URI, IFileStore> fResources = new TreeMap<URI, IFileStore>();
	private TreeMap<URI, TreeMap<QualifiedName, Object>> fSessionProperties = new TreeMap<URI, TreeMap<QualifiedName, Object>>();

	public IFileStore getUniqueStore(URI location)
			throws GNATbenchCoreException {
		
		//  TODO: How to get the exact name, not the lowercased one?
		//  Equivalent of canonical name...
		
		if (fResources.containsKey(location)) {
			return fResources.get(location);
		} else {
			IFileStore store = EFS.getLocalFileSystem().getStore(location);
			
			if (!store.fetchInfo().exists()) {
				throw new GNATbenchCoreException("The file " + location.getPath()
						+ " doesn't exist");
			} else {
				File file;
				IPath canonicalPath;

				try {
					//  TODO: There seems to be implementation problems with
					//  EFS.CACHE here...
					file = store.toLocalFile(EFS.NONE,
							new NullProgressMonitor());

					canonicalPath = new Path (file.getCanonicalPath());
				} catch (CoreException e) {
					throw new GNATbenchCoreException (e.getMessage());
				} catch (IOException e) {
					throw new GNATbenchCoreException (e.getMessage());
				}

				URI canonicalURI = URIUtil.toURI(canonicalPath);

				if (fResources.containsKey(canonicalURI)) {
					return fResources.get(canonicalURI);
				}

				store = EFS.getLocalFileSystem().getStore(canonicalURI);

				//  We have potential multiple URIs pointing to the same
				//  store

				fResources.put(location, store);
			}
			
			return store;
		}
	}
	
	public IFileStore getUniqueStore(IPath location)
			throws GNATbenchCoreException {

		// TODO: How to get the exact name, not the lowercased one?
		// Equivalent of canonical name...
		
		if (location.isAbsolute()) {
			return getUniqueStore (URIUtil.toURI(location));
		} else {
			IFileStore tmpLoc = EFS.getLocalFileSystem().getStore(location);

			if (fResources.containsKey(tmpLoc.toURI())) {
				return fResources.get(tmpLoc.toURI());
			} else {
				return getUniqueStore(tmpLoc.toURI());
			}
		}
	}
	
	/** 
	 * Return true if the path point to an actual file or folder, false
	 * otherwise.
	 */
	public boolean exists (IPath location) {
		return EFS.getLocalFileSystem().getStore(location).fetchInfo().exists();
	}
	
	public IFileStore [] getUniqueChildren(IFileStore store)
			throws GNATbenchCoreException, CoreException {
		
		IFileStore [] children = store.childStores (EFS.NONE, null);
	
		for (int j = 0; j < children.length; ++j) {
			children [j] = getUniqueStore(children [j].toURI());
		}
		
		return children;
	}
	
	
	
	public TreeMap<QualifiedName, Object> getPropertiesFor (URI location) {
		if (!fSessionProperties.containsKey(location)) {
			fSessionProperties.put(location, new TreeMap<QualifiedName, Object>(
					new NameComparator()));
		}
		
		return fSessionProperties.get(location);
	}
	
	public static IResource [] getResources (IFileStore store) {
		IResource [] res;
		
		if (store.fetchInfo().isDirectory()) {
			res = ResourcesPlugin.getWorkspace().getRoot().findContainersForLocationURI(store.toURI());
		} else {
			res = ResourcesPlugin.getWorkspace().getRoot().findFilesForLocationURI(store.toURI());
		}
		
		return res;
	}
	
	public static IResource getDefaultResource (IFileStore store) {
		IResource [] res = getResources (store);
		
		if (res.length > 0) {
			return res [0];
		} else {
			return null;
		}
	}
	
	public static IProject getDefaultProject (IFileStore store) {
		IResource res = getDefaultResource(store);
		
		if (res != null) {
			return res.getProject();
		} else {
			return null;
		}
	}
	
	public Object getSessionProperty(IFileStore store, QualifiedName name) {
		TreeMap<QualifiedName, Object> props = getPropertiesFor(store.toURI());
		
		if (props.containsKey(name)) {
			return props.get(name);
		} else {
			return null;
		}
	}

	public void setSessionProperty(IFileStore store, QualifiedName name, Object value) {
		TreeMap<QualifiedName, Object> props = getPropertiesFor(store.toURI());
		
		if (props.containsKey(name)) {
			props.remove(name);
		}
		
		props.put(name, value);
	}
}
