/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal.codingstyle;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.TextSelection;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.adaeditor.AdaDocumentBuffer;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.Case_Handling.Case_Handling_Package;
import com.adacore.gnatbench.library.Case_Handling.Casing_Policy;
import com.adacore.gnatbench.library.Language.Indent_Parameters;
import com.adacore.gnatbench.library.Language.Replace_Text_Callback;
import com.adacore.gnatbench.library.Language.Ada.Ada_Package;
import com.adacore.gnatbench.library.Standard.AdaString;

public class AdaIndentBuffer {
	private AdaDocumentBuffer fBuffer;
	private int fFirstOffset;

	private int fSelectionBegin;
	private int fSelectionEnd;

	private StringBuffer fSelected;

	public AdaIndentBuffer (AdaDocumentBuffer buffer) {
		fBuffer = buffer;
	}

	class Text_Callback extends Replace_Text_Callback {

		private int fLastLine = -1;

		/**
		 * The cumulative delta offset computed for the previous lines,
		 * excluding the current one.
		 */
		private int fCumulativeDeltaOffset = 0;

		/**
		 * The delta offset computed for the current line.
		 */
		private int fCurrentDeltaOffset = 0;

		public void Replace_Text_Callback_Body(int Line, int First, int Last, AdaString Replace) {
			if (Line != fLastLine) {
				fCumulativeDeltaOffset += fCurrentDeltaOffset;
				fCurrentDeltaOffset = 0;
				fLastLine = Line;
			}

			final IDocument document = fBuffer.getDocument();

			try {
				final int lineOffset = document.getLineOffset(Line - 1)
						+ fCumulativeDeltaOffset;
				String replaceStr = Replace.toString();

				fCurrentDeltaOffset += replaceStr.length()
						- (Last - First);

				if (Last - First >= 0) {
					fSelected.replace(lineOffset + First - 1 - fFirstOffset,
							lineOffset + Last - 1 - fFirstOffset, replaceStr);
				}

				if (lineOffset + First - 1 < fSelectionBegin) {
					fSelectionBegin += replaceStr.length() - (Last - First);
				}

				fSelectionEnd += replaceStr.length() - (Last - First);
			} catch (BadLocationException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		}
	}

	/**
	 * Perform buffer format - does only indentation if indent only is true.
	 *
	 * @param selection
	 * @param indentOnly
	 * @return
	 * @throws BadLocationException
	 */
	public TextSelection format(TextSelection selection, boolean indentOnly)
			throws BadLocationException {
		IRegion lineStartRegion = fBuffer.getDocument().getLineInformation(
				selection.getStartLine());

		IRegion lineEndRegion = fBuffer.getDocument().getLineInformation(
				selection.getEndLine());


		int offsetLineBegin = lineStartRegion.getOffset();

		int offsetLineEnd = lineEndRegion.getOffset() + lineEndRegion.getLength();

		fSelected = new StringBuffer(fBuffer.getJavaString().substring(
				offsetLineBegin, offsetLineEnd));

		fFirstOffset = offsetLineBegin;

		fSelectionBegin = selection.getOffset();
		fSelectionEnd = selection.getOffset() + selection.getLength();

		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try{
			Indent_Parameters formatParams = GNATbenchCorePlugin.getDefault()
					.getIndentParameters();

			if (indentOnly) {
				Indent_Parameters indentParams = new Indent_Parameters ();

				indentParams.Indent_Case_Extra(formatParams.Indent_Case_Extra());
				indentParams.Indent_Comments(formatParams.Indent_Comments());
				indentParams.Indent_Conditional(formatParams.Indent_Conditional());
				indentParams.Indent_Continue(formatParams.Indent_Continue());
				indentParams.Indent_Decl(formatParams.Indent_Decl());
				indentParams.Indent_Level(formatParams.Indent_Level());
				indentParams.Indent_Record(formatParams.Indent_Record());
				indentParams.Use_Tabs(formatParams.Use_Tabs());
				indentParams.Tab_Width(formatParams.Tab_Width());

				indentParams.Align_Decl_On_Colon(false);
				indentParams.Align_On_Arrows(false);
				indentParams.Align_On_Colons(false);
				indentParams.Format_Operators(false);
				indentParams.Casing_Policy(Casing_Policy.Disabled);
				indentParams.Stick_Comments(false);

				Ada_Package.Ada_Lang ().Format_Buffer (
						fBuffer.getAdaString(),
						new Text_Callback (),
						selection.getStartLine() + 1,
						selection.getEndLine() + 1,
						indentParams,
						0,
						Case_Handling_Package.No_Casing_Exception(),
						null);
			} else {
				Ada_Package.Ada_Lang ().Format_Buffer (
						fBuffer.getAdaString(),
						new Text_Callback (),
						selection.getStartLine() + 1,
						selection.getEndLine() + 1,
						formatParams,
						0,
						Case_Handling_Package.No_Casing_Exception(),
						null);
			}
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}

		// In order to be efficient on large files, we must perform only
		// one replace operation.
		String result = fSelected.toString();

		fBuffer.getDocument().replace(offsetLineBegin,
				offsetLineEnd - offsetLineBegin, result);

		return new TextSelection(fBuffer.getDocument(), fSelectionBegin,
				fSelectionEnd - fSelectionBegin);
	}
}
