/*******************************************************************************
 * Copyright (C) 2006-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal.builder;

import java.util.LinkedList;
import java.util.Map;
import java.util.TreeMap;

import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.jobs.IJobChangeEvent;
import org.eclipse.core.runtime.jobs.JobChangeAdapter;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.PlatformUI;

import com.adacore.gnatbench.core.GNATbenchCoreException;
import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.GNATbenchProjectProperties;
import com.adacore.gnatbench.core.internal.filesystem.ResourceComparator;
import com.adacore.gnatbench.core.internal.jobs.AnalyzeAdaFilesJob;
import com.adacore.gnatbench.core.internal.make.MakefileTarget;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.core.internal.toolchain.ToolChain;
import com.adacore.gnatbench.core.internal.toolchain.ToolChain.NoExtensionFound;

public class GNATbenchIncrementalBuilder extends IncrementalProjectBuilder {

	public static final String AUTO_BUILD_TARGET = "autobuild";
	public static final String ANALYZE_TARGET    = "analyze";
	public static final String FULL_BUILD_TARGET = "build";
	public static final String CLEAN_TARGET      = "clean";
	public static final String CLEAN_TREE_TARGET = "clean_tree";
	public static final String REBUILD_TARGET    = "rebuild";

	// the name of the macro used to define the path to the gpr file in makefiles
	public static final String GPR_MACRO_NAME = "GPRPATH";
	// the name of the macro used to define the compiler command in makefiles
	public static final String COMPILE_MACRO_NAME = "GNATMAKE";
	// the name of the macro used to define the clean command in makefiles
	public static final String CLEAN_MACRO_NAME = "GNATCLEAN";

	public final static String BUILDER_ID = GNATbenchCorePlugin.getId()	+ ".GNATbenchBuilder";

	static private TreeMap<IProject, GNATbenchIncrementalBuilder> fgBuilders =
		new TreeMap<IProject, GNATbenchIncrementalBuilder>(new ResourceComparator ());
	private boolean fForcedBuild = false;

	public GNATbenchIncrementalBuilder() {
		super();
	} // ctor

	/**
	 * Returns the list of Ada files concerned by this change.
	 * @return
	 */
	public LinkedList <IFile> affectedAdaSources() {
		final LinkedList <IFile> result = new LinkedList <IFile> ();

		try {
			IResourceDelta delta = getDelta(getProject ());

			if (delta == null) {
				return result;
			}

			delta.accept(new IResourceDeltaVisitor () {

				public boolean visit(IResourceDelta delta) throws CoreException {
					try {
						if (GNATProjectRegistry.isManagedSourceFile(delta.getResource())) {
							result.add((IFile) delta.getResource());
						}
					} catch (GNATbenchCoreException e) {
						GNATbenchCorePlugin.getDefault().logError(null, e);
					}
					return true;
				}});
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		return result;
	}

	/**
	 * @see IncrementalProjectBuilder#build(int, java.util.Map, org.eclipse.core.runtime.IProgressMonitor)
	 */
	@SuppressWarnings("unchecked")
	protected IProject[] build(int kind, Map args, IProgressMonitor monitor) throws CoreException {
		boolean localForcedBuild = fForcedBuild;
		fForcedBuild = false;

		if (fgBuilders.containsKey(getProject ())) {
			fgBuilders.remove(getProject ());
		}

		fgBuilders.put(getProject (), this);

		final boolean autoSavePrefEnabled = GNATbenchCorePlugin
			.getDefault()
			.getPreferenceStore()
			.getBoolean(
				AdaBuilderPreferencesProvider.ADA_SAVE_DIRTY_ON_BUILD);

		GNATProjectRegistry registry = (GNATProjectRegistry) GNATbenchSession
				.getDefault().getOrLoadRegistry(getProject());

		//  If we're on an invalid kernel, then we won't do any build
		if (!registry.isValid()) {
			return null;
		}

		LinkedList <IFile> files = affectedAdaSources();

		if ((kind == AUTO_BUILD || kind == INCREMENTAL_BUILD)
				&& files.size() == 0) {
			// only do a build in response to changes to *Ada source files*

			return null;
		}

		if (ResourcesPlugin.getWorkspace().isAutoBuilding() && !localForcedBuild) {
			//  In auto build mode, there are several cases where we don't want
			//  to do anything

			if (kind == FULL_BUILD) {
				// We auto-build fully only root projects

				if (!GNATbenchProjectProperties.getPropertiesFor(getProject())
						.isRootProject()) {
					return null;
				}
			} else if (files.size() == 0) {
				// We don't auto-build anything if there is no Ada file involved

				return null;
			}
		} else {
			//  if we query a manual build, then gnatmake is responsible of
			//  knowing what and if there is build or link to be done in that
			//  case, so cancel Eclipse heuristics.

			forgetLastBuiltState();

			// We save the open editors only when doing a manual build
			Display.getDefault().syncExec(new Runnable() {
				public void run() {
					PlatformUI.getWorkbench().saveAllEditors(
							!autoSavePrefEnabled);
				}
			});
		}

		if (kind == AUTO_BUILD) {
			LinkedList <IFileStore> stores = new LinkedList <IFileStore> ();

			for (IFile file : files) {
				try {
					stores.add(GNATbenchCorePlugin.getDefault().getEFSRegistry()
							.getUniqueStore(file.getLocation()));
				} catch (GNATbenchCoreException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				}
			}

			AnalyzeAdaFilesJob job = new AnalyzeAdaFilesJob(stores
					.toArray(new IFileStore[stores.size()]), false);

			job.setRaiseViews(false);

			//  Since we are already in the protection of the workspace, we
			//  just run the job here instead of creating a sub job (which
			//  is likely to raise deadlocks anyway).

			job.runInWorkspace(monitor);
		} else {
			invokeMakefile(FULL_BUILD_TARGET);
		}

		return null;
	} // build

	public static String [] macroParamsFromGPR(final IProject project) {
		GNATbenchProjectProperties props = GNATbenchProjectProperties
			.getPropertiesFor(project);

		final String gprFileLocation = props.getGPROSPath();

		final String [] buildCmd;
		final String builderName;
		try {
			buildCmd = ToolChain.active().buildCommand(project);
			builderName = ToolChain.active().name(project);
			// eg: i586-wrs-vxworks-gnatmake, or just gnatmake, or ...
		} catch (NoExtensionFound e) {
			GNATbenchCorePlugin.getDefault().logError(
					"No toolchain definition was found", null);
			return null;
		} // try

		final String cleanCommand;
		final int makePos = builderName.lastIndexOf("make");
		// all gnat compilers have names like gnatmake, powerpc-elf-gnatmake,
		// gnaaampmake, etc.
		if (makePos == -1) {
			// not a gnat compiler, so we punt.  the builder name won't be
			// right either, but that will appear in the command for the user to see.
			cleanCommand = builderName + "clean";
		} else {
			cleanCommand = builderName.subSequence(0,makePos) + "clean";
		} // if

		// Flatten build command: this is OK since the final expansion will be done
		// by make.
		String flatBuildCmd = buildCmd[0];
		for (int j=1; j<buildCmd.length; j++)
			flatBuildCmd = flatBuildCmd + " " + buildCmd[j];

		final String [] params = {
				COMPILE_MACRO_NAME + "=" + flatBuildCmd +"",
				CLEAN_MACRO_NAME + "=" + cleanCommand +"",
				GPR_MACRO_NAME + "=" + gprFileLocation +""
		};

		return params;
	} // macroParamsFromGPR


	protected void invokeMakefile(final String target) {
		final IProject project = getProject();
		new MakefileTarget(project, target, null).invoke(macroParamsFromGPR(project));
	} // invokeMakefile


	protected void clean(IProgressMonitor monitor) throws CoreException {
		super.clean(monitor);
		try {
			final IMarker[] markers = getProject()
				.findMarkers(GNATbenchCorePlugin.ADA_ERROR_MARKER, true, IProject.DEPTH_ZERO);

			for (int i = 0; i < markers.length; i++) {
				markers[i].delete();
			} // for
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} // try
		invokeMakefile(CLEAN_TARGET);
	} // clean


	static public class JobCompleteAdapter extends JobChangeAdapter  {

		IProject project;

		public JobCompleteAdapter(IProject project) {
			this.project = project;
		} // ctor

		public void done(IJobChangeEvent event) {
			if (event.getResult().isOK()) {
				try {
					project.refreshLocal(IResource.DEPTH_INFINITE, null);
				} catch (CoreException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
					return;
				} // try
			} // if
		} // done

	} // JobCompleteAdapter


	// this is predicated on a GNAT compiler, both for the tool names and their args
	public static String makefileForGPR() {
		final String builderName = "$(" + COMPILE_MACRO_NAME  + ")";

		final String projectFileName = "\"$(" + GPR_MACRO_NAME  + ")\"";

		final String checksOnly = " -gnatc -c -k ";

		final String buildCommand = builderName + " -P " + projectFileName;

		final String autoBuildCommand = builderName + checksOnly + " -P " + projectFileName;
		//  checks those files that have changed, and only those, because it is intended
		//  to be invoked by the Workbench automatic build option (if the user enables it)

		final String analyzeCommand = builderName + " -f " + checksOnly + " -P " + projectFileName;
		//  NB: analyze command adds -f to compileCommand because it should analyze all of them
		//  and is intended to be invoked manually by the user

		final String cleanRootCommand = "$(" + CLEAN_MACRO_NAME  + ")" + " -P " + projectFileName;

		final String cleanTreeCommand = cleanRootCommand + " -r";

		final String makeFile =
			"# You may edit this makefile as long as you keep these original \n" +
			"# target names defined.\n" +
			"\n" +
			makefileFromStrings (
					buildCommand,
					cleanRootCommand,
					cleanTreeCommand,
					analyzeCommand,
					autoBuildCommand);

		return makeFile;
	} // makefileForGPR


	public static String makefileFromStrings(
	    	final String buildCommand,
	       	final String cleanRootCommand,
	       	final String cleanTreeCommand,
	       	final String analyzeCommand,
	       	final String autoBuildCommand)
	{
		final String makeFile =
			"# Not intended for manual invocation.\n" +
			"# Invoked if automatic builds are enabled.\n" +
			"# Analyzes only on those sources that have changed.\n" +
			"# Does not build executables.\n" +
			GNATbenchIncrementalBuilder.AUTO_BUILD_TARGET + ":\n" +
			"\t" + autoBuildCommand + "\n\n" +
			"# Clean the root project of all build products.\n" +
			GNATbenchIncrementalBuilder.CLEAN_TARGET + ":\n" +
			"\t" + cleanRootCommand + "\n\n" +
			"# Clean root project and all imported projects too.\n" +
			GNATbenchIncrementalBuilder.CLEAN_TREE_TARGET + ":\n" +
			"\t" + cleanTreeCommand + "\n\n" +
			"# Check *all* sources for errors, even those not changed.\n" +
			"# Does not build executables.\n" +
			GNATbenchIncrementalBuilder.ANALYZE_TARGET + ":\n" +
			"\t" + analyzeCommand + "\n\n" +
			"# Build executables for all mains defined by the project.\n" +
			GNATbenchIncrementalBuilder.FULL_BUILD_TARGET + ":\n" +
			"\t" + buildCommand + "\n\n" +
			"# Clean, then build executables for all mains defined by the project.\n" +
			GNATbenchIncrementalBuilder.REBUILD_TARGET + ": " +
				GNATbenchIncrementalBuilder.CLEAN_TARGET + " " +
				GNATbenchIncrementalBuilder.FULL_BUILD_TARGET +  "\n";

		return makeFile;
	} // makefileFromStrings

	static public GNATbenchIncrementalBuilder getBuilder (IProject project) {
		return fgBuilders.get(project);
	}

	/**
	 * Forces the next build command to actually build, no matter how is the
	 * internal build state.
	 */
	public void forceNextBuild () {
		forgetLastBuiltState();
		fForcedBuild  = true;
	}

} // GNATbenchIncrementalBuilder
