/*****************************************************************************
 * Copyright (C) 2006-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *****************************************************************************/

package com.adacore.gnatbench.core.internal.builder;

import java.io.File;
import java.io.IOException;
import java.text.DateFormat;
import java.util.LinkedList;
import java.util.Map;
import java.util.TreeMap;

import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.resources.WorkspaceJob;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.InvalidRegistryObjectException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.osgi.framework.Bundle;

import com.adacore.gnatbench.core.GNATbenchCoreException;
import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.gpswrappers.GPRProject;
import com.adacore.gnatbench.core.internal.jobs.ToolMessage;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.core.internal.utils.ViewUtils;
import com.adacore.gnatbench.core.projects.IScenarioVariable;

public class GNATbenchCommandJob extends WorkspaceJob {

	/**
	 * The command and all it's argument
	 * in an array of String
	 */
	private String[] fCommand;

	/**
	 * The project associated with the selected
	 * ressource
	 */
	private IProject fProject;

	/**
	 *  The console to display the results
	 */
	private BuildConsole fConsole;

	/**
	 *  A name for the command which will be display
	 *  in the console
	 */
	private String commandsName;

	/**
	 * The message handler for the command
	 */
	private Class <? extends ToolMessage> fMessageClass;

	/**
	 * the working directory for the command
	 * @default the project directory
	 */
	private File fWorkingDir;

	private final boolean fRaiseViews;

	public GNATbenchCommandJob(final String[] command, final IProject project,
			final String commandsName, final Class<? extends ToolMessage> messageClass,
			boolean raiseViews) {
		super(commandsName + " [" + project.getName() + "]");
		init(command, project, messageClass, commandsName);

		// we initialize the workingDirectory to the project directory
		fWorkingDir = fProject.getLocation().toFile();

		fRaiseViews = raiseViews;
	} // ctor

	public GNATbenchCommandJob(final String[] command, final IProject project,
			final String commandsName, final Class<? extends ToolMessage> messageClass,
			String dir, boolean raiseViews) {
		super(commandsName + " [" + project.getName() + "]");

		fRaiseViews = raiseViews;
		init(command, project, messageClass, commandsName);

		if (dir != null){
			fWorkingDir = new File(dir);
		} else {
			fWorkingDir = fProject.getLocation().toFile();
		}
	} // ctor

	/**
	 * Initialize the variables, to be called by the constructors
	 * @param command
	 * @param project
	 * @param commandsName
	 * @param markerType
	 */
	private void init(final String[] command, final IProject project,
			final Class<? extends ToolMessage> messageClass,
			final String commandsName) {
		this.commandsName = commandsName;
		fProject = project;
		fCommand = command;
		fMessageClass = messageClass;

		fConsole = BuildConsoleManager.findConsole(commandsName + " [" + project.getName() + "]");
		// we don't use the command in the console name because that
		// would create too many consoles.

		if (fRaiseViews) {
			ViewUtils.showConsoleView();
		}
	}

	@Override
	final public IStatus runInWorkspace(IProgressMonitor monitor) {
		final String [] env;

		monitor.beginTask(fCommand[0], 100);

		try {
			// Load environment vars (to get any scenario vars but also the
			// path, etc) and also load those scenario vars set by the user
			// within Workbench.
			env = environment(fProject, monitor);
		} catch (GNATbenchCoreException e) {
			return new Status(IStatus.ERROR, GNATbenchCorePlugin.getId(),
					IStatus.OK,
					"Could not get environment",
					e);
		} // try

		fConsole.clearConsole();
		StringBuffer command = new StringBuffer();
		for (int i = 0; i < fCommand.length; i++) {
			command.append(fCommand[i] + " ");
		}

		// display the command in the console
		fConsole.print(command.toString() + "\n");

		final Process proc;
		try {
			proc = Runtime.getRuntime().exec(fCommand, env, fWorkingDir);
		} catch (IOException e) {
			return new Status(IStatus.ERROR, GNATbenchCorePlugin.getId(),
					IStatus.OK,
					"Could not execute fCommand '" + command.toString() + "'",
					e);
		} // try

		ConsolePrinter errorMessages = new ConsolePrinter(
				proc.getErrorStream(), monitor, fConsole);
		ConsolePrinter normalMessages = new ConsolePrinter(proc
				.getInputStream(), monitor, fConsole);

		errorMessages.start();
		normalMessages.start();

		try {
			proc.waitFor();
		} catch (InterruptedException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return new Status(IStatus.ERROR, GNATbenchCorePlugin.getId(),
					IStatus.OK,
					"Could not await process completion",
					null);
		} // try

		printCompletionMsg();

		// we need to refresh the object directories since there may be new files
		// there that are referenced in the error/warning messages (eg binder-
		// generated Ada files).  this must be done before we signal the completion
		// of the compilation since that is when the error messages are processed.
		refreshObjectDirs(fProject, monitor);

		String [] errorMessagesStr = errorMessages.getMessages();
		String [] normalMessagesStr = normalMessages.getMessages();

		String[] messages = new String[errorMessagesStr.length
				+ normalMessagesStr.length];

		for (int j = 0; j < errorMessagesStr.length; ++j) {
			messages [j] = errorMessagesStr [j];
		}

		for (int j = errorMessagesStr.length; j < messages.length; ++j) {
			messages [j] = normalMessagesStr [j - errorMessagesStr.length];
		}

		signalCompilationCompleted(fProject, messages, monitor);

		monitor.done();

		return new Status(IStatus.OK, GNATbenchCorePlugin.getId(), IStatus.OK, "", null);
	} // runInWorkspace


	/**
	 * Returns the environment variables defined by the user process and
	 * those that are scenario variables defined by the fProject.  Any
	 * scenario vars with the same names as environment variables overwrite
	 * those environment variables so that they take precedence.
	 */
	protected String[] environment(IProject project, IProgressMonitor monitor)
			throws GNATbenchCoreException {

		String [] result = null;
		Map <String, String> allVars = new TreeMap <String, String> ();

		// put the process's environment variables into the map.
		// allVars will not be null on return but may be empty.
		try {
			allVars.putAll(System.getenv());
		} catch (Throwable e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return null;
		} // try

		// now put the fProject's scenario variables in, if any are defined,
		// overwriting any values for corresponding environment variables
		GPRProject gprProject = ((GNATProjectRegistry) GNATbenchSession
				.getDefault().getOrLoadRegistry(project)).getGPRFor(project);
		gprProject.loadScenarioVariables();
		LinkedList <IScenarioVariable> scenarioVars = gprProject.getScenarioVariables();

		for (IScenarioVariable var : scenarioVars) {
			allVars.put(var.getExternalName(), var.getValue());
		} // for

		// put all into the result, each in the form key=value
		int mapsize = allVars.size();
		if (mapsize != 0) {
			result = new String[mapsize];
		} // if

		int k = 0;
		for (Map.Entry <String, String> entry : allVars.entrySet()) {
			String key = entry.getKey();
			String value = entry.getValue();
			result[k] = key + "=" + value;
			k = k + 1;
		} // for

		return result;
	} // environment

	/**
	 * Notifies listeners to buildListener extension point that the build is
	 * finished.
	 *
	 * @param compiledFiles
	 * @param errors
	 * @param monitor
	 */
	protected void signalCompilationCompleted
	   (final IProject project, final String[] errors, final IProgressMonitor monitor)
	{
		if (errors == null) {
			GNATbenchCorePlugin.getDefault()
				.logError("Null errors passed to signalCompilationCompleted", null);
			return;
		} // if

		IExtensionRegistry er = Platform.getExtensionRegistry();
		IExtensionPoint ep = er.getExtensionPoint(GNATbenchCorePlugin.getId(),
				"buildListener");

		IExtension[] extensions = ep.getExtensions();

		if (extensions.length == 0) {
			return;
		} // if

		monitor.setTaskName("Analyzing builder messages");

		CompilationEvent event = new CompilationEvent ();

		event.messages = errors;
		event.project = project;
		event.rootMessageClass = fMessageClass;
		event.raiseViews = fRaiseViews;

		for (int extInd = 0; extInd < extensions.length; ++extInd) {

			IConfigurationElement[] elements = extensions[extInd].getConfigurationElements();

			for (int i = 0; i < elements.length; ++i) {
				if (elements[i].getName().equals("listener")) {
					try {
						Bundle bundle = Platform.getBundle(extensions[extInd]
								.getNamespaceIdentifier());

						final IGNATbenchBuildListener builderListener =
							(IGNATbenchBuildListener) bundle
								.loadClass(elements[i].getAttribute("class"))
								.newInstance();

						builderListener.compilationCompleted(event, monitor);

					} catch (InvalidRegistryObjectException e) {
						GNATbenchCorePlugin.getDefault().logError(null, e);
					} catch (InstantiationException e) {
						GNATbenchCorePlugin.getDefault().logError(null, e);
					} catch (IllegalAccessException e) {
						GNATbenchCorePlugin.getDefault().logError(null, e);
					} catch (ClassNotFoundException e) {
						GNATbenchCorePlugin.getDefault().logError(null, e);
					} // try
				} // if
			} // for
		} // for

		for (IGNATbenchBuildListener listener : GNATbenchCorePlugin
				.getDefault().getBuildListeners()) {
			listener.compilationCompleted(event, monitor);
		}
	} // signalCompilationCompleted


	// refresh all the object directories (ie including the dependencies)
	protected void refreshObjectDirs(final IProject project,
			IProgressMonitor monitor) {
		try {
			final GNATProjectRegistry reg = (GNATProjectRegistry) GNATbenchSession
					.getDefault().getOrLoadRegistry(project);
			final LinkedList<IFileStore> objDirs = reg.getRootProject().getObjectDirectories(true);
			final IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
			for (IFileStore dir : objDirs) {
				monitor.setTaskName("Refreshing " + dir.toURI());

				final IContainer[] containers = root
						.findContainersForLocationURI(dir.toURI());

				for (IContainer cont : containers) {
					cont.refreshLocal(IResource.DEPTH_ONE, monitor);
				}
			} // for
		} catch (GNATbenchCoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} // try
	} // refreshObjectDirs


	protected void printCompletionMsg() {
		final DateFormat df = DateFormat.getDateTimeInstance(DateFormat.MEDIUM, DateFormat.FULL);
		final String now = df.format(new java.util.Date());
		fConsole.print(commandsName + " for [" + fProject.getName() + "] completed " + now + ".");
	} // printCompletionMsg


	public IProject getProject () {
		return fProject;
	} // getProject


} // GNATbenchCommandJob
