/*******************************************************************************
 * Copyright (C) 2005-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal.browsing;

import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.jface.text.BadLocationException;

import com.adacore.gnatbench.core.GNATbenchCoreException;
import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.analyzer.IAdaEntityReference;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.GNATbenchProjectProperties;
import com.adacore.gnatbench.core.internal.analyzer.AdaEntityReference;
import com.adacore.gnatbench.core.internal.analyzer.Column;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedFile;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedLocation;
import com.adacore.gnatbench.core.internal.gpswrappers.Entity;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.Entities.Entities_Package;
import com.adacore.gnatbench.library.Entities.Entity_Reference;
import com.adacore.gnatbench.library.Entities.File_Location;
import com.adacore.gnatbench.library.Entities.Queries.Entity_Reference_Iterator;
import com.adacore.ajis.BooleanRef;

/**
 * This class encapsulates a iterator over entity references found from the ALI
 * files.
 */
public class ReferencesIterator implements Iterator <IAdaEntityReference> {
	private HashSet <String> fAnalyzedFiles = new HashSet <String> ();
	private String fCurrentFile = "";
	private IProject fCurrentProject;
	private GNATProjectRegistry fCurrentRegistry;

	private LinkedList <IProject> fProjects = new LinkedList<IProject> ();
	private Iterator <IProject> fProjectIt;

	private Entity_Reference_Iterator fIter;
	private GeneralizedLocation fLocation;
	private String fName;

	private GeneralizedLocation fCurrentLoc;

	private IAdaEntityReference.SearchScopes fScope;

	public ReferencesIterator(AdaEntityReference reference,
			IAdaEntityReference.SearchScopes scope) {
		fLocation = reference.getLocation();
		fName = reference.getName();
		fScope = scope;

		if (scope == IAdaEntityReference.SearchScopes.SCOPE_WORKSPACE) {
			IProject[] projects = ResourcesPlugin.getWorkspace().getRoot()
					.getProjects();

			for (IProject prj : projects) {
				GNATProjectRegistry reg = (GNATProjectRegistry) GNATbenchSession
						.getDefault().getOrLoadRegistry(prj);

				if (reg != null
						&& GNATbenchProjectProperties.getPropertiesFor(prj)
								.isRootProject()) {
					fProjects.add(prj);
				}
			}
		} else if (scope == IAdaEntityReference.SearchScopes.SCOPE_PROJECT
				|| scope == IAdaEntityReference.SearchScopes.SCOPE_PROJECT_HIERARCHY) {
			fProjects.add(reference.getLocation().getFile().getProject());
		}

		fProjectIt = fProjects.iterator();

		if (!atEnd ()) {
			next ();
		}
	}

	private boolean isValid () {
		if (atEnd ()) {
			return true;
		}

		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			if (!(fIter != null && !fIter.At_End() && current () != null)) {
				return false;
			}

			if (fScope == IAdaEntityReference.SearchScopes.SCOPE_PROJECT) {
				IProject prj = current().getFile().getProject();

				return prj != null && prj.equals(fCurrentProject);
			} else if (fScope == IAdaEntityReference.SearchScopes.SCOPE_PROJECT_HIERARCHY) {
				IProject prj = current().getFile().getProject();

				if (prj.equals(fCurrentProject)) {
					return true;
				}

				BooleanRef result1 = new BooleanRef ();
				BooleanRef result2 = new BooleanRef ();

				fCurrentRegistry.getGPRFor(fCurrentProject).getGPRProject()
						.Project_Imports(
								fCurrentRegistry.getGPRFor(prj).getGPRProject(),
								true, result1, result2);

				return result1.getValue() || result2.getValue();
			} else {
				return true;
			}
		} catch (GNATbenchCoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);

			fProjects.clear();
			fIter.Destroy();
			fIter = null;

			return true;
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Return the current location pointed by the iterator, null if none.
	 */
	public GeneralizedLocation current () {

		if (fCurrentLoc == null) {
			LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

			try {
				File_Location loc = fIter.Get().Get_Location();

				if (loc.Get_File() != null) {
					GeneralizedLocation location = GeneralizedLocation
					.fromLineColumn(GeneralizedFile.fromVirtualFile(
							fLocation.getFile()
							.getProject(), loc.Get_File()
							.Get_Filename()), loc
							.Get_Line(), new Column(loc.Get_Column(),
									Column.DEFAULT_GNAT_TAB_WIDTH));
					location.setLength(fName.length());

					fCurrentLoc = location;
				}
			} finally {
				LibrarySemaphore.stopGPSWork(libMonitor);
			}
		}

		return fCurrentLoc;
	}

	/**
	 * Iterate over the next reference.
	 */
	public IAdaEntityReference next() {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			while (true) {
				fCurrentLoc = null;

				if (fIter == null || fIter.At_End()) {
					if (fIter != null) {
						fIter.Destroy();
					}

					fCurrentProject = fProjectIt.next();
					fCurrentRegistry = (GNATProjectRegistry) GNATbenchSession
							.getDefault().getOrLoadRegistry(fCurrentProject);

					Entity entity = new Entity(fCurrentProject, fLocation
							.getFile().getOSPath(), fName, fLocation.getLine(),
							fLocation.getColumn());

					if (entity.fEntityInformation == null) {
						//  The entity has not been found in this project.

						if (atEnd ()) {
							break;
						} else {
							continue;
						}
					}

					fIter = new Entity_Reference_Iterator ();
					fIter.Find_All_References (
							entity.fEntityInformation,
							null,
							null,
							null,
							Entities_Package.Real_References_Filter(),
							false,
							false);
				} else {
					fIter.Next();
				}

				if (atEnd ()) {
					break;
				}

				if (isValid ()) {
					//  We avoid extracting results twice from the same file,
					//  in case we analyze several kernels.

					GeneralizedLocation loc = current ();
					String currentFilePath = loc.getFile().getOSPath();

					if (currentFilePath.equals(fCurrentFile)) {
						break;
					} else if (!fAnalyzedFiles.contains(currentFilePath)) {
						fAnalyzedFiles.add(currentFilePath);
						fCurrentFile = currentFilePath;

						break;
					}
				}
			}
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}

		GeneralizedLocation result = current ();

		if (result != null) {
			try {
				return new AdaEntityReference(current());
			} catch (BadLocationException e) {

			}
		}

		return null;
	}

	/**
	 * Return true if there's no more references to return.
	 */
	public boolean atEnd () {
		if (fProjectIt.hasNext()) {
			return false;
		}

		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			return fIter == null || (fIter != null && fIter.At_End());
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Return the total number of progress to be made.
	 */
	public int totalProgress () {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			if (fIter == null) {
				return 0;
			} else {
				return fIter.Get_Total_Progress();
			}
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Return the current progress state.
	 */
	public int currentProgress () {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			return fIter.Get_Current_Progress();
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Return true if the element pointed by the iterator should be shown in
	 * the call graph.
	 *
	 * @return
	 */
	public boolean showInCallGraph() {
		LibraryMonitor monitor = LibrarySemaphore.startGPSWork();

		try {
			return Entities_Package.Show_In_Call_Graph(fIter.Get().Get_Kind());
		} finally {
			LibrarySemaphore.stopGPSWork(monitor);
		}
	}

	/**
	 * Return the Entity_Reference currently pointed by the iterator.
	 * @return
	 */
	public Entity_Reference get () {
		return fIter.Get();
	}

	/**
	 * Free the data associated to this iterator.
	 */
	public void destroy () {
		if (fIter != null) {
			LibraryMonitor monitor = LibrarySemaphore.startGPSWork();

			try {
				fIter.Destroy();
			} finally {
				LibrarySemaphore.stopGPSWork(monitor);
			}
		}
	}

	public boolean hasNext() {
		return !atEnd ();
	}

	public void remove() {
		//  This optional operation is not implemented, since there's no way
		//  to remove an entry from the actual references written in the
		//  program.
	}
}
