/*******************************************************************************
 * Copyright (C) 2005-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal.browsing;

import java.util.LinkedList;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.text.BadLocationException;

import com.adacore.gnatbench.core.analyzer.IAdaEntityReference;
import com.adacore.gnatbench.core.analyzer.IAdaEntityReference.SearchScopes;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.analyzer.AdaEntityReference;
import com.adacore.gnatbench.core.internal.analyzer.Column;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedFile;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedLocation;
import com.adacore.gnatbench.core.internal.gpswrappers.Entity;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.Entities.Entities_Package;
import com.adacore.gnatbench.library.Entities.Entity_Information_Record;
import com.adacore.gnatbench.library.Entities.File_Location;
import com.adacore.gnatbench.library.Entities.Queries.Calls_Iterator;
import com.adacore.gnatbench.library.Entities.Queries.Entity_Reference_Iterator;
import com.adacore.gnatbench.library.Entities.Queries.Queries_Package;

/**
 * Holds high-level browsing capabilities.
 */
public class AdaCodeBrowser {
	/**
	 * Returns a linked list of AdaLocation, containing the references to the
	 * given EntityReference.
	 *
	 * @param reference
	 * @param scope can be SCOPE_PROJECT or SCOPE_WORKSPACE
	 * @return
	 */
	public static ReferencesIterator getLocations(AdaEntityReference reference,
			IAdaEntityReference.SearchScopes scope, IProgressMonitor monitor) {

			return new ReferencesIterator (reference, scope);
	}

	/**
	 * Returns all the calls of a subprogram reference, or all the functions
	 * that calls this reference. The linked list returned contains a list list
	 * per occurence, each sub link list containing a set of
	 * AdaEntityReference.
	 *
	 * @param reference
	 * @param calledBy
	 * @param monitor
	 * @return
	 * @throws GPSError
	 */
	public static LinkedList <LinkedList<AdaEntityReference>>
	getCalls(AdaEntityReference reference,
			boolean calledBy, IProgressMonitor monitor) {

		LibraryMonitor libMonitor;

		LinkedList<LinkedList<AdaEntityReference>> referencesResult =
			new LinkedList<LinkedList<AdaEntityReference>>();

		Entity entity = new Entity (reference);

		if (calledBy) {
			ReferencesIterator iter = new ReferencesIterator(reference,
					SearchScopes.SCOPE_WORKSPACE);

			Entity_Information_Record lastCaller = null;
			LinkedList<AdaEntityReference> occurences =
				new LinkedList<AdaEntityReference>();

			while (true) {
				libMonitor = LibrarySemaphore.startGPSWork();

				try {
					if (iter.atEnd()) break;

					if (iter.showInCallGraph())  {
						GeneralizedLocation loc = iter.current();
						loc.setLength(reference.getName().length());

						AdaEntityReference entityRef = null;

						try {
							entityRef = GNATbenchCorePlugin
									.getDefault().getDocumentBuffer(
											loc.getFile())
									.getEntity(loc);
						} catch (BadLocationException e) {
							GNATbenchCorePlugin.getDefault().logError(null, e);
						}

						if (entityRef == null) {
							//  If the file is not up to date - or if the xrefs
							//  are out of date, we want to avoid crashing the
							//  view but instead just display a message in
							//  the log.

							GNATbenchCorePlugin.getDefault().logError(
									"Could not find entity at " + loc, null);

							iter.next();

							continue;
						}

						Entity_Information_Record caller =
							Queries_Package.Get_Caller(iter.get());


						//  ??? Why isn't the At_End test enough ?
						if (caller != null) {
							if (lastCaller == null) {
								lastCaller = caller;
							} else if (!
									(lastCaller.Get_Declaration_Of().Line()
											== caller.Get_Declaration_Of().Line()
											&& lastCaller.Get_Declaration_Of().Column ()
											== caller.Get_Declaration_Of().Column()
											&& lastCaller.Get_Declaration_Of().File().
											Get_Filename().Full_Name(true).toString().
											equals(caller.Get_Declaration_Of().File().
													Get_Filename().Full_Name(true).toString()))) {

								//  If the previous and the current enclosing
								//  declarations are different, then we switch to
								//  an other result group

								referencesResult.add(occurences);
								occurences = new LinkedList<AdaEntityReference>();
								lastCaller = caller;
							}

							occurences.add(entityRef);
						}
					}

					iter.next();
				} finally {
					LibrarySemaphore.stopGPSWork(libMonitor);
				}
			}

			if (lastCaller != null) {
				referencesResult.add(occurences);
			}

			iter.destroy();
		} else {
			// We don't need an actual workspace-wide iterator here, since
			// we're only concerned about internal calls which are by
			// definition in the hierarchy

			Calls_Iterator iter;
			libMonitor = LibrarySemaphore.startGPSWork();

			try {
				iter = Queries_Package.Get_All_Called_Entities
				(entity.fEntityInformation);
			} finally {
				LibrarySemaphore.stopGPSWork(libMonitor);
			}

			while (true) {
				libMonitor = LibrarySemaphore.startGPSWork();

				try {
					if (iter.At_End()) break;

					Entity_Information_Record info = iter.Get();

					Entity_Reference_Iterator refs =
						new Entity_Reference_Iterator ();

					refs.Find_All_References
					(info,
							null,
							null,
							entity.fEntityInformation,
							Entities_Package.Real_References_Filter(),
							false,
							false);

					LinkedList <AdaEntityReference> occurences =
						new LinkedList <AdaEntityReference> ();

					while (!refs.At_End()) {
						if (!refs.Get().equals(Entities_Package.No_Entity_Reference())
								&& Entities_Package.Show_In_Call_Graph(refs.Get()
								.Get_Kind())
								&& refs.Get_Entity().Is_Subprogram()
								&& Queries_Package.Get_Caller(refs.Get())
										.equals(entity.fEntityInformation))
						{
							File_Location loc = refs.Get().Get_Location();

							GeneralizedLocation location = GeneralizedLocation
									.fromLineColumn(
											GeneralizedFile.fromVirtualFile(
													null, loc.Get_File()
															.Get_Filename()),
											loc.Get_Line(),
											new Column(
													loc.Get_Column(),
													Column.DEFAULT_GNAT_TAB_WIDTH));

							location.setLength(reference.getName().length());

							AdaEntityReference entityRef = null;

							try {
								entityRef = GNATbenchCorePlugin
										.getDefault().getDocumentBuffer(
												location.getFile())
										.getEntity(location);
							} catch (BadLocationException e) {
								GNATbenchCorePlugin.getDefault().logError(null,
										e);
							}

							if (entityRef != null) {
								occurences.add(entityRef);
							}
						}

						refs.Next();
					}

					if (occurences.size() > 0) {
						referencesResult.add(occurences);
					}

					iter.Next();
				} finally {
					LibrarySemaphore.stopGPSWork(libMonitor);
				}
			}
		}

		return referencesResult;
	}
}
