/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal.analyzer;

import java.lang.reflect.InvocationTargetException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.Position;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.texteditor.AbstractMarkerAnnotationModel;

import com.adacore.gnatbench.core.GNATbenchCoreException;
import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;

public class GeneralizedLocation {

	/**
	 * Matches strings like something:path:line:column
	 */
	private static Pattern fgLocationPattern = Pattern
	   .compile(
	   		"(?:(?:[^:]+):)??" +
			// Removes the entity name that can be in front of the location.
			// Made reluctant in order to avoid things like C:\ which are part
			// of the path and taken into account later.
	   		"((?:[A-Za-z]:\\\\)?[^:]+)" +
			// The path, with a drive:\ in front of it when relevant
			":([0-9]+):([0-9]+)"
			// the line number and the column number
			);

	private Column fColumn = null;
	private int fOffset = -1;
	private int fLine = -1;
	private int fLength = 0;
	private GeneralizedFile fFile;

	/**
	 * It's not allowed to create directly a GeneralizedLocation - use the
	 * from functions instead.
	 */
	private GeneralizedLocation () {

	}

	/**
	 * Create a location based on a line number (starting at 1) and a file
	 * offset (starting at 0).
	 */
	public static GeneralizedLocation fromAbsoluteLocation(GeneralizedFile file,
			int line, int offset) {
		GeneralizedLocation result = new GeneralizedLocation ();

		result.fLine = line;
		result.fOffset = offset;
		result.fFile = file;

		return result;
	}

	/**
	 * Create a location based on a line number (starting at 1) and a column
	 * number (starting at 1). Tabulations are taked into account when
	 * considering the column number.
	 */
	public static GeneralizedLocation fromLineColumn(GeneralizedFile file,
			int line, Column column) {

		GeneralizedLocation result = new GeneralizedLocation ();

		result.fLine = line;
		result.fColumn = column;
		result.fFile = file;

		return result;
	}

	/**
	 * Create a location from a text marker. If the marker is coming from an
	 * opened document, then the location will be updated according to the
	 * current location in the document, even if the document is not saved yet.
	 *
	 * @param marker
	 * @return
	 */
	public static GeneralizedLocation fromTextMarker (IMarker marker) {
		IFile file = (IFile) marker.getResource();

		GeneralizedFile genFile = GeneralizedFile.fromWorkspace(file);

		AbstractMarkerAnnotationModel model = genFile.getAnnotationModel();

		int line = -1, offset = -1, length = -1;

		if (model != null) {
			Position pos = model.getMarkerPosition(marker);

			if (pos != null) {
				try {
					offset = pos.getOffset();
					line = genFile.getDocument().getLineOfOffset(pos.getOffset()) + 1;
					length = pos.getLength();
				} catch (BadLocationException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				}
			}
		}

		if (line == -1) {
			//  The default are chosen here so that even if the attributes are
			//  not set the values will make sense.

			line = marker.getAttribute(IMarker.LINE_NUMBER, 1);
			offset = marker.getAttribute(IMarker.CHAR_START, 0);
			length = marker.getAttribute(IMarker.CHAR_END, 0) - offset + 1;
		}

		GeneralizedLocation loc = GeneralizedLocation.fromAbsoluteLocation(
				genFile, line, offset);

		loc.setLength(length);

		return loc;
	}

	/**
	 * Create a generalized location based on a string.
	 *
	 * @param location formated filename:line:column
	 * @param isGPSFileName True if the file is supposed to be only the file
	 * name. We will try to retreive the full path by asking GPS. This is very
	 * expensive, callers should try to get the full path by other ways.
	 * @return
	 * @throws GPSError
	 * @throws PythonError
	 */
	public static GeneralizedLocation fromLineColumn(String location, IProject project,
			int tabWidth, boolean isGPSFileName) {
		Matcher matchesEntity = fgLocationPattern.matcher(location);

		if (!matchesEntity.find()) {
			return null;
		}

		String filePath = matchesEntity.group(1).replaceAll("////////", "////");

		GeneralizedFile adaFile = GeneralizedFile.fromOSPath(project, filePath);

		if (isGPSFileName) {
			// This mean that the file we've parsed is unknown by Eclipse.
			// Let's see if GPS knows it.

			GNATProjectRegistry registry = (GNATProjectRegistry) GNATbenchSession
					.getDefault().getOrLoadRegistry(project);

			adaFile = GeneralizedFile.fromOSPath(project, registry
					.getSourcePath(matchesEntity.group(1)).toOSString());
		}

		GeneralizedLocation result = fromLineColumn(adaFile, Integer
				.parseInt(matchesEntity.group(2)), new Column(Integer
				.parseInt(matchesEntity.group(3)), tabWidth));

		return result;
	}

	/**
	 * This class is reponsible to add a marker corresponding to that
	 * location.
	 */
	private class AddMarker extends WorkspaceModifyOperation {
		public IMarker fMarker;
		public String fType;

		public AddMarker (String type) {
			fType = type;
		}

		@Override
		protected void execute(IProgressMonitor monitor) throws CoreException,
				InvocationTargetException, InterruptedException {

			try {
				if (fFile.getFile() != null && fFile.getFile().exists()) {
					fMarker = fFile.getFile().createMarker(fType);

					fMarker.setAttribute(IMarker.CHAR_START, getOffset ());
					fMarker.setAttribute(IMarker.CHAR_END, fOffset + fLength);
					fMarker.setAttribute(IMarker.LINE_NUMBER, fLine);
				}
			} catch (BadLocationException e) {
				throw new InvocationTargetException (e);
			}

		}


	}

	/**
	 * Creates an returns a marker of the given type created from that
	 * location. The type has to be a descendant of textmarker.
	 *
	 * @param type
	 * @return
	 * @throws GNATbenchCoreException
	 */
	public IMarker toMarker(final String type) {
		try {
			AddMarker m = new AddMarker (type);
			m.run(new NullProgressMonitor ());

			return m.fMarker;
		} catch (InvocationTargetException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} catch (InterruptedException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		return null;
	}

	/**
	 * Return the column of that location, taking tabs into account.
	 * @return
	 */
	public Column getColumn () {
		if (fColumn == null) {
			fColumn = fFile.computeColumnNumber(fLine, fOffset,
					Column.DEFAULT_GNAT_TAB_WIDTH);
		}

		return fColumn;
	}

	/**
	 * Return the offset from the beginnig of the file for that location.
	 *
	 * @return
	 */
	public int getOffset () throws BadLocationException {
		if (fOffset == -1) {
			fOffset = fFile.computeOffset(fLine, fColumn);
		}

		return fOffset;
	}

	/**
	 * Set the length of the data pointed by that location.
	 * @param length
	 */
	public void setLength (int length) {
		fLength = length;
	}

	/**
	 * Return the length of the data pointed by that location.
	 *
	 * @return
	 */
	public int getLength () {
		return fLength;
	}

	/**
	 * Return the line number, first line of the file is 1.
	 *
	 * @return
	 */
	public int getLine () {
		return fLine;
	}

	/**
	 * Return the file which that location is refering to.
	 *
	 * @return
	 */
	public GeneralizedFile getFile () {
		return fFile;
	}

	public String toString () {
		return fFile + ":" + fLine + ":" + getColumn().getColumnIndex();
	}

	public boolean equals (Object cmp) {
		if (!(cmp instanceof GeneralizedLocation))
			return false;

		GeneralizedLocation loc = (GeneralizedLocation) cmp;

		if (!getFile ().equals(loc.getFile())) {
			//  If the files are not equals, return false.
			return false;
		}

		try {
			if (fOffset != -1 && loc.fOffset != -1) {
				//  If we can compare the offset, that's the easiest

				if (getOffset () != loc.getOffset()) {
					return false;
				}
			} else if (fLine != -1 && fColumn != null && loc.fLine != -1
					&& loc.fColumn != null) {
				//  Otherwise, we try to compare lines/column, not too bad

				if (getLine ()!= loc.getLine () || !getColumn ().equals (getColumn())) {
					return false;
				}
			} else {
				//  Otherwise, we'll compute the offset of one of the two and
				//  compare it

				if (getOffset () != loc.getOffset()) {
					return false;
				}
			}
		} catch (BadLocationException e) {
			return false;
		}

		//  If we managed to go through all of these stages, then we're good

		return true;
	}

}
