/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal.analyzer;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;

import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IStorage;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IRegion;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IPathEditorInput;
import org.eclipse.ui.IStorageEditorInput;
import org.eclipse.ui.IURIEditorInput;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.texteditor.AbstractMarkerAnnotationModel;

import com.adacore.gnatbench.core.GNATbenchCoreException;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.adaeditor.AdaFileEditorInput;
import com.adacore.gnatbench.library.GNATCOLL.Filesystem.Filesystem_String;
import com.adacore.gnatbench.library.GNATCOLL.VFS.VFS_Package;
import com.adacore.gnatbench.library.GNATCOLL.VFS.Virtual_File;
import com.adacore.ajis.NativeException;

public class GeneralizedFile {

	private String fFullPath = null;
	private IFile fFile = null;
	private IProject fProject = null;
	private IEditorInput fInput;
	private IFileStore fFileStore = null;

	/**
	 * Creates a file from an absolute OS path.
	 *
	 * Warning ! If project is null, then the file will possibly not be
	 * associated with any project and therefore some capabilities (like
	 * browsing) will not work on it.
	 *
	 * @param project
	 * @param path
	 * @return
	 */
	public static GeneralizedFile fromOSPath(IProject project, String path) {
		GeneralizedFile theFile = null;

		IFile linkedFile;

		// Then, check if the file can be found within the worspace

		IFile[] files = ResourcesPlugin.getWorkspace().getRoot()
				.findFilesForLocation(new Path(path));

		if (files.length > 0) {
			theFile = new GeneralizedFile(files[0]);

			return theFile;
		}

		// The file is a linked file from the project given in parameter

		if (project != null) {
			linkedFile = getLinkedResourceFor(project, path);

			if (linkedFile != null) {
				// If the file is contained into a linked folder, then use a
				// file with the linked folder rather than the full path
				// name.
				theFile = new GeneralizedFile(linkedFile);
			}
		}

		// The file is a linked file, but from an other project
		if (theFile == null) {
			IProject[] projects = ResourcesPlugin.getWorkspace().getRoot()
					.getProjects();

			for (int i = 0; i < projects.length; ++i) {
				linkedFile = getLinkedResourceFor(projects[i], path);

				if (linkedFile != null) {
					// If the file is contained into a linked folder, then use a
					// file with the linked folder rather than the full path
					// name.
					theFile = new GeneralizedFile(linkedFile);
					break;
				}
			}
		}

		// The file is not a linked file.
		if (theFile == null) {
			theFile = new GeneralizedFile(path);
		}

		// The file is external, so we assume the current project is the
		// proper one
		if (theFile.fProject == null) {
			theFile.fProject = project;
		}

		return theFile;
	}

	/**
	 * Create a generalized file from a GPS Virtual_File.
	 *
	 * @param project
	 * @param file
	 * @return
	 */
	public static GeneralizedFile fromVirtualFile(IProject project,
			Virtual_File file) {
		return fromOSPath(project, file.Full_Name(true).toString());
	}

	/**
	 * Return the Virtual_File corresponding to that file.
	 *
	 * @return
	 */
	public Virtual_File toVirtualFile() {
		return VFS_Package.Create(new Filesystem_String(getOSPath()));
	}

	/**
	 * Creates a file from an eclipse-relative path.
	 *
	 * @param path
	 * @return
	 */
	public static GeneralizedFile fromWorkspace(String path) {
		return new GeneralizedFile(ResourcesPlugin.getWorkspace().getRoot()
				.getFile(new Path(path)));
	}

	/**
	 * Create a generalized file based on a workspace resource.
	 *
	 * @param file
	 * @return
	 */
	public static GeneralizedFile fromWorkspace(IFile file) {
		return new GeneralizedFile(file);
	}

	public static GeneralizedFile fromEditorInput (IEditorInput input) {
		GeneralizedFile file;

		try {
			if (input instanceof AdaFileEditorInput) {
				AdaFileEditorInput adaFileInput = (AdaFileEditorInput) input;

				file = fromOSPath(adaFileInput.getProject(), adaFileInput
						.getPath().toOSString());
			} else if (input instanceof IFileEditorInput) {
				IFileEditorInput fileInput = (IFileEditorInput) input;

				file = fromWorkspace(fileInput.getFile());
			} else if (input instanceof IPathEditorInput) {
				IPathEditorInput pathInput = (IPathEditorInput) input;

				file = fromOSPath(null, pathInput.getPath().toOSString());
			} else if (input instanceof IStorageEditorInput) {
				IStorageEditorInput storageInput = (IStorageEditorInput) input;

				IStorage storage = storageInput.getStorage();

				if (storage instanceof IFile) {
					file = fromWorkspace((IFile) storage);
				} else {
					file = fromOSPath(null, storageInput.getStorage()
							.getFullPath().toOSString());
				}
			} else if (input instanceof IURIEditorInput) {
				IURIEditorInput uriInput = (IURIEditorInput) input;

				file = fromOSPath(null, uriInput.getURI().getPath());
			} else {
				GNATbenchCorePlugin.getDefault().logError(
						"Input " + input.getClass()
								+ " not supported for GeneralizedFile", null);

				return null;
			}
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);

			return null;
		}

		file.fInput = input;

		return file;
	}

	private GeneralizedFile(String fullPath) {
		fFullPath = fullPath;
	}

	private GeneralizedFile(IFile file) {
		fFile = file;
		fProject = file.getProject();
	}

	/**
	 * Returns the IFile linked to this location. If the file is not managed by
	 * the workspace, then this function may return either null or an IFile that
	 * will returns false on its exists() call.
	 */
	public IFile getFile() {
		return fFile;
	}

	/**
	 * Returns the full path of the file, according to the os naming scheme.
	 *
	 * @return
	 */
	public String getOSPath() {
		if (fFullPath == null) {
			fFullPath = fFile.getLocation().toOSString();
		}

		return fFullPath;
	}

	/**
	 * Returns the path according to the Eclipse naming scheme. This will return
	 * the full path if the file is not on the workspace.
	 *
	 * @return
	 */
	public String getEclipsePath() {
		IFile file = getFile();

		if (file != null)
			return file.getFullPath().toString();
		else
			return fFullPath;
	}

	/**
	 * Returns The project holding this location, null if none.
	 */
	public IProject getProject() {
		if (fProject != null)
			return fProject;

		IFile file = getFile();

		if (file == null || !file.exists()) {
			return null;
		} else {
			return file.getProject();
		}
	}

	/**
	 * If the path is in one of the linked folders/files of the project, returns
	 * the corresponding resource, null otherwise.
	 *
	 * @param project
	 * @param path
	 * @return
	 */
	private static IFile getLinkedResourceFor(IProject project, String path) {
		IResource[] resources = null;
		IPath sourcePath = new Path(path);

		try {
			resources = project.members();
		} catch (CoreException e) {
			return null;
		}

		for (int i = 0; i < resources.length; ++i) {
			if (resources[i] instanceof IFolder) {
				IFolder folder = (IFolder) resources[i];

				if (folder.isLinked()) {
					IPath folderPath = folder.getLocation();

					if (folderPath != null
							&& sourcePath.matchingFirstSegments(folderPath) == folderPath
									.segmentCount()) {

						return folder
								.getFile(sourcePath
										.removeFirstSegments(folderPath
												.segmentCount()));
					}
				}
			}
		}

		return null;
	}

	/**
	 * Return the string of the line (starting at 1) from a file
	 * @param line
	 * @return
	 */
	public String getLine(int line) {
		try {
			IRegion lineInfo;

			lineInfo = getDocument().getLineInformation(line - 1);

			if (lineInfo.getLength() == 0) {
				return "";
			} else {
				return GNATbenchCorePlugin.getDefault().getDocumentBuffer(this)
						.getDocument().get(lineInfo.getOffset(),
								lineInfo.getLength());
			}
		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		return "";
	}

	/**
	 * Computes the column number based on a line number (from 1) and an offset
	 * from the file (from 0).
	 *
	 * @param line
	 * @param offset
	 * @return
	 */
	public Column computeColumnNumber(int line, int offset, int tabWidth) {
		try {
			int lineStartOffset = getDocument().getLineOffset(line - 1);

			String lineStr = getLine(line);

			int offsetFromLine = offset - lineStartOffset;

			return Column.toColumn(offsetFromLine, lineStr,
					Column.DEFAULT_GNAT_TAB_WIDTH);
		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} catch (NativeException e1) {
			throw e1;
		}

		return null;
	}

	/**
	 * Load and return the data of the file from the disk.
	 *
	 * @return
	 */
	public String loadFile() {
		try {
			FileInputStream fileInput;

			fileInput = new FileInputStream(getOSPath());

			String result = "";
			byte[] byteBuffer = new byte[2048];

			int nbRed = 0;

			while ((nbRed = fileInput.read(byteBuffer)) > 0) {
				result += new String(byteBuffer, 0, nbRed);
			}

			return result;
		} catch (FileNotFoundException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} catch (IOException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		return "";
	}

	/**
	 * Computes the offset from the beginning of the file, based on a line
	 * number (from 0) and a column number (from 1).
	 *
	 * @param line
	 * @param offset
	 * @return
	 */
	public int computeOffset(int line, Column column)
			throws BadLocationException {
		int lineStartOffset = getDocument().getLineOffset(line - 1);

		String lineStr = getLine(line);

		return Column.toIndex(column, lineStr) + lineStartOffset - 1;
	}

	public IEditorInput getInput () {
		if (fInput == null) {
			IFile theFile = getFile ();

			if (theFile != null) {
				fInput = new FileEditorInput (theFile);
			} else {
				fInput = new AdaFileEditorInput(new Path(getOSPath()),
						getProject());
			}
		}

		return fInput;
	}

	/**
	 * Return the document holding that file information - if there's an editor
	 * available, the document will be extracted from that editor, otherwise it
	 * will be taken from the disk.
	 *
	 * This will also install default GNATbench document partitioners.
	 *
	 * @return
	 */
	public IDocument getDocument () {
		return GNATbenchCorePlugin.getDefault().getDocumentBuffer(this)
				.getDocument();
	}

	/**
	 * Return the annotation model associated to the document storing contents
	 * of this file.
	 *
	 * @return
	 */
	public AbstractMarkerAnnotationModel getAnnotationModel () {
		IEditorInput input = getInput ();

		return (AbstractMarkerAnnotationModel) GNATbenchCorePlugin.getDefault()
				.getDocumentProvider().getAnnotationModel(input);
	}

	public String toString () {
		return getOSPath();
	}

	/**
	 * Save the contents of the document currently storing that file.
	 */
	public void saveFile () {
		try {
			GNATbenchCorePlugin.getDefault().getDocumentProvider().saveDocument(
					new NullProgressMonitor(), getInput(), getDocument(), true);
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}
	}

	public boolean equals (Object cmp) {
		if (!(cmp instanceof GeneralizedFile)) {
			return false;
		}

		GeneralizedFile file = (GeneralizedFile) cmp;

		if (getFile () != null) {
			//  If the first one is coming from the workspace, then use a
			//  workspace comparison

			return getFile ().equals (file.getFile ());
		} else if (file.getFile () == null) {
			//  If the second one is not coming from the workspace either,
			//  then try a path comparison.

			return getOSPath().equals(file.getOSPath());
		} else {
			//  If one is coming from the workspace and one is not, the files
			//  are not equals.

			return false;
		}
	}

	public IPath getLocation() {
		return new Path (getOSPath());
	}

	public IResource getResource() {
		return getFile ();
	}

	public IFileStore getFileStore () {
		if (fFileStore == null) {
			try {
				fFileStore = GNATbenchCorePlugin.getDefault().getEFSRegistry()
						.getUniqueStore(URIUtil.toURI(getOSPath()));
			} catch (GNATbenchCoreException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		}

		return fFileStore;
	}
}
