/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal.analyzer;

/**
 * This class represent a column, computed from a given with of tab.
 */
public class Column {

	public static int DEFAULT_GNAT_TAB_WIDTH = 8;

	private int fColumnIndex;
	private int fTabWidth;

	/**
	 * Creates a column associated with a given tab width. Since the column
	 * information can come from various contextes (Eclipse, GNAT...) diffent
	 * column computed with different tab with may  be detected.
	 *
	 * @param columnIndex
	 * @param tabWidth
	 */
	public Column(int columnIndex, int tabWidth) {
		fColumnIndex = columnIndex;
		fTabWidth = tabWidth;
	}

	/**
	 * Return the actual column number.
	 *
	 * @return
	 */
	public int getColumnIndex() {
		return fColumnIndex;
	}

	/**
	 * Return the tab width on which this column has been computed.
	 *
	 * @return
	 */
	public int getTabWidth() {
		return fTabWidth;
	}

	/**
	 * Return a column from an character index computed from the beginning of
	 * the line.
	 *
	 * @param index
	 * @param line
	 * @param tabWidth
	 * @return
	 */
	public static Column toColumn(int index, String line, int tabWidth) {
		int currentIndex = 0;
		int currentCol = 1;

		while (true) {
			if (currentIndex >= index) {
				break;
			}

			if (currentIndex <= line.length() - 1
					&& line.charAt(currentIndex) == '\t') {
				currentCol = currentCol + tabWidth
						- ((currentCol - 1) % tabWidth);
			} else {
				currentCol = currentCol + 1;
			}

			currentIndex = currentIndex + 1;

		}

		return new Column(currentCol, tabWidth);
	}

	/**
	 * Return the index of the character computed from the beginning of the
	 * line, using the column given in parameter.
	 *
	 * @param column
	 * @param line
	 * @return
	 */
	public static int toIndex(Column column, String line) {
		int currentCol = 1;
		int index = 0;

		while (currentCol < column.fColumnIndex
				&& index <= line.length() - 1
				&& line.charAt(index) != '\n') {

			if (index < line.length() - 1 && line.charAt(index) == '\t') {
				currentCol = currentCol + column.fTabWidth
						- ((currentCol - 1) % column.fTabWidth);
			} else {
				currentCol = currentCol + 1;
			}

			index = index + 1;
		}

		return index + 1;
	}

	public boolean equals (Object cmp) {
		if (!(cmp instanceof Column)) {
			return false;
		} else {
			//  XXX: This is not completely true - in case of different tab
			//  with we would need to do some more checks - but so far let's
			//  assume that it's always the case (= DEFAULT_GNAT_TAB_WIDTH).
			return fColumnIndex == ((Column) cmp).getColumnIndex();
		}
	}
}