/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal.analyzer;

import java.util.LinkedList;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.InvalidRegistryObjectException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.Image;
import org.osgi.framework.Bundle;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.Language.Simple_Construct_Information;
import com.adacore.gnatbench.library.Standard.AdaString;

public class AdaSimpleConstruct {

	protected int fOffsetBegin;
	protected int fOffsetEntity;
	protected int fOffsetEnd;
	protected int fLineBegin;
	protected int fLineEntity;
	protected int fLineEnd;
	protected String fName;
	protected int fCategory;
	protected GeneralizedLocation fLocation;
	protected AdaConstruct fParent;
	protected int fVisibility;
	protected boolean fIsDeclaration;

	protected Image fImage = null;
	protected ImageDescriptor fImageDesc = null;

	private LinkedList <Object> fAdapters = new LinkedList<Object>();

	public AdaSimpleConstruct(Simple_Construct_Information content,
			GeneralizedFile file) {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			fOffsetBegin = content.Sloc_Start().Index() - 1;
			fOffsetEntity = content.Sloc_Entity().Index() - 1;
			fOffsetEnd = content.Sloc_End().Index() - 1;
			fLineBegin = content.Sloc_Start().Line() - 1;
			fLineEntity = content.Sloc_Entity().Line() - 1;
			fLineEnd = content.Sloc_End().Line() - 1;

			AdaString name = content.Name();

			if (name != null) {
				fName = content.Name().toString();
			} else {
				fName = "";
			}

			fCategory = content.Category();
			fVisibility = content.Visibility();

			fIsDeclaration = content.Is_Declaration();

			fLocation = GeneralizedLocation.fromAbsoluteLocation(file,
					fLineEntity, content.Sloc_Entity().Index() - 1);
			fLocation.setLength(fName.length());
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	public boolean equals (Object o) {
		if (!(o instanceof AdaSimpleConstruct)) {
			return false;
		}

		AdaSimpleConstruct construct = (AdaSimpleConstruct) o;

		return construct.fLocation.equals(fLocation);
	}

	/**
	 * @return the location of the construct.
	 */
	public GeneralizedLocation getLocation() {
		return fLocation;
	}

	/**
	 * Return the image corresponding to the construct
	 */
	public Image getImage() {
		if (fImage == null) {
			fImage = getImageFromExtension();
		}

		return fImage;
	}

	public ImageDescriptor getImageDescriptor() {
		if (fImageDesc == null) {
			fImageDesc = getImageDescriptorFromExtension();
		}

		return fImageDesc;
	}

	/**
	 * Return the image by looking at the proper extension
	 *
	 * @return
	 */
	private Image getImageFromExtension () {
		return getIconProvider().getImage(this);
	}

	private ImageDescriptor getImageDescriptorFromExtension () {
		return getIconProvider().getImageDescriptor(this);
	}

	public static Image getImage (int category, int visibility) {
		return getIconProvider ().getImage(category, visibility);
	}

	private static IAdaConstructIconProvider getIconProvider () {
		IExtensionRegistry er = Platform.getExtensionRegistry();
		IExtensionPoint ep = er.getExtensionPoint(GNATbenchCorePlugin.getId(),
				"constructIcon");

		IExtension[] extensions = ep.getExtensions();

		if (extensions.length == 0) {
			return null;
		}

		IExtension relevantExtension = extensions[extensions.length - 1];

		IConfigurationElement[] elements = relevantExtension
				.getConfigurationElements();

		for (int i = 0; i < elements.length; ++i) {
			if (elements[i].getName().equals("provider")) {
				try {
					Bundle bundle = Platform.getBundle(relevantExtension
							.getNamespaceIdentifier());

					IAdaConstructIconProvider iconProvider =
					(IAdaConstructIconProvider) bundle
							.loadClass(elements[i].getAttribute("class"))
							.newInstance();

					return iconProvider;
				} catch (InvalidRegistryObjectException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				} catch (InstantiationException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				} catch (IllegalAccessException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				} catch (ClassNotFoundException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				}
			}
		}

		return null;
	}

	public void setAdapter (Object adapter) {
		fAdapters.add(adapter);
	}

	public Object getAdapter (Class <? extends Object> adapterClass) {
		for (Object element : fAdapters) {
			if (element.getClass().isAssignableFrom (adapterClass)) {
				return element;
			}

		}
		return null;
	}

	/**
	 *
	 * @return The category of the construct.
	 */
	public int getCategory() {
		return fCategory;
	}

	/**
	 *
	 * @return The offset where the construct start (Eclipse formatted).
	 */
	public int getOffsetBegin() {
		return fOffsetBegin;
	}

	/**
	 *
	 * @return The offset where the construct name starts (Eclipse formatted).
	 */
	public int getOffsetEntity() {
		return fOffsetEntity;
	}

	/**
	 *
	 * @return The offset where the construct ends (Eclipse formatted).
	 */
	public int getOffsetEnd() {
		return fOffsetEnd;
	}

	/**
	 *
	 * @return The line where the construct starts (Eclipse formatted).
	 */
	public int getLineBegin() {
		return fLineBegin;
	}

	/**
	 * @return The line where the construct name begins (Eclipse formatted).
	 */
	public int getLineEntity() {
		return fLineEntity;
	}

	/**
	 * @return The line where the construct ends (Eclipse formatted).
	 */
	public int getLineEnd() {
		return fLineEnd;
	}

	/**
	 *
	 * @return The name of the construct. According to the gps_utils
	 * implementation, <no name> will be returned otherwise.
	 */
	public String getName() {
		return fName;
	}


	/**
	 * Returns the visibility of the construct if relevant, otherwise always
	 * VISIBILITY_PUBLIC.
	 *
	 * @return
	 */
	public int getVisibility() {
		return fVisibility;
	}

	/**
	 * @return The name of the construct and its profile, if relevant.
	 */
	public String toString() {
		return fName;
	}
}
