/*******************************************************************************
 * Copyright (C) 2005-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal.analyzer;

import java.util.Iterator;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.text.BadLocationException;

import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.analyzer.IAdaDocument;
import com.adacore.gnatbench.core.analyzer.IAdaEntityReference;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.adaeditor.AdaDocumentBuffer;
import com.adacore.gnatbench.core.internal.browsing.AdaCodeBrowser;
import com.adacore.gnatbench.core.internal.gpswrappers.Entity;
import com.adacore.gnatbench.core.internal.utils.SyntaxUtils;

/**
 * This class hold a reference to an entity from an Ada file.
 */
public class AdaEntityReference implements IAdaEntityReference {

	private String fName;
	private GeneralizedLocation fLocation;
	private int fOffset;
	private AdaDocumentBuffer fBuffer;

	public AdaEntityReference(GeneralizedLocation location)
			throws BadLocationException {

		fBuffer = (AdaDocumentBuffer) GNATbenchSession.getDefault()
				.getAdaDocument(location.getFile().getFileStore());

		fName = fBuffer.getContent(location);
		fLocation = location;
		fOffset = location.getOffset();

	}

	public AdaEntityReference(AdaDocumentBuffer buffer,
			GeneralizedLocation location, int offset) {

		fName = buffer.getContent(location);
		fLocation = location;
		fOffset = offset;
		fBuffer = buffer;
	}

	public AdaEntityReference(AdaDocumentBuffer buffer, AdaConstruct construct) {
		try {
			fName = construct.getName();
			fLocation = construct.getLocation();
			fOffset = fLocation.getOffset();
		} catch (BadLocationException e) {
			fOffset = 0;
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}
	}

	/**
	 *
	 * @return The name of the entity.
	 */
	public String getName() {
		return fName;
	}

	/**
	 *
	 * @return The location of the entity.
	 */
	public GeneralizedLocation getLocation() {
		return fLocation;
	}

	/**
	 *
	 * @return True if the entity name is a regular Ada identifier, false
	 * otherwise.
	 */
	public boolean isRegularIdentifier() {
		if (fName.length() == 0)
			return false;

		if (!(fName.charAt(0) >= 'A' && fName.charAt(0) <= 'Z')
				&& !(fName.charAt(0) >= 'a' && fName.charAt(0) <= 'z')) {
			return false;
		}

		for (int i = 0; i < SyntaxUtils.fgAdaKeywords.length; ++i) {
			if (SyntaxUtils.fgAdaKeywords[i].equals(fName))
				return false;
		}

		return true;
	}

	/**
	 *
	 * @return The offset of the beginning of the entity in the buffer.
	 */
	public int getOffset() {
		return fOffset;
	}

	public IAdaDocument getAdaDocument() {
		return fBuffer;
	}

	public IAdaEntityReference getDeclaration() {
		Entity entity = new Entity (this);

		GeneralizedLocation specLoc = entity.getSpec ();

		AdaDocumentBuffer specBuffer = (AdaDocumentBuffer) GNATbenchSession
				.getDefault().getAdaDocument(specLoc.getFile().getFileStore());

		try {
			return new AdaEntityReference(specBuffer, specLoc, specLoc.getOffset());
		} catch (BadLocationException e) {
			return null;
		}
	}

	public Iterable <IAdaEntityReference> getReferences(final SearchScopes scope,
			final IProgressMonitor monitor) {
		return new Iterable <IAdaEntityReference>() {
			public Iterator<IAdaEntityReference> iterator() {
				return AdaCodeBrowser.getLocations(AdaEntityReference.this,
						scope, monitor);
			}

		};
	}

}