/*******************************************************************************
 * Copyright (c) 2005 AdaCore and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal.analyzer;

import java.util.LinkedList;

/**
 * This file provides base type and functionalities for construct filters.
 * 
 * It is possible to add a filter after another one. Then, each filter has to 
 * valid the construct.
 * 
 * @author Quentin Ochem
 */
public abstract class AdaConstructFilter {

	public AdaConstructFilter fNextFilter = null;
	
	public AdaConstructFilter () {
		
	}
	
	/**
	 * Creates a new filter in front of the one given in parameter.
	 * 
	 * @param nextFilter
	 */
	public AdaConstructFilter (AdaConstructFilter nextFilter) {
		fNextFilter = nextFilter;
	}
	
	/**
	 * This function has to be implemented by final filter implementation. It
	 * holds the algorithm that decides if the construct should be filterer or
	 * not
	 *  
	 * @param construct
	 * @return False if the construct should be blocked, true otherwise.
	 */
	public abstract boolean simpleFilter(AdaConstruct construct);
	
	/**
	 * This function has to be implemented by the structural layer of the 
	 * filter. It should use simpleFilter to determine if a single construct
	 * is filtered or not.
	 *  
	 * @param construct
	 * @return
	 */
	protected abstract boolean internalFilter(AdaConstruct construct);
	
	/**
	 * Initializes the filter before the calls of internalFilter. This 
	 * function is called before the analyze of a whole structure.
	 *
	 */
	protected abstract void internalInitialize ();
	
	/**
	 * Returns the list minus the filtered chidren, plus subchildren if any 
	 * have to be added.
	 * 
	 * @param list The children to be filtered. Each element must be an intance
	 * of AdaConstruct.
	 * @return
	 */
	protected abstract LinkedList<AdaConstruct> internalFilterChildren(
			LinkedList<AdaConstruct> list);
	
	/**
	 * Returns the list minus the filtered chidren, plus subchildren if any 
	 * have to be added.
	 * 
	 * @param list The children to be filtered. Each element must be an intance
	 * of AdaConstruct.
	 * @return
	 */
	public LinkedList <AdaConstruct>getFilteredChildren (LinkedList <AdaConstruct> list) {
		LinkedList <AdaConstruct> result;
		
		if (fNextFilter != null) {
			result = fNextFilter.getFilteredChildren(list); 
		} else {
			result = list;
		}
		
		result = internalFilterChildren(result);
		
		return result;
	}
	
	/**
	 * @param construct
	 * @return True if the construct passes, false if it is blocked. 
	 */
	public final boolean passFilter (AdaConstruct construct) {
		if (fNextFilter == null) {
			return internalFilter(construct);
		} else {
			if (!internalFilter(construct)) {
				return false;
			} else {
				return fNextFilter.passFilter(construct);
			}
		}
	}
	
	/**
	 * This function has to be implemented by the structural layer of the 
	 * filter. It should use simpleFilter to determine if a single construct
	 * is filtered or not.
	 *  
	 * @param construct
	 * @return
	 */
	public final void initialize () {
		internalInitialize();
		
		if (fNextFilter != null) {
			fNextFilter.internalInitialize();
		}
	}
	
}
