/*******************************************************************************
 * Copyright (c) 2005, 2007 AdaCore.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal.analyzer;

import java.util.LinkedList;

/**
 * This filter will systematically cut the items below the unfiltered ones. 
 * That means that a child, direct or indirect, of a blocked construct is
 * blocked.
 */
public abstract class AdaConstructCutFilter extends AdaConstructFilter {

	private AdaConstruct fConstructFilter = null;
	
	/**
	 * @param filter
	 */
	public AdaConstructCutFilter(AdaConstructFilter filter) {
		super(filter);
	}
	
	public AdaConstructCutFilter() {
		super();
	}

	public boolean internalFilter (AdaConstruct construct) {
		// If there is already a construct filtered for the 
		// construct and if it is a parent of the current 
		// construct, then do not fire it.
		if (fConstructFilter != null) {
			if (fConstructFilter.isParent(construct))
				return false;
		}

		fConstructFilter = getFilteredConstruct(construct);

		// This means that there is at least one construct in the 
		// construct parent that is filtered by the listener.
		if (fConstructFilter != null)
			return false;
		
		return true;
	}
	
	public void internalInitialize () {
		fConstructFilter = null;
	}
	
	protected LinkedList<AdaConstruct> internalFilterChildren(
			LinkedList<AdaConstruct> list) {
		LinkedList<AdaConstruct> result = new LinkedList<AdaConstruct>();

		for (AdaConstruct element : list) {
			if (simpleFilter(element)) {
				result.add(element);
			}
		}
		
		return result;
	}
	
	
	/**
	 * 
	 * @param constructFilter
	 * @return The construct closest to the root that does not pass through
	 * the filter, null if none.
	 */
	public AdaConstruct getFilteredConstruct(AdaConstruct baseConstruct) {
		AdaConstruct current = baseConstruct;
		AdaConstruct deeper = null;

		while (current != null) {
			if (!simpleFilter(current)) {
				deeper = current;
			}

			current = current.getParent(null);
		}

		return deeper;
	}
}
