/*******************************************************************************
 * Copyright (c) 2005, 2008 AdaCore.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal.analyzer;

import java.util.LinkedList;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.model.IWorkbenchAdapter;

import com.adacore.gnatbench.core.internal.adaeditor.AdaDocumentBuffer;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.Language.Construct_Information;
import com.adacore.gnatbench.library.Language.Simple_Construct_Information;
import com.adacore.gnatbench.library.Standard.AdaString;
import com.adacore.gnatbench.library.Language.Language_Category;

/**
 * Holds all the information for a construct. This class is <b>NOT</b> 
 * intended to be subclassed or instancied outside the analyzer package.
 * 
 * @author ochem
 */
public class AdaConstruct extends AdaSimpleConstruct implements IWorkbenchAdapter {

	private LinkedList<AdaConstruct> fChildren = new LinkedList<AdaConstruct>();
	private String fProfile;
	
	private String fOriginalText;
	
	/**
	 * These two variables are used to store other part of the same construct.
	 */
	private AdaConstruct fPart1 = null;
	private AdaConstruct fPart2 = null;
	
	/**
	 * Creates a construct by extracting all the relevant data from the 
	 * construct access. The construct access will not be stored, so it
	 * can be deleted after this call.
	 * 
	 * @param content
	 * @param filePath
	 */
	AdaConstruct(Construct_Information content, AdaDocumentBuffer buffer) {
		super(toSimpleConstructInformation(content), buffer.getFile());
		
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();
		
		try {
			AdaString prof = content.Profile ();	
			
			if (prof != null) {
				fProfile = prof.toString();
			} else {
				fProfile = new String("");
			}

			fOriginalText = buffer.toString();
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * 
	 * @param node
	 * @return True if the position of the node given in parameters is enclosed
	 * by the current instance.
	 */
	public boolean include(AdaConstruct node) {
		return fOffsetBegin <= node.fOffsetBegin
				&& fOffsetEnd >= node.fOffsetEnd;
	}
	
	/**
	 * 
	 * @param node
	 * @return True if the position of the node given in parameters is enclosed
	 * by the current instance.
	 */
	public boolean include(GeneralizedLocation location) {
		// TODO: add a test on the columns.
		return fLineBegin <= location.getLine() - 1
				&& fLineEnd >= location.getLine() - 1;
	}


	/**
	 * Adds a node in the construct children.
	 * @param node
	 */
	public void add(AdaConstruct node) {
		fChildren.add(node);
		node.fParent = this;
	}

	/**
	 * Returns true if there is children for the construct, according to
	 * the filter, false otherwise.
	 * 
	 * @param filter
	 * @return
	 */
	public boolean hasChildren(IAdaConstructFilterProvider filter) {
		if (filter == null) {
			return fChildren.size() > 0;
		} else {
			return getChildren(filter).length > 0;
		}	
	}

	/**
	 * Returns the constructs according to the filter given in parameter.
	 * 
	 * @param constructFilter The actual filters for the children, if null all
	 * children will be returned.
	 * @return
	 */
	public AdaConstruct[] getChildren(IAdaConstructFilterProvider constructFilter) {
		if (constructFilter == null) {
			return fChildren.toArray(new AdaConstruct [fChildren.size()]);
		} else {
			LinkedList <AdaConstruct> result = getChildrenList(constructFilter);
			
			return result.toArray(new AdaConstruct [result.size()]);
		}
	}

	/**
	 * Returns the constructs according to the filter given in parameter.
	 * 
	 * @param constructFilter The actual filters for the children, if null all
	 * children will be returned.
	 * @return
	 */
	public LinkedList <AdaConstruct> getChildrenList(IAdaConstructFilterProvider constructFilter) {
		if (constructFilter == null) {
			return fChildren;
		} else {
			return constructFilter.getFilter().getFilteredChildren(fChildren);
		}
	}

	/**
	 * Add to the destination LinkedList only the main parts of the construcs
	 * contained in the source parameter.
	 * 
	 * @param destination
	 * @param source
	 */
	private void addOnlyMains (LinkedList <AdaConstruct> destination, LinkedList <AdaConstruct> source) {
		for (AdaConstruct element : source) {
			if (element.isMainConstruct()) {
				destination.add(element);
			}
		}		
	}
	
	/**
	 * This functions returns all the children of all the parts. Different 
	 * parts of a same entity will be gathered in the main entity.
	 * 
	 * @param constructFilter
	 * @return The children of this construct and its various parts.
	 */
	public AdaConstruct[] getAllMainChildren(
			IAdaConstructFilterProvider constructFilter) {
		LinkedList <AdaConstruct> allChildren = new LinkedList<AdaConstruct>();
		
		addOnlyMains (allChildren, getChildrenList(constructFilter));
		
		AdaConstructFilter filter = constructFilter.getFilter();
		
		if (fPart1 != null && filter.passFilter(fPart1))
			addOnlyMains (allChildren, fPart1.getChildrenList(constructFilter));
		
		if (fPart2 != null && filter.passFilter(fPart2))
			addOnlyMains (allChildren, fPart2.getChildrenList(constructFilter));
		
		return allChildren.toArray(new AdaConstruct [allChildren.size()]);
	}
	
	/**
	 * Removes the current children of the construct and replace them by the
	 * ones given in parameter.
	 * 
	 * @param children
	 */
	public void setChildren(AdaConstruct[] children) {
		fChildren.clear();

		for (int i = 0; i < children.length; ++i) {
			children[i].fParent = this;
			fChildren.add(children[i]);
		}
	}

	/**
	 * @param filter If not null, will jump accross the parents that do not 
	 * correspond to the filter.
	 * @return The parent of the construct if any, null otherwise.
	 */
	public AdaConstruct getParent(IAdaConstructFilterProvider constructFilter) {
		if (fParent == null) return null;
		
		if (constructFilter == null) {
			return fParent;
		} else {
			if (constructFilter.getFilter().simpleFilter(fParent)) {
				return fParent;
			} else {
				return fParent.getParent(constructFilter);
			}
		}
	}

	/**
	 * @param filter If not null, will jump accross the parents that do not 
	 * correspond to the filter.
	 * @return The main parent part of the construct if any, null otherwise.
	 */
	public AdaConstruct getMainParent(IAdaConstructFilterProvider constructFilter) {
		if (fParent == null) return null;
		
		AdaConstruct mainParent = fParent.getMainConstruct();
		
		if (constructFilter == null) {
			return mainParent;
		} else {
			
			if (constructFilter.getFilter().simpleFilter(mainParent)) {
				return mainParent;
			} else {
				return mainParent.getMainParent(constructFilter);
			}
		}
	}

	/**
	 * @return The name of the construct and its profile, if relevant.
	 */
	public String toString() {
		if ((fCategory >= Language_Category.Cat_Task 
				&& fCategory <= Language_Category.Cat_Destructor)
				|| fCategory == Language_Category.Cat_Entry) {
			return fName + " " + getProfile();
		} else {
			return fName;
		}
	}

	/**
	 * Inverts the position of the construct children.
	 */
	public void invert() {
		LinkedList<AdaConstruct> newList = new LinkedList<AdaConstruct>();

		for (int i = fChildren.size() - 1; i >= 0; --i) {
			newList.add(fChildren.get((i)));
		}

		fChildren = newList;
	}

	/**
	 * 
	 * @param construct
	 * @return True if the instance needs to be adapter to the construct given
	 * in parameter.
	 */
	public boolean needsAdaptation(AdaConstruct construct) {
		return construct.fOffsetBegin != fOffsetBegin
				|| construct.fOffsetEnd != fOffsetEnd
				|| construct.fOffsetEntity != fOffsetEntity
				|| !fProfile.equals(construct.fProfile)
				|| fVisibility != construct.fVisibility
				|| fCategory != construct.fCategory;
	}

	/**
	 * Copy the position information from the construct given in parameter to
	 * the current one. This function is used when a construct has changed, and
	 * need to be updated according to new information.
	 * 
	 * @param e
	 */
	public void adapt(AdaConstruct e) {
		fOffsetBegin = e.fOffsetBegin;
		fOffsetEntity = e.fOffsetEntity;
		fOffsetEnd = e.fOffsetEnd;
		fLineBegin = e.fLineBegin;
		fLineEntity = e.fLineEntity;
		fLineEnd = e.fLineEnd;
		fProfile = e.fProfile;
		fVisibility = e.fVisibility;
		fCategory = e.fCategory;
		fOriginalText = e.fOriginalText;
		
		/* The corresponding image might have changed, in this case it has to
		 * be reloaded on the next call of getImage, that's why we reset its
		 * value here. */
		fImage = null;
	}

	/**
	 * Returns the full file path of the file that contains this construct.
	 * 
	 * @return
	 */
	public String getFilePath() {
		return fLocation.getFile().getOSPath();
	}

	/**
	 * 
	 * @param construct
	 * @return True if the current construct is the parent of the construct
	 * given is parameter.
	 */
	public boolean isParent(AdaConstruct construct) {
		AdaConstruct current = construct;

		while (current != null) {
			if (current == this)
				return true;

			current = current.fParent;
		}

		return false;
	}

	/**
	 * Returns the profile of the construct if any, otherwise an empty string. 
	 * The profile is normalized (extra spaces and lines are removed).
	 * 
	 * @return
	 */
	public String getProfile() {
		return fProfile.replaceAll("[\n\r]", "")
			.replaceAll("[ ]+", " ");
	}
	
	/**
	 * This function is used on debug purpose. It displays in the console the
	 * elements name and its children.
	 * 
	 * @param constructFilter
	 */
	public void printTree(IAdaConstructFilterProvider constructFilter) {	
		AdaConstruct [] children = getChildren(constructFilter);
		
		System.out.println(this);
		
		for (int i = 0; i < children.length; ++i) {
			children[i].printTree(constructFilter);
		}
	}
	
	/**
	 * Links the children that are parts of the same entity.
	 */
	public void groupChildren() {
		Object [] elements = fChildren.toArray();
		
		for (int i = 0; i < elements.length; ++i) {
			AdaConstruct construct = (AdaConstruct) elements [i];
			
			construct.fPart1 = null;
			construct.fPart2 = null;
		}
		
		for (int i = 0; i < elements.length; ++i) {
			AdaConstruct construct1 = (AdaConstruct) elements [i];
			
			if (construct1.fCategory < Language_Category.Cat_Package
					|| construct1.fCategory > Language_Category.Cat_Field) {
				continue;
			}
			
			construct1.groupChildren();
			
			for (int j = i + 1; j < elements.length; ++j) {
				AdaConstruct construct2 = (AdaConstruct) elements [j];

				if (construct2.fCategory < Language_Category.Cat_Package
						|| construct2.fCategory > Language_Category.Cat_Field) {
					continue;
				}
				
				if (construct1.fName.toLowerCase().equals(
						construct2.fName.toLowerCase())) {
					if (construct1.fCategory >= Language_Category.Cat_Class
							&& construct1.fCategory <= Language_Category.Cat_Field) {
						construct1.linkParts(construct2);
					} else {
						if (construct1.fCategory == construct2.fCategory
								&& construct1.getProfile().replaceAll(" ", "")
										.toLowerCase().equals(
												construct2.getProfile()
														.replaceAll(" ", "")
														.toLowerCase())
								&& !construct1.isGenericInstanciation()
								&& !construct2.isGenericInstanciation()) {
							construct1.linkParts(construct2);
						}
					}
				}
			}
		}
	}
	
	/**
	 * This code is supposed to analyze the text to determine if the 
	 * procedure / function is the instanciation of a generic unit. Would be
	 * better to use some support from the GPS parser.
	 * 
	 * @return
	 */
	private boolean isGenericInstanciation() {
		int i = fOffsetEntity;
		
		for (; i < fOriginalText.length(); ++i) {
			if (Character.isWhitespace(fOriginalText.charAt(i))) {
				break;
			}
		}

		for (; i < fOriginalText.length(); ++i) {
			if (!Character.isWhitespace(fOriginalText.charAt(i))) {
				break;
			}
		}
		
		i += fProfile.length();

		for (; i < fOriginalText.length(); ++i) {
			if (!Character.isWhitespace(fOriginalText.charAt(i))) {
				break;
			}
		}
		
		if (i >= fOriginalText.length() || fOriginalText.charAt(i) == ';') {
			return false;
		}

		for (; i < fOriginalText.length(); ++i) {
			if (!Character.isLetterOrDigit(fOriginalText.charAt(i))) {
				break;
			}
		}
		
		for (; i < fOriginalText.length(); ++i) {
			if (!Character.isWhitespace(fOriginalText.charAt(i))) {
				break;
			}
		}

		if (i + 3 >= fOriginalText.length()) return false;
		
		if ((fOriginalText.charAt(i) == 'n' || fOriginalText.charAt(i) == 'N')
				&& (fOriginalText.charAt(i + 1) == 'e' || fOriginalText
						.charAt(i + 1) == 'E')
				&& (fOriginalText.charAt(i + 2) == 'w' || fOriginalText
						.charAt(i + 2) == 'W')) {
			
			if (i + 3 >= fOriginalText.length()) return true;
			
			if (Character.isWhitespace(fOriginalText.charAt(i + 3))) {
				return true;	
			}
		}
		
		return false;
	}

	/**
	 * Links the construct given in parameter with the current one, assuming
	 * that the two are differents parts of the same entity.
	 * 
	 * @param construct
	 */
	private void linkParts (AdaConstruct construct) {
		if (fPart1 == null) fPart1 = construct;
		else fPart2 = construct;
		
		if (construct.fPart1 == null) construct.fPart1 = this;
		else construct.fPart2 = this;
	}
	
	/**
	 * 
	 * @param construct
	 * @return True if the instance is before the construct given in 
	 * parameter, false otherwise. 
	 */
	public boolean isBefore (AdaConstruct construct) {
		if (fLineBegin < construct.fLineBegin) return true;
		if (fLineBegin == construct.fLineBegin
				&& fLocation.getColumn().getColumnIndex() < construct.fLocation
						.getColumn().getColumnIndex())
			return true;
		
		return false;
	}
	
	/**
	 * @return True if the construct is the main one, compared to its other 
	 * parts. 
	 */
	public boolean isMainConstruct() {
		if (fPart1 != null) {
			if (!isBefore (fPart1)) return false;
		}
		
		if (fPart2 != null) {
			if (!isBefore (fPart2)) return false;
		}
		
		return true;
	}
	
	/**
	 * 
	 * @return The main part of this construct.
	 */
	public AdaConstruct getMainConstruct() {
		if (isMainConstruct()) return this;
		if (fPart1.isMainConstruct()) return fPart1;
		return fPart2;
	}
	
	/**
	 * @return The next part of this construct if multiple one, the first one
	 * if the last has already been reached, the current one if no other 
	 * part.
	 */
	public AdaConstruct getNextPart() {
		if (fPart1 == null) return this;
		
		if (fPart2 != null) {
			if (isBefore (fPart1)) return fPart1;
			if (isBefore (fPart2)) return fPart2;
		}
		
		return fPart1;
	}

	/**
	 * @return the numbers of others parts detected in the file of this 
	 * construct.
	 */
	public int getNbParts() {
		if (fPart1 == null) return 0;
		if (fPart2 == null) return 1;
		return 2;
	}
	
	/**
	 * 
	 * @return another part of the construct, if there are at least two.
	 */
	public AdaConstruct getPart1() {
		return fPart1;
	}
	
	/**
	 * 
	 * @return another part of the construct, if there are tree.
	 */
	public AdaConstruct getPart2() {
		return fPart2;
	}
	
	public boolean getIsDeclaration() {
		return fIsDeclaration;
	}
	
	static private Simple_Construct_Information toSimpleConstructInformation(
			Construct_Information content) {

		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			Simple_Construct_Information simple = new Simple_Construct_Information();

			content.To_Simple_Construct_Information(simple, true);

			return simple;
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	public Object[] getChildren(Object o) {
		return getChildren (null);
	}

	public ImageDescriptor getImageDescriptor(Object object) {
		return getImageDescriptor();
	}

	public String getLabel(Object o) {
		return getName();
	}

	public Object getParent(Object o) {
		return getParent(null);
	}	
}