/*******************************************************************************
 * Copyright (C) 2005-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal.analyzer;

import java.util.Comparator;
import java.util.LinkedList;
import java.util.TreeSet;

import com.adacore.gnatbench.core.internal.adaeditor.AdaDocumentBuffer;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;

import com.adacore.gnatbench.library.Standard.AdaString;

import com.adacore.gnatbench.library.Language.Construct_Information;
import com.adacore.gnatbench.library.Language.Construct_List;
import com.adacore.gnatbench.library.Language.Ada.Ada_Package;

/**
 * This class is able to analyze a document buffer and return a tree of
 * constructs. Events can be registered in order to be aware of changes in the
 * structure. Between two updates, most of the construct should remain exactly
 * the same, so it is possible to make a reference to a construct and to
 * follow its changes. This feature is very important for features like outline
 * bloc folding, while a state (expanded/collasped) is associated with each
 * construct and we don't want them to be lost at each update.
 *
 */
public class AdaAnalyzer {

	private AdaDocumentBuffer fBuffer = null;
	private AdaConstruct[] fData = new AdaConstruct[]{};
	private TreeSet<IAdaAnalyzerListener> fListeners = new TreeSet<IAdaAnalyzerListener>(new Comparator <IAdaAnalyzerListener> () {

		public int compare(IAdaAnalyzerListener arg0, IAdaAnalyzerListener arg1) {
			if (arg1 == arg0) {
				return 0;
			} else if (arg1.hashCode() > arg1.hashCode()) {
				return 1;
			} else {
				return -1;
			}
		}});
	private LinkedList<Operation> fElementsModified = new LinkedList<Operation>();

	/**
	 * This class is used to store operation made during the merge processing.
	 * It will be used later to fire proper events.
	 *
	 */
	private class Operation {
		static final int ADD_OP = 0;
		static final int REMOVE_OP = 1;
		static final int MODIFY_OP = 2;

		int fOp;
		AdaConstruct fConstruct;

		public Operation(int op, AdaConstruct construct) {
			fOp = op;
			fConstruct = construct;
		}

	}

	/**
	 * This class is used to transform the list of elements given by GPS into
	 * a tree. A stack is used to perform this operation.
	 */
	private class ElementStack {
		LinkedList <AdaConstruct> fNodes;

		public ElementStack() {
			fNodes = new LinkedList<AdaConstruct>();
		}

		/**
		 * Adds a node in the stack. All tops elements of the stack included
		 * in this node will be removed and added as children.
		 *
		 * @param node
		 */
		public void add(AdaConstruct node) {
			while (fNodes.size() > 0) {
				AdaConstruct head = fNodes.getLast();

				if (node.include(head)) {
					node.add(head);
					fNodes.removeLast();
				} else {
					break;
				}
			}

			node.invert();
			fNodes.add(node);
		}

		/**
		 * Returns the roots elements of the tree. There can be more than one
		 * root tree detected by the stack, for example, with clauses are not
		 * part of any tree.
		 *
		 * @return
		 */
		public AdaConstruct[] getRootArray() {
			return fNodes.toArray(new AdaConstruct [fNodes.size()]);
		}
	}

	/**
	 * Set the document buffer analyzed.
	 *
	 * @param buffer
	 */
	public void setDocumentBuffer(AdaDocumentBuffer buffer) {
		fBuffer = buffer;
	}

	/**
	 * Adds a listener to analyzer events.
	 *
	 * @param listener
	 */
	public void addAdaAnalyzerListener(IAdaAnalyzerListener listener) {
		fListeners.add(listener);
	}

	/**
	 * Removes a listener to analyzer events.
	 *
	 * @param listener
	 */
	public void removeAdaAnalyzerListener(IAdaAnalyzerListener listener) {
		fListeners.remove(listener);
	}

	/**
	 * Fires the events stored during the merge operation, and clear the event
	 * list.
	 */
	public void fireElementsModifs() {
		/* It is very important that the modifications are send after having
		 * updated the internal structure, that's why we store them in lists.
		 * Do not fire any event during the merge. Listerners must be allowed
		 * to analyze the whole structure as soon as they receive an event.
		 *
		 * The filters are used to call add / removed / modified events only on
		 * those that whants the listener to receive. The whole parent tree has
		 * to be asked to determine if there is a filtering node under the
		 * current one. To improve performaces, we put flags on the
		 * AdaConstructs themslevses to remember which nodes are filtered. Those
		 * flags are removed as soon as possible, and any new listener is
		 * supposed to get a clean tree.
		 */

		LinkedList<IAdaAnalyzerListener> listenersCopy = new LinkedList<IAdaAnalyzerListener>(
				fListeners);

		for (IAdaAnalyzerListener listener : listenersCopy) {
			AdaConstructFilter filter = listener.getFilter();

			filter.initialize();

			listener.startChangesBatch();

			for (Operation operation : fElementsModified) {
				AdaConstruct construct = operation.fConstruct;

				if (!filter.passFilter(construct)) continue;

				switch (operation.fOp) {
					case Operation.ADD_OP :
						listener.elementAdded(construct);
						break;
					case Operation.REMOVE_OP :
						listener.elementRemoved(construct);
						break;
					case Operation.MODIFY_OP :
						listener.elementModified(construct);
						break;
				}
			}

			listener.finishedChangesBatch();
		}

		fElementsModified.clear();
	}

	/**
	 * Analyzes the current state of the buffer. Events will be fired according
	 * to changes discovered in the tree.
	 *
	 * @return the new version of the tree.
	 */
	public AdaConstruct[] analyze() {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			if (fBuffer == null) return new AdaConstruct [0];

			Construct_List list = new Construct_List ();

			Ada_Package.Ada_Lang().Parse_Constructs(new AdaString (fBuffer.getContent()), list);

			ElementStack stack = new ElementStack();

			if (list == null) {
				fData = new AdaConstruct[]{};
				return fData;
			}

			Construct_Information current = list.First();

			while (current != null) {
				stack.add(new AdaConstruct(current, fBuffer));
				current = current.Next();
			}

			fData = mergeTrees(fData, stack.getRootArray());

			for (int i = 0; i < fData.length; ++i) {
				fData[i].groupChildren();
			}

			fireElementsModifs();

			return fData;
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	/**
	 * Merge the new tree into the old tree, and return the result of the
	 * merge. Old elements are saved whenever possible. This function also
	 * adds entries into the event list, to be called after the final return
	 * of the recursive call.
	 *
	 * @param oldTree
	 * @param newTree
	 * @return
	 */
	private AdaConstruct[] mergeTrees(AdaConstruct[] oldTree,
			AdaConstruct[] newTree) {
		int indexOld = 0, indexNew = 0;
		LinkedList<AdaConstruct> result = new LinkedList<AdaConstruct>();

		while (true) {
			if (indexOld >= oldTree.length && indexNew >= newTree.length)
				break;

			// CASE 1: everything has been tested from fData, just add the remaining from
			// tree
			if (indexOld >= oldTree.length) {
				while (indexNew < newTree.length) {
					result.add(newTree[indexNew]);
					fElementsModified.add(new Operation(Operation.ADD_OP,
							newTree[indexNew]));
					indexNew++;
				}
			}
			// CASE 2: everything has been tested from tree, just remove the remaining from
			// fData, which means finish the process
			else if (indexNew >= newTree.length) {
				while (indexOld < oldTree.length) {
					fElementsModified.add(new Operation(Operation.REMOVE_OP,
							oldTree[indexOld]));
					indexOld++;
				}
				break;
			}
			// CASE 3: object are equals, then try to merge the sub trees
			else if (oldTree[indexOld].getName().equals(
					newTree[indexNew].getName())) {
				result.add(oldTree[indexOld]);

				// WARNING ! The merge has to be done before the addition to
				// the list (see implementation of AdaBlocFolding).
				oldTree[indexOld]
						.setChildren(mergeTrees(oldTree[indexOld]
								.getChildren(null), newTree[indexNew]
								.getChildren(null)));

				if (oldTree[indexOld].needsAdaptation(newTree[indexNew])) {
					oldTree[indexOld].adapt(newTree[indexNew]);
					fElementsModified.add(new Operation(Operation.MODIFY_OP,
							oldTree[indexOld]));
				}

				indexNew++;
				indexOld++;
			}
			// CASE 4: object are differents, but fData is equal to the next tree, so
			// just add the current tree
			else if (indexNew + 1 < newTree.length
					&& oldTree[indexOld].getName().equals(
							newTree[indexNew + 1].getName())) {
				result.add(newTree[indexNew]);
				fElementsModified.add(new Operation(Operation.ADD_OP,
						newTree[indexNew]));
				indexNew++;
			}
			// CASE 5: object are differents, and fData is not equal to the next tree, so
			// remove the current fData
			else {
				fElementsModified.add(new Operation(Operation.REMOVE_OP,
						oldTree[indexOld]));
				indexOld++;
			}
		}

		return result.toArray(new AdaConstruct [result.size ()]);
	}

	/**
	 * return the roots elements according to a filter.
	 *
	 * @param filter filter the elements, return all elements if null.
	 * @return The roots elements of the semantics. Typically, a bench of
	 * with/use clauses and, finally, the main unit.
	 */
	public AdaConstruct[] getRoots(IAdaConstructFilterProvider constructFilter) {
		if (constructFilter == null)
			return fData;
		else {
			LinkedList<AdaConstruct> root = new LinkedList<AdaConstruct>();

			for (int i = 0; i < fData.length; ++i) root.add(fData[i]);

			LinkedList <AdaConstruct>result = constructFilter.getFilter()
					.getFilteredChildren(root);

			return result.toArray(new AdaConstruct [result.size()]);
		}

	}

	/**
	 * Returns the depper construct that encloses the location given in
	 * parameter, null if none. Wrapps internalGetConstructAt.
	 *
	 * @param location The location of the construct, filename will be ignored
	 * @param location filter
	 * @return
	 */
	public AdaConstruct getConstructAt(GeneralizedLocation location,
			IAdaConstructFilterProvider filter) {

		return internalGetConstructAt(location, filter, getRoots(filter));
	}

	/**
	 * Returns the depper construct that encloses the location given in
	 * parameter, null if none.
	 *
	 * @param location The location of the construct, filename will be ignored
	 * @param location filter
	 * @return
	 */
	private AdaConstruct internalGetConstructAt(GeneralizedLocation location,
			IAdaConstructFilterProvider filter, AdaConstruct[] elements) {

		for (int i = 0; i < elements.length; i++) {
			if (elements[i].include(location)) {
				AdaConstruct subElement = internalGetConstructAt
						(location, filter, elements[i].getChildren(filter));

				if (subElement == null) {
					return elements[i];
				} else {
					return subElement;
				}
			}
		}

		return null;
	}

	/**
	 * return the document buffer associated to this analyzer
	 *
	 * @return
	 */
	public AdaDocumentBuffer getDocumentBuffer () {
		return fBuffer;
	}
}
