/*******************************************************************************
 * Copyright (C) 2005-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/
package com.adacore.gnatbench.core.internal.adaeditor;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.DocumentEvent;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IDocumentExtension3;
import org.eclipse.jface.text.IDocumentListener;
import org.eclipse.jface.text.IDocumentPartitioner;
import org.eclipse.jface.text.rules.FastPartitioner;

import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.analyzer.IAdaDocument;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.analyzer.AdaAnalyzer;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstruct;
import com.adacore.gnatbench.core.internal.analyzer.AdaEntityReference;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedFile;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedLocation;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.ajis.IProxy;
import com.adacore.ajis.IProxy.Owner;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.Standard.AdaString;

/**
 * This class is supposed to improve the capabilities of the IDocument class. It
 * adds functionalities, and improve the performances. The source document not be
 * used except if it has to be.
 *
 * It can be created without any editor, by GNATbenchCorePlugin, if the content of the
 * file has to be analyzed.
 *
 * Note: Even if it hold the analyzer, the AdaDocumentBuffer is not
 * responsible for updating semantic information. Constructs are *only*
 * created during the construction of the instance.
 *
 * TODO: add here document listeners !!!
 *
 */
public class AdaDocumentBuffer implements IDocumentListener, IAdaDocument {

	private IDocument fDocument;
	private String fContent;
	private GeneralizedFile fFile;
	private AdaAnalyzer fAnalyzer;
	private AdaString fAdaString;
	private String fJavaString;

	private IDocumentPartitioner fPartitioner;
	private GNATProjectRegistry fReg;

	/**
	 * Create a document buffer based on a document.
	 *
	 * @param document
	 * @param begin This parameter is only used to identify the file. Length,
	 * column and line are not used.
	 */
	public AdaDocumentBuffer(GeneralizedFile file) {
		fDocument = GNATbenchCorePlugin.getDefault().getDocumentProvider()
				.getDocument(file.getInput());

		if (fDocument instanceof IDocumentExtension3) {
			IDocumentExtension3 extension3 = (IDocumentExtension3) fDocument;

			fPartitioner = new FastPartitioner(
					GNATbenchCorePlugin.getDefault()
					.getPartitionScanner(),
					AdaPartitionScanner.ADA_PARTITION_TYPES);
			extension3.setDocumentPartitioner(
					GNATbenchCorePlugin.ADA_PARTITIONING, fPartitioner);
			fPartitioner.connect(fDocument);
		}

		fDocument.addDocumentListener(this);
		fFile = file;
		fContent = fDocument.get();

		fAnalyzer = new AdaAnalyzer();
		fAnalyzer.setDocumentBuffer(this);
		fAnalyzer.analyze();

		fReg = (GNATProjectRegistry) GNATbenchSession.getDefault()
				.getOrLoadRegistry(file.getProject());
	}

	/**
	 * This procedure is quite more efficient than the IDocument.get() one,
	 * because it does not create a new string each time it is called, but
	 * only each time modifications has been made.
	 *
	 * @return
	 */
	public String getContent() {
		return fContent;
	}

	/**
	 * Returns the the text for the given location. If the length is 0, then
	 * the result is empty. Return null if there's no content at the specified
	 * location.
	 *
	 * @param location
	 * @return
	 */
	public String getContent(GeneralizedLocation location) {
		if (location.getLength() == 0)
			return "";

		int offset;

		try {
			offset = location.getOffset();
		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return "";
		}

		return fContent.substring(offset, offset + location.getLength());
	}

	/**
	 * Returns the first position of the entity which has one of its
	 * characters at offset.
	 *
	 * @param offset
	 * @return
	 */
	private int getBeginEntity(int offset) {
		int offsetBegin = offset;

		if (offsetBegin >= fContent.length()) {
			offsetBegin = fContent.length() - 1;
		}

		while (offsetBegin > 0
				&& isSimpleAdaIdLetter(fContent.charAt(offsetBegin))) {
			offsetBegin--;
		}

		// TODO : This very basic algorithm do not care about the cases where
		// the entity is on the very beginning / end of the string.
		offsetBegin++;

		if (offsetBegin > offset)
			offsetBegin = offset;

		return offsetBegin;
	}

	/**
	 * Returns the last position of the entity which has one of its
	 * characters at offset.
	 *
	 * @param offset
	 * @return
	 */
	private int getEndEntity(int offset) {
		int offsetEnd = offset;

		while (offsetEnd < fContent.length() &&
				isSimpleAdaIdLetter(fContent.charAt(offsetEnd))) {
			offsetEnd++;
		}

		// TODO : This very basic algorithm do not care about the cases where the
		// entity is on the very beginning / end of the string.
		offsetEnd--;

		if (offsetEnd < offset)
			offsetEnd = offset;

		return offsetEnd;
	}

	/**
	 * Returns the entity reference around the given offset, null if there's
	 * no such entity
	 *
	 * @param offset
	 * @return
	 */
	public AdaEntityReference getEntity(int offset) {
		if (GNATbenchSession.getDefault().getOrLoadRegistry(fFile.getProject()) == null) {
			return null;
		}

		GeneralizedLocation newLocation;
		int offsetBegin = getBeginEntity(offset);

		try {
			int line = fDocument.getLineOfOffset(offsetBegin);
			newLocation = GeneralizedLocation.fromAbsoluteLocation
				(fFile, line + 1, offsetBegin);

			newLocation.setLength (getEndEntity(offset) - offsetBegin + 1);

			if (exists(newLocation)) {
				return new AdaEntityReference(this, newLocation, offsetBegin);
			} else {
				return null;
			}
		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		return null;
	}

	/**
	 * Returns the file managed by this buffer
	 *
	 * @return
	 */
	public GeneralizedFile getFile() {
		return fFile;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.text.IDocumentListener#documentAboutToBeChanged(org.eclipse.jface.text.DocumentEvent)
	 */
	public void documentAboutToBeChanged(DocumentEvent event) {
		// TODO Auto-generated method stub
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.text.IDocumentListener#documentChanged(org.eclipse.jface.text.DocumentEvent)
	 */
	public void documentChanged(DocumentEvent event) {
		fContent = event.getDocument().get();
		releaseInternalString ();
	}

	private static class ReleaseJob extends Job {

		private AdaString fAdaString;

		public ReleaseJob(String name, AdaString adaString) {
			super(name);

			fAdaString = adaString;
		}

		@Override
		protected IStatus run(IProgressMonitor monitor) {
			//  At this stage, there should not be any further references
			//  to the object from the native environment - so we're
			//  switching it back to the owned state so that it gets freed
			//  when there is no more Java references either.

			((IProxy) fAdaString).setOwner(Owner.PROXY);

			//  Reseting the reference in order to run the garbage collector
			//  on this as soon as possible.
			fAdaString = null;

			return GNATbenchCorePlugin.OK_STATUS;
		}
	}

	private void releaseInternalString () {

		if (fAdaString != null) {
			final AdaString stringToFree = fAdaString;


			if (fReg != null) {
				fReg.scheduleJob(new ReleaseJob ("release memory", fAdaString));
			} else {
				((IProxy) stringToFree).setOwner(Owner.PROXY);
			}
		}

		fAdaString = null;
		fJavaString = null;

	}

	/**
	 * Returns the closest location of the entity based on the initialLocation.
	 * If the location is already at the proper place, an identical position
	 * will be returned.
	 *
	 * @param initialLocation
	 * @param entityName
	 * @return
	 */
	public GeneralizedLocation getClosestLocation(
			GeneralizedLocation initialLocation, String entityName) {

		int offset;

		try {
			offset = initialLocation.getOffset();
		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);

			return initialLocation;
		}

		int closest1 = fContent.indexOf(entityName, offset);
		int closest2 = fContent.lastIndexOf(entityName, offset);

		if (closest1 == -1 && closest2 == -1) {
			return initialLocation;
		}
		else if (closest1 == -1) {
			offset = closest2;
		} else if (closest2 == -1) {
			offset = closest1;
		} else if (closest1 != -1 && closest2 != -1) {
			if (Math.abs (offset - closest1) <= Math.abs (offset - closest2)) {
				offset = closest1;
			} else {
				offset = closest2;
			}
		}

		int lineNumber = 0;

		try {
			lineNumber = fDocument.getLineOfOffset(offset);
		} catch (BadLocationException e1) {
			GNATbenchCorePlugin.getDefault().logError(null, e1);
		}

		GeneralizedLocation result = GeneralizedLocation
				.fromAbsoluteLocation(initialLocation.getFile(), lineNumber + 1,
						offset);

		result.setLength(entityName.length());

		return result;
	}

	public IDocument getDocument() {
		return fDocument;
	}

	/**
	 * @param construct
	 * @return
	 */
	public AdaEntityReference getEntity(AdaConstruct construct) {
		if (GNATbenchSession.getDefault().getOrLoadRegistry(fFile.getProject()) == null) {
			return null;
		}

		GeneralizedLocation location = GeneralizedLocation
				.fromAbsoluteLocation(construct.getLocation().getFile(),
						construct.getLineEntity() + 1, construct
								.getOffsetEntity());
		location.setLength(construct.getName().length());

		return new AdaEntityReference(this, location, construct
				.getOffsetBegin());
	}

	/**
	 * Returns the offset from the beginnig of the buffer for the given
	 * couple line / columns, assuming that they are computed in eclipse style,
	 * e.g. from 0.
	 *
	 * @param lineNumber
	 * @param columnNumber
	 * @return
	 */
	public int getOffset (int lineNumber, int columnNumber) {
		try {
			return fDocument.getLineOffset(lineNumber) + columnNumber;
		} catch (BadLocationException e) {
			return 0;
		}
	}

	public AdaAnalyzer getAnalyzer() {
		return fAnalyzer;
	}

	/**
	 * Return the entity located at the location given in parameter, null if
	 * there's no such entity.
	 *
	 * @param location
	 * @return
	 * @throws BadLocationException
	 */
	public AdaEntityReference getEntity(GeneralizedLocation location)
			throws BadLocationException {
		return getEntity(location.getOffset());
	}

	public int getFirstNonWhite(int offset) {

		while (offset < fContent.length()
				&& Character.isWhitespace(fContent.charAt(offset)))
			offset++;

		if (offset >= fContent.length()) return -1;
		else return offset;
	}

	/**
	 * Return and AdaString holding the current text. The returned proxy is
	 * marked as not owned, as it may need to be referenced by native code
	 * during short period of time. However, such code should create an
	 * explicit reference to the returned value in order to avoid garbage
	 * collection while used in the native code - as soon as the file is
	 * updated, a kernel job will be launched whose purpose is to switch back
	 * the owning state by the proxy to true.
	 *
	 * @return
	 */
	public AdaString getAdaString () {
		if (fAdaString == null) {

			LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();
			try {
				fJavaString = getContent();
				fAdaString = new AdaString (fJavaString);
			} finally {
				LibrarySemaphore.stopGPSWork(libMonitor);
			}

			//  TODO: We could just let a java reference lying around as long
			//  as needed instead of switching back and forth the owning state.
			((IProxy) fAdaString).setOwner(Owner.NATIVE);
		}

		return fAdaString;
	}

	public String getJavaString () {
		if (fJavaString == null) {
			getAdaString();
			//  This forces to update the internal java representation of the
			//  buffer
		}

		return fJavaString;
	}

	/**
	 * Return true if the location pointed by AdaLocation exists in the
	 * buffer.
	 *
	 * @param loc
	 * @return
	 */
	public boolean exists (GeneralizedLocation loc) {
		try {
			if (fDocument.getNumberOfLines() < loc.getLine()) {
				return false;
			}

			return fContent.length() >= loc.getOffset() + loc.getLength();
		} catch (BadLocationException e) {
			return false;
		}
	}

	public void finalize () {
		dispose();
	}

	public void dispose () {
		if (fPartitioner != null) {
			fPartitioner.disconnect();
		}

		if (fDocument != null) {
			fDocument.removeDocumentListener(this);
		}

		fPartitioner = null;
		fDocument = null;

		releaseInternalString();

		fContent = "";

	}

	protected boolean isSimpleAdaIdLetter(char c) {
		return 	(c >= '0' && c <= '9')
				|| (c >= 'a' && c <= 'z')
				|| (c >= 'A' && c <= 'Z')
				|| (c == '_');
	}

}
