/*******************************************************************************
 * Copyright (C) 2006-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal;

import java.util.Arrays;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.runtime.CoreException;

import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.internal.builder.GNATbenchProjectNature;
import com.adacore.gnatbench.core.internal.gpswrappers.GPRProject;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistryFactory;
import com.adacore.gnatbench.core.internal.utils.ProjectUtils;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.Language.Language_Root;
import com.adacore.gnatbench.library.Language.Ada.Ada_Package;
import com.adacore.gnatbench.library.GNATCOLL.Filesystem.Filesystem_String;
import com.adacore.gnatbench.library.GNATCOLL.VFS.VFS_Package;
import com.adacore.gnatbench.library.GNATCOLL.VFS.Virtual_File;

public class GPRResourceSpy implements IResourceChangeListener, IResourceDeltaVisitor {

	// A flag indicating whether processing has determined that recomputing the projects
	// is necessary.  We use this approach to minimize the number of times recompute()
	// is called, as opposed to calling it in the places where the flag is set to true,
	// due to the nature of the visitor behavior.
	protected boolean shouldRecomputeProjects;

	// A flag to indicate if we should update the scenario variable settings in the
	// gnatbench project file (.gb_project).
	protected boolean shouldSaveScenarioVars;
	// The project in question when saving scenario variables.
	protected IProject project = null;

	static private boolean fActivated = true;

	private GNATProjectRegistry fReg;

	/**
	 * Set whether the GPRSpy should react to resource changes or not. Changes
	 * caught during a deactivated period will be lost by the spy.
	 *
	 * @param activated
	 */
	static public synchronized void setActivated (boolean activated) {
		fActivated = activated;
	}

	/**
	 * Return wether the GPRSpy should react to resource changes or not
	 * @return
	 */
	static public synchronized boolean isActivated () {
		return fActivated;
	}

	public void resourceChanged(IResourceChangeEvent event) {
		if (!isActivated()) {
			return;
		}

		shouldRecomputeProjects = false;
		shouldSaveScenarioVars = false;

		switch (event.getType()) {
			case IResourceChangeEvent.POST_CHANGE:
				// New GNATbench projects are added to the list of known projects for GPS,
				// and Ada source file addition/deletion causes a recompute(). Project
				// deletion is handled by the PRE_DELETE case above.  Changes to GPR files
				// are also recognized and cause a recompute().

				try {
					event.getDelta().accept(this);
				} catch (Exception e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				} // try

				break;

			case IResourceChangeEvent.PRE_DELETE:
			case IResourceChangeEvent.PRE_CLOSE:
				if (event.getResource() instanceof IProject) {
					IProject project = (IProject) event.getResource();

					try {
						if (project.isOpen()
							&& project
									.hasNature(GNATbenchProjectNature.NATURE_ID)) {
							if (GNATbenchProjectProperties
								.getPropertiesFor(project).isRootProject()) {

								//  Remove the registry only if the removed
								//  project is a root one.

								fReg = (GNATProjectRegistry) GNATbenchSession
									.getDefault().getOrLoadRegistry(
										event.getResource().getProject());

								shouldRecomputeProjects = false;
								GPRProject.setActive(project, false);

								((GNATProjectRegistryFactory) GNATbenchSession
									.getDefault().getProjectRegistryFactory())
									.unloadRegistry(fReg);
							}
						}
					} catch (CoreException e) {
						GNATbenchCorePlugin.getDefault().logError(null, e);
					}
				} else {
					try {
						event.getDelta().accept(this);
					} catch (Exception e) {
						GNATbenchCorePlugin.getDefault().logError(null, e);
					} // try
				}

		} // switch

		if (shouldRecomputeProjects) {
			fReg.recompute();

			shouldRecomputeProjects = false;

			if (fReg.isValid()) {
				if (shouldSaveScenarioVars) {
					project = null;
					shouldSaveScenarioVars = false;
				} // if
			}
		} // if
	} // resourceChanged


	public boolean visit(IResourceDelta delta) throws CoreException {
		int flags = delta.getFlags();
		int kind = delta.getKind();

		if (((kind & (IResourceDelta.ADDED | IResourceDelta.REMOVED)) == 0)
				&& ((flags & (~IResourceDelta.MARKERS)) == 0)) {
			// If the delta is neither ADDED or REMOVED, and if there's no other
			// flag than MARKER, we don't visit this resource
			return true;
		}

		final IResource resource = delta.getResource();

		IProject prj = null;

		if (resource instanceof IProject) {
			prj = (IProject) resource;
		} else {
			prj = resource.getProject();
		}

		if (prj == null || !prj.exists() || !prj.isAccessible()) {
			return false;
		}

		fReg = (GNATProjectRegistry) GNATbenchSession.getDefault()
				.getOrLoadRegistry(prj);

		if (prj != null
				&& (!prj.isOpen() || !prj
						.hasNature(GNATbenchProjectNature.NATURE_ID))) {

			return false;
		}

		switch (kind) {
			case IResourceDelta.ADDED:

				switch (resource.getType()) {

					case IResource.FILE:
						handleFileExistence(resource);
						break;

					case IResource.PROJECT:
						handleNewProject(resource);
						break;

					default:
						break;
				} // switch
				break;

			case IResourceDelta.REMOVED:

				switch (resource.getType()) {

					case IResource.FILE:
						handleFileExistence(resource);
						break;

					default:
						break;
				} // switch
				break;

			case IResourceDelta.CHANGED:
				int type = resource.getType();

				switch (type) {

					case IResource.FILE:
						handleFileChange(resource);
						break;

					case IResource.PROJECT:
						if ((delta.getFlags() & IResourceDelta.OPEN) != 0) {
							if (((IProject)resource).isOpen()) {
								handleNewProject(resource);
							}
						} // if
						break;

					default:
						break;
				} // switch
				break;

			default:
				break;
		} // switch

		return true;
	} // visit


	protected void handleFileExistence(final IResource resource) {
		final String extension = resource.getFileExtension();

		if (extension == null) {
			// Don't do anything
		} else if (extension.equalsIgnoreCase("gpr")) {
			shouldRecomputeProjects = true;
		} else {
			final String name = resource.getName();
			if (ProjectUtils.isAdaSourceFileName(name, resource.getProject())) {
				// ignore binder-generated Ada files
				if (!name.startsWith("b~")) {
					shouldRecomputeProjects = true;
				} // if
			} // if
		} // if
	} // handleFileExistence


	protected void handleFileChange(final IResource resource) {
		if (resource.getFileExtension() == null) {
			// Don't do anything
		} else if (resource.getFileExtension().equalsIgnoreCase("gpr")) {
			project = resource.getProject();
			shouldSaveScenarioVars = true;
			shouldRecomputeProjects = true;

		} else if (resource instanceof IFile) {
			if (fReg != null) {
				LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();
				try {
					Virtual_File file = VFS_Package
							.Create(new Filesystem_String(resource
									.getLocation().toOSString()));
					Language_Root lang = fReg.fLanguageHandler
					.Get_Language_From_File_Language(file, false);

					if (Arrays.equals(lang.getAccess(), Ada_Package.Ada_Lang()
							.getAccess())) {
						fReg.reloadConstructFile(file);
					}
				} finally {
					LibrarySemaphore.stopGPSWork(libMonitor);
				}
			}
		} // if
	} // handleFileChange


	protected void handleNewProject(IResource resource) {
		IProject project = (IProject) resource;

		if (!project.isOpen()) return;

		try {
			if (project.hasNature(GNATbenchProjectNature.NATURE_ID)) {
				GNATbenchProjectProperties props = GNATbenchProjectProperties
						.getPropertiesFor(project);

				if (props.isRootProject()) {
					GPRProject.setActive(project, true);
					shouldRecomputeProjects = true;
				}
			}
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} // if
	} // handleNewProject

} // GPRResourceSpy
