/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal;

import java.util.LinkedList;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;

import com.adacore.gnatbench.core.internal.gpswrappers.GPRProject;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.core.projects.IGNATProjectMessage;

public class GPRMessage implements IGNATProjectMessage {

	static private final String MESSAGE_WITH_COLUMNS = "(.*?):([0-9]+)(:([0-9]+))?: (warning: )?(.*)";
	static private final String MESSAGE_WITH_NO_COLUMNS = "((\\b[a-z]:)?[^:*?\"<>|\r\n]*):(.*): (warning: )?([^:]+)$";

    static private final Pattern patternColumns =
    	Pattern.compile(MESSAGE_WITH_COLUMNS, Pattern.CASE_INSENSITIVE);

    static private final Pattern patternNoColumns =
    	Pattern.compile(MESSAGE_WITH_NO_COLUMNS, Pattern.CASE_INSENSITIVE);

	private IPath fFilePath = null;
	private int fLine = 1;
	private int fCol = 1;
	private boolean fContainsWarning;
	private String fCaption = null;

	private GNATProjectRegistry fRegistry;

	public GPRMessage(final String message, final GNATProjectRegistry registry) {
		String mess = message.replaceAll("\n", "").replaceAll("\r", "");
		Matcher recognizer = patternColumns.matcher(mess);
		fRegistry = registry;
		String fileName = null;

		if (recognizer.matches()) {
			IPath path = new Path(recognizer.group(1));

			if (path.getFileExtension() == null) {
				fileName = recognizer.group(1) + ".gpr";
			} else {
				fileName = recognizer.group(1);
			} // if

			fLine = Integer.parseInt(recognizer.group(2));
			fCol = Integer.parseInt(recognizer.group(4));
			fContainsWarning = recognizer.group(5) != null;
			fCaption = recognizer.group(6).trim();
		} else {
			recognizer = patternNoColumns.matcher(mess);

			if (recognizer.matches()) {
				IPath path = new Path(recognizer.group(1));

				if (path.getFileExtension() == null) {
					fileName = recognizer.group(1) + ".gpr";
				} else {
					fileName = recognizer.group(1);
				} // if

				fCaption = recognizer.group(5).trim();
				fContainsWarning = recognizer.group(3) != null;
			} else {
				return;
			}
		}

		fFilePath = new Path (fileName);

		if (!fFilePath.isAbsolute()) {
			GPRProject prj = fRegistry.getProjectFromLocation(fFilePath);

			if (prj != null) {
				fFilePath = URIUtil.toPath(prj.getProjectFile().toURI());
			} else {
				fFilePath = null;
			}
		}
	} // GPRMessage


	/**
	 * Create a marker for the specific message, and return the impacted
	 * resources.
	 *
	 * @return
	 */
	public LinkedList <IResource> createMarker() {
		LinkedList <IResource> result = new LinkedList <IResource> ();

		if (fCaption == null || fFilePath == null) {
			//  In this case, the message has not been parsed correctly, no
			//  markers have to be added.
			return result;
		}

		try {
			IResource [] files = ResourcesPlugin.getWorkspace().getRoot()
					.findFilesForLocation(fFilePath);
			IMarker marker;

			if (files == null) {
				return result;
			} // if

			for (IResource res : files) {
				marker = res
						.createMarker(GNATbenchCorePlugin.GPR_ERROR_MARKER);
				marker.setAttribute(IMarker.MESSAGE, fCaption);

				marker.setAttribute(IMarker.LINE_NUMBER, fLine);

				marker.setAttribute(IMarker.TRANSIENT, true);

				if (fContainsWarning) {
					marker.setAttribute(IMarker.SEVERITY,
							IMarker.SEVERITY_WARNING);
				} else {
					marker.setAttribute(IMarker.SEVERITY,
							IMarker.SEVERITY_ERROR);
				} // if

				result.add(res);
			}
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} // try

		return result;
	} // createMarker

	public void addErrorLog () {
		if (fContainsWarning) {
			GNATbenchCorePlugin.getDefault().getLog().log(
					new Status(IStatus.WARNING, GNATbenchCorePlugin.getId(),
							IStatus.OK, fCaption, null));
		} else {
			GNATbenchCorePlugin.getDefault().getLog().log(
					new Status(IStatus.ERROR, GNATbenchCorePlugin.getId(),
							IStatus.OK, fCaption, null));
		}
	}


	public Kind getKind() {
		if (fContainsWarning) {
			return Kind.WARNING;
		} else {
			return Kind.ERROR;
		}
	}


	public String getMessage() {
		return fFilePath.lastSegment() + ":" + fLine + ":" + fCol + ":"
				+ fCaption;
	}

} // GPRMessage
