/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal;

import java.io.IOException;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.net.URI;
import java.util.TreeMap;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.WorkspaceJob;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.core.runtime.jobs.Job;
import org.w3c.dom.Node;

public class GNATbenchProjectProperties {

	public static String IS_ROOT_PROJECT_PROPERTY =
		"com.adacore.gnatbench.isRootProject";

	public static String GPR_PATH_PROPERTY =
		"com.adacore.gnatbench.gprPath";

	private static TreeMap<URI, GNATbenchProjectProperties> fgPropertiesTable =
		new TreeMap<URI, GNATbenchProjectProperties>();

	private static DocumentBuilderFactory fgDocFactory;
	private static XPathFactory fgXPathFactory;

	private static DocumentBuilder fgDocBuilder;

	private org.w3c.dom.Document fDocument;

	private IProject fPrj;

	private GNATbenchProjectProperties (IProject prj) {
		fPrj = prj;

		loadProjectFile();
	}

	/**
	 * Get the project properties associated to this project - create one if
	 * it hasn't been created already.
	 *
	 * @param prj
	 * @return
	 */
	public static GNATbenchProjectProperties getPropertiesFor(
			IProject prj) {
		URI uri = URIUtil.toURI(prj.getLocation());

		GNATbenchProjectProperties props;

		synchronized (fgPropertiesTable) {
			if (!fgPropertiesTable.containsKey(uri)) {
				props = new GNATbenchProjectProperties(prj);

				fgPropertiesTable.put(uri, props);
			}

			props = fgPropertiesTable.get(uri);
		}

		return props;
	}

	/**
	 * If there's a project in the properties matching the gpr path given in
	 * parameter, then this project is returned by this function, null
	 * otherwise.
	 * @param path
	 * @return
	 */
	public static synchronized IProject getProjectForURI(URI uri) {
		for (Map.Entry <URI, GNATbenchProjectProperties> e : fgPropertiesTable.entrySet()) {
			String gprPath = e.getValue().getGPROSPath();

			if (gprPath != null) {
				URI gprURI = URIUtil.toURI(gprPath);

				if (gprURI != null && gprURI.equals(uri)) {
					return e.getValue().fPrj;
				}
			}
		}

		return null;
	}

	/**
	 * Set the property of the given name in the XML file. The file will still
	 * need to be saved afterwards.
	 *
	 * @param name
	 * @param value
	 */
	public synchronized void setProperty (String name, String value) {
		try {
			XPath gprPath = fgXPathFactory.newXPath();
			XPathExpression exp = null;

			exp = gprPath.compile("//Root/Properties/" + name);

			Node gprNode = null;

			gprNode = ((Node) exp.evaluate(fDocument, XPathConstants.NODE));

			if (gprNode == null) {
				exp = gprPath.compile("//Root/Properties");

				Node propNode = (Node) exp.evaluate(fDocument,
						XPathConstants.NODE);

				if (propNode == null) {
					Node rootNode = fDocument.getFirstChild();

					propNode = fDocument.createElement("Properties");

					rootNode.appendChild(propNode);
				}

				gprNode = fDocument.createElement(name);

				propNode.appendChild(gprNode);
			}

			gprNode.setTextContent(value);
		} catch (XPathExpressionException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
	}

	/**
	 * Get the property of the given name from the XML file.
	 *
	 * @param name
	 * @return
	 */
	public synchronized String getProperty (String name) {
		XPath gprPath = fgXPathFactory.newXPath();
		XPathExpression exp = null;

		try {
			exp = gprPath.compile("//Root/Properties/" + name);
		} catch (XPathExpressionException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);

			return null;
		}

		Node gprNode = null;

		try {
			gprNode = ((Node) exp.evaluate(fDocument, XPathConstants.NODE));
		} catch (XPathExpressionException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);

			return null;
		}

		if (gprNode == null) {
			return null;
		} else {
			return gprNode.getTextContent().trim();
		}
	}

	/**
	 * Set the "isRootProject" property.
	 *
	 * @param isRootProject
	 */
	public synchronized void setRootProject (boolean isRootProject) {
		setProperty(IS_ROOT_PROJECT_PROPERTY, Boolean
				.toString(isRootProject));
	}

	/**
	 * Set the "gprPath" property. This can be either a full path, or a path
	 * relative to the root of the project.
	 *
	 * @param path
	 */
	public synchronized void setGPRPath (String path) {
		setProperty(GPR_PATH_PROPERTY, path);
	}


	/**
	 * Return the "isRootProject" property.
	 *
	 * @return
	 */
	public synchronized boolean isRootProject() {
		String resultStr = getProperty(IS_ROOT_PROJECT_PROPERTY);
		boolean result;

		if (resultStr != null) {
			result =  Boolean.parseBoolean(resultStr);
		} else {
			result = deprecatedIsRootProject();

			setProperty (IS_ROOT_PROJECT_PROPERTY, Boolean.toString(result));

			new Job ("Saving project properties file") {

				@Override
				protected IStatus run(IProgressMonitor monitor) {
					saveProjectFile();
					return GNATbenchCorePlugin.OK_STATUS;
				}

			}.schedule ();
		}

		return result;
	}

	/**
	 * Return the gpr path property, as set in the XML file. See getGPROSPath
	 * for the full path to the actual resource.
	 *
	 * @return
	 */
	public synchronized String getGPRPath () {
		String result = getProperty(GPR_PATH_PROPERTY);

		if (result == null) {
			result = deprecatedGPRPath();

			if (result != null) {
				setProperty(GPR_PATH_PROPERTY, result);

				new Job ("Saving project properties file") {

					@Override
					protected IStatus run(IProgressMonitor monitor) {
						saveProjectFile();
						return GNATbenchCorePlugin.OK_STATUS;
					}

				}.schedule ();
			}
		}

		return result;
	}

	/**
	 * Return the full os path stored in the project.
	 *
	 * @param project
	 * @return
	 */
	public synchronized String getGPROSPath () {
		if (!fPrj.isOpen()) {
			return null;
		}

		String path = getGPRPath ();
		IFile file;

		if (path != null) {
			file = fPrj.getFile(new Path (path));
		} else {
			return null;
		}

		if (file != null && file.exists()) {
			return file.getLocation().toOSString();
		} else {
			return path;
		}
	}

	/**
	 * Save the current project file to the appropriate .gb_project file.
	 *
	 */
	public void saveProjectFile () {
		PipedInputStream is = null;

		synchronized (this) {

			Transformer transformer;

			TransformerFactory transformerFactory = TransformerFactory
					.newInstance();
			try {
				transformer = transformerFactory.newTransformer();
			} catch (TransformerConfigurationException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);

				return;
			}

			is = new PipedInputStream();
			PipedOutputStream os = null;

			try {
				os = new PipedOutputStream(is);
			} catch (IOException e1) {
				GNATbenchCorePlugin.getDefault().logError(null, e1);

				return;
			}

			DOMSource source = new DOMSource(fDocument);
			StreamResult result = new StreamResult(os);

			try {
				transformer.transform(source, result);

				os.close();
			} catch (TransformerException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);

				return;
			} catch (IOException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);

				return;
			}
		}

		IFile prjFile = fPrj.getFile(new Path(".gb_project"));

		try {
			Job.getJobManager().beginRule(prjFile, null);

			if (!prjFile.exists()) {
				prjFile.create(is, IFile.NONE, new NullProgressMonitor());
			} else {
				prjFile.setContents(is, IFile.NONE, new NullProgressMonitor());
			}

			Job.getJobManager().endRule(prjFile);
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}
	}

	/**
	 * Load the XML project file - create one if needed.
	 */
	private void loadProjectFile () {
		try {
			final IFile prjFile = fPrj.getFile(new Path (".gb_project"));

			if (!prjFile.exists()) {
				fDocument = fgDocBuilder.newDocument();

				fDocument.appendChild(fDocument.createElement("Root"));

				WorkspaceJob job = new WorkspaceJob ("Save project file") {

					@Override
					public IStatus runInWorkspace(IProgressMonitor monitor)
							throws CoreException {

						if (prjFile.getProject().exists()
								&& prjFile.getProject().isOpen()) {
							saveProjectFile();
						}

						return GNATbenchCorePlugin.OK_STATUS;
					}

				};

				job.setRule(fPrj);

				job.schedule();

				return;
			}

			prjFile.refreshLocal(IFile.DEPTH_INFINITE,
					new NullProgressMonitor());
			fDocument = fgDocFactory.newDocumentBuilder().parse(
					prjFile.getContents());
		} catch (Exception e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);

			fDocument = fgDocBuilder.newDocument();
		}
	}

	static {
		fgDocFactory = DocumentBuilderFactory.newInstance();
		fgXPathFactory = XPathFactory.newInstance ();

		try {
			fgDocBuilder = fgDocFactory.newDocumentBuilder();
		} catch (ParserConfigurationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}
	}


	/**
	 * Return the full os path stored in the project, in a GB 2.0 project.
	 *
	 * @param project
	 * @return
	 */
	private String deprecatedGPRPath () {
		if (!fPrj.isOpen()) {
			return "";
		}

		QualifiedName name = new QualifiedName(
				GNATbenchCorePlugin.getId(), "gprName");

		try {
			String path = fPrj.getPersistentProperty(name);

			if (path != null) {
				return path;
			} else {
				return null;
			}

		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		return null;
	}


	/**
	 * Return true is the project is a root project GPR-wise, in a GB 2.0
	 * project.
	 *
	 * @param project
	 * @return
	 */
	private boolean deprecatedIsRootProject () {
		QualifiedName name = new QualifiedName(
				GNATbenchCorePlugin.getId(), "isRootProject");

		try {
			String prop = fPrj.getPersistentProperty(name);

			return prop != null && prop.equals("TRUE");
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		return false;
	}

}
