/*******************************************************************************
 * Copyright (C) 2005-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.core.internal;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Set;
import java.util.Map.Entry;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.InvalidRegistryObjectException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.PlatformObject;
import org.eclipse.core.runtime.Status;
import org.eclipse.ui.editors.text.TextFileDocumentProvider;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.eclipse.ui.texteditor.IDocumentProvider;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;

import com.adacore.ajis.IProxy;
import com.adacore.ajis.IProxy.Owner;
import com.adacore.gnatbench.core.internal.adaeditor.AbstractAdaEditor;
import com.adacore.gnatbench.core.internal.adaeditor.AdaDocumentBuffer;
import com.adacore.gnatbench.core.internal.adaeditor.AdaPartitionScanner;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedFile;
import com.adacore.gnatbench.core.internal.builder.IGNATbenchBuildListener;
import com.adacore.gnatbench.core.internal.filesystem.FileStoreAdapterFactory;
import com.adacore.gnatbench.core.internal.filesystem.FileStoreRegistry;
import com.adacore.gnatbench.core.internal.jobs.MessageFactory;
import com.adacore.gnatbench.core.internal.preferences.AdaPreferencePage;
import com.adacore.gnatbench.core.internal.preferences.IGNATbenchPreferencesProvider;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.Case_Handling.Casing_Policy;
import com.adacore.gnatbench.library.Case_Handling.Casing_Type;
import com.adacore.gnatbench.library.GNATCOLL.Traces.On_Exception_Mode;
import com.adacore.gnatbench.library.GNATCOLL.Traces.Stream_Factory;
import com.adacore.gnatbench.library.GNATCOLL.Traces.Trace_Stream_Record;
import com.adacore.gnatbench.library.GNATCOLL.Traces.Traces_Package;
import com.adacore.gnatbench.library.Language.Indent_Parameters;
import com.adacore.gnatbench.library.Language.Indent_Style;
import com.adacore.gnatbench.library.Standard.AdaString;

/**
 * The main plugin class to be used in the desktop.
 *
 * @author ochem
 */
public class GNATbenchCorePlugin extends AbstractUIPlugin {
	//The shared instance.
	private static GNATbenchCorePlugin plugin;

	private HashMap<String, AdaDocumentBuffer> fBuffers = new HashMap<String, AdaDocumentBuffer>();

	private AdaPartitionScanner fPartitionScanner;

	private LinkedList <AbstractAdaEditor> fEditors = new LinkedList <AbstractAdaEditor> ();

	public final static String ADA_PARTITIONING = "__ada_gnatbench_partitioning";

	public static final String ADA_LOCATION_MARKER =
		"com.adacore.gnatbench.core.AdaLocationMarker";

	public static final String ADA_ERROR_MARKER =
		"com.adacore.gnatbench.core.AdaErrorMarker";

	public static final String GPR_ERROR_MARKER =
		"com.adacore.gnatbench.core.GPRErrorMarker";

	public static final String CODEFIX_MARKER =
		"com.adacore.gnatbench.core.CodefixMarker";

	public static final Status OK_STATUS = new Status
	(IStatus.OK, GNATbenchCorePlugin.getId(), IStatus.OK, "", null);

	private IDocumentProvider fDocProvider = new TextFileDocumentProvider();

	private Indent_Parameters fIndentParams;

	private FileStoreRegistry fFileStoreReg = new FileStoreRegistry ();

	private static final String defaultTracesConfig =
		">&GNATbench\n"
		+ "+\n"
		+ "DEBUG.COLORS=no\n"
		+ "DEBUG.ABSOLUTE_TIME=yes\n"
		+ "DEBUG.ELAPSED_TIME=no\n"
		+ "DEBUG.STACK_TRACE=no\n"
		+ "DEBUG.LOCATION=no\n"
		+ "DEBUG.ENCLOSING_ENTITY=no\n"
		+ "ALI=yes\n"
		+ "ALI.ASSERT=yes\n"
		+ "ENTITIES.ASSERT=yes\n"
		//  This resolve links trace is crucial, because we often assume that
		//  a file path is unique (which is not true if there are links).
		+ "VFS.Resolve_Links=yes\n";

	static final public String LOCATION_MARKER_TEXT =
		"AdaLocationMarker.Text";

	private Set<IGNATbenchBuildListener> fBuildListeners = new HashSet<IGNATbenchBuildListener>();

	private MessageFactory fMessageFactory = new MessageFactory ();

	/**
	 * This streams captures traces sent by GPS - we do not currenty capture
	 * them but should propably at some point in a file.
	 */
	private static class GNATbenchStream extends Trace_Stream_Record {

		@Override
		public void Newline() {
//			System.out.println();
		}

		@Override
		public void Put(AdaString Str) {
//			System.out.print(Str.toString());
		}

	}

	/**
	 * Factory for GNATbench trace streams.
	 */
	private static class GNATbenchStreamFactory extends Stream_Factory {

		@Override
		public Trace_Stream_Record New_Stream(AdaString Args) {
			GNATbenchStream newStream = new GNATbenchStream ();

			((IProxy) newStream).setOwner(Owner.NATIVE);
			// FIXME: Where do we free this object?

			return newStream;
		}

	}

	/**
	 * The constructor.
	 */
	public GNATbenchCorePlugin() {
		super();

		plugin = this;

		Platform.getAdapterManager().registerAdapters(
				new FileStoreAdapterFactory(), IFile.class);

		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			fIndentParams = new Indent_Parameters ();
			fIndentParams.Indent_Level(3);
			fIndentParams.Indent_Continue(2);
			fIndentParams.Indent_Conditional(1);
			fIndentParams.Indent_Record(3);
			fIndentParams.Indent_Decl(0);
			fIndentParams.Tab_Width(8);
			fIndentParams.Indent_Case_Extra(Indent_Style.Automatic);
			fIndentParams.Casing_Policy(Casing_Policy.Disabled);
			fIndentParams.Reserved_Casing(Casing_Type.Unchanged);
			fIndentParams.Ident_Casing(Casing_Type.Unchanged);
			fIndentParams.Format_Operators(false);
			fIndentParams.Use_Tabs(false);
			fIndentParams.Align_On_Colons(false);
			fIndentParams.Align_On_Arrows(false);
			fIndentParams.Align_Decl_On_Colon(false);
			fIndentParams.Indent_Comments(true);
			fIndentParams.Stick_Comments(false);

			String pluginLoc = ResourcesPlugin.getWorkspace().getRoot()
					.getLocation().toOSString()
					+ "/.metadata/.plugins/com.adacore.gnatbench.core/";

			String loc = pluginLoc + "gnatbench_config.cfg";

			File folder = new File (pluginLoc);

			folder.mkdirs();

			File file = new File (loc);

			if (!file.exists()) {
				file.createNewFile();
				FileOutputStream fi = new FileOutputStream (file);
				fi.write(defaultTracesConfig.getBytes());
				fi.close();
			}

			GNATbenchStreamFactory factory = new GNATbenchStreamFactory ();
			((IProxy) factory).setOwner(Owner.NATIVE);

			Traces_Package.Register_Stream_Factory(new AdaString ("GNATbench"),
					factory);
			Traces_Package.Parse_Config_File(new AdaString(loc), new AdaString(
					""), On_Exception_Mode.Propagate);
		} catch (IOException e) {
			logError(null, e);
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}

		// Install a listener so that we recompute the gpr projects' contents
		// whenever projects and files are added or removed
		ResourcesPlugin.getWorkspace().addResourceChangeListener(
				new GPRResourceSpy());
	}

	@Override
	public void start (BundleContext context) throws Exception {
		super.start(context);
	}

	public Indent_Parameters getIndentParameters () {
		return fIndentParams;
	}

	public void setIndentParameters (Indent_Parameters params) {
		fIndentParams = params;
	}

	/**
	 * Returns the shared instance.
	 */
	public static GNATbenchCorePlugin getDefault() {
		return plugin;
	}

	public void notifyError(final String error) {
		notifyError(error, null);
	}

	public void notifyError(final String error, final Exception e) {
		//  TODO: add an extension point here in order to allow GNATbenchUI to
		//  add contribute to this.
	}

	/**
	 *
	 * @return The current id of the plugin.
	 */
	static public String getId() {
		return "com.adacore.gnatbench.core";
	}

	public String getPluginPath() {
		try {
			String path = FileLocator.toFileURL(getBundle().getEntry("/"))
					.getPath();

			if (Platform.getOS().equals("win32")) {
				// Transforms path like /c:/ into c:/
				return path.substring(1);
			} else {
				return path;
			}
		} catch (IOException e) {
			logError(null, e);
			return "";
		}
	}

	/**
	 * Returns the buffer corresponding to the location given in parameter. If
	 * an editor is opened, then its buffer will be send. Otherwise, the file
	 * will be loaded. Files are only loaded once.
	 *
	 * @param location
	 * @return
	 * deprecated this should now use the GeneralizedLocation based version.
	 */
	public AdaDocumentBuffer getDocumentBuffer(GeneralizedFile file) {
		AbstractAdaEditor editor = getEditor(file);

		if (editor != null) {
			return editor.getDocumentBuffer();
		} else {
			//  FIXME: buffer don't work if the file has been modified
			//  externally, we need to compare some kind of timestamp here.

			AdaDocumentBuffer buffer = fBuffers
					.get(file.getEclipsePath());

			if (buffer != null) {
				return buffer;
			} else {
				try {
					getDocumentProvider().connect(file.getInput());
				} catch (CoreException e) {

					logError(null, e);
				}

				buffer = new AdaDocumentBuffer(file);
				fBuffers.put(file.getEclipsePath(), buffer);

				return buffer;
			}
		}
	}

	/**
	 * Returns the editor opened for the file name specified in the location,
	 * null if none.
	 *
	 * @param location
	 * @return
	 */
	public AbstractAdaEditor getEditor(GeneralizedFile file) {
		synchronized (fEditors) {
			try {
				for (AbstractAdaEditor editor : fEditors) {
					GeneralizedFile editorFile = editor.getFile();

					if (editorFile == null) {
						continue;
					}

					//  TODO: See problems about case sensitivity and so here.
					if (file.getOSPath().equals(
							editorFile.getOSPath())) {
						return editor;
					}
				}
			} catch (Exception e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		}

		return null;
	}

	/**
	 * Returns the editor opened on the specific file, null if none.
	 *
	 * @param location
	 * @return
	 */
	public AbstractAdaEditor getEditor (IFile file) {
		synchronized (fEditors) {
			for (AbstractAdaEditor editor : fEditors) {
				if (file
						.equals(editor.getFile().getFile())) {
					return editor;
				}
			}
		}

		return null;
	}

	/**
	 *
	 * @return the partition scanner shared by all editors.
	 */
	public AdaPartitionScanner getPartitionScanner() {
		if (fPartitionScanner == null) {
			fPartitionScanner = new AdaPartitionScanner();
		}

		return fPartitionScanner;
	}

	/**
	 * Remove the buffer pointing to the file hold in the location, if any.
	 *
	 * @param eclipsePath
	 */
	public void removeBuffer(GeneralizedFile file) {
		AdaDocumentBuffer buffer = fBuffers.get(file.getEclipsePath());

		if (buffer != null) {
			getDocumentProvider().disconnect(file.getInput());
			fBuffers.remove(file.getEclipsePath());
			buffer.dispose();
		}
	}

	/**
	 * This have to be called each time an editor is opened.
	 */
	public void registerEditor(AbstractAdaEditor editor) {
		synchronized (fEditors) {
			fEditors.add (editor);
		}
	}

	/**
	 * This have to be called each time an editor is closed.
	 */
	public void unregisterEditor (AbstractAdaEditor editor) {
		synchronized (fEditors) {
			fEditors.remove (editor);
		}
	}

	/**
	 * Creates the preferences fields related to a given page.
	 *
	 * @param page
	 */
	public void createPreferenceFields (AdaPreferencePage page) {
		IExtensionRegistry er = Platform.getExtensionRegistry();
		IExtensionPoint ep = er.getExtensionPoint(GNATbenchCorePlugin.getId(),
				"preference");

		IExtension[] extensions = ep.getExtensions();

		if (extensions.length == 0) {
			return;
		}

		for (int extInd = 0; extInd < extensions.length; ++extInd) {

			IConfigurationElement[] elements = extensions[extInd]
					.getConfigurationElements();

			for (int i = 0; i < elements.length; ++i) {
				if (elements[i].getName().equals("provider")) {
					try {
						Bundle bundle = Platform.getBundle(extensions[extInd]
								.getNamespaceIdentifier());

						String id = elements[i].getAttribute("pageId");

						if (id.equals(page.getId())) {
							IGNATbenchPreferencesProvider preferenceProvider =
								(IGNATbenchPreferencesProvider) bundle
									.loadClass(
											elements[i].getAttribute("class"))
									.newInstance();

							preferenceProvider.createPreferenceFields(page);
						}
					} catch (InvalidRegistryObjectException e) {
						logError (null, e);
					} catch (InstantiationException e) {
						logError (null, e);
					} catch (IllegalAccessException e) {
						logError (null, e);
					} catch (ClassNotFoundException e) {
						logError (null, e);
					}
				}
			}
		}
	}

	public IDocumentProvider getDocumentProvider () {
		return fDocProvider;
	}

	/**
	 * @param sourceObject
	 * @param adapterType
	 * @return
	 */
    @SuppressWarnings("unchecked")
	public static Object getAdapter(Object sourceObject, Class adapterType) {
    	Assert.isNotNull(adapterType);
        if (sourceObject == null) {
            return null;
        }
        if (adapterType.isInstance(sourceObject)) {
            return sourceObject;
        }

        if (sourceObject instanceof IAdaptable) {
            IAdaptable adaptable = (IAdaptable) sourceObject;

            Object result = adaptable.getAdapter(adapterType);
            if (result != null) {
                // Sanity-check
                Assert.isTrue(adapterType.isInstance(result));
                return result;
            }
        }

        if (!(sourceObject instanceof PlatformObject)) {
            Object result = Platform.getAdapterManager().getAdapter(sourceObject, adapterType);
            if (result != null) {
                return result;
            }
        }

        return null;
    }

    public FileStoreRegistry getEFSRegistry () {
    	return fFileStoreReg;
    }

	/**
	 * Stop a process if its execution is too long
	 *
	 * @param GNATbenchProcess
	 * @param Time
	 */
	public void timedProcess (Process GNATbenchProcess, final int Time) {

		final Process process = GNATbenchProcess;
		final Thread registryThread = Thread.currentThread();

		try {

			new Thread() {
				@Override
				public void run() {
					try {
						Thread.sleep(Time);
					} catch (InterruptedException e) {

					}

					// If we did not get the result of the command in "Time" sec,
					// then there's something wrong and we should
					// interrupt the thread. This may happen in certain
					// cases using Windows for example.

					try {
						process.exitValue();
					} catch (IllegalThreadStateException e) {
						registryThread.interrupt();
					}
				}
			}.start();

			GNATbenchProcess.waitFor();
		} catch (InterruptedException e2) {
			// If the thread has been interrupted, this should be logged.
			// But we can still make use of the output so no need to
			// stop completely the process.

			logError(null, e2);
		}

	}

	/**
	 * Adds the listener given in parameter to the GNATbench build listeners.
	 * Does nothing if the listener is alreay there.
	 */
	public void addBuildListener (IGNATbenchBuildListener listener) {
		fBuildListeners.add(listener);
	}

	/**
	 * Remove the listener in parameter from the GNATbench build listeners.
	 * Does nothing if the listener is not there.
	 */
	public void removeBuildListener (IGNATbenchBuildListener listener) {
		fBuildListeners.remove(listener);
	}

	/**
	 * Return a copy of the current list of the build listeners.
	 */
	public Set <IGNATbenchBuildListener> getBuildListeners () {
		return new HashSet <IGNATbenchBuildListener> (fBuildListeners);
	}

	/**
	 * Remove all the AdaBuffer stored.
	 * FIXME: this is only needed because we don't react propertly to buffer
	 * modifications outside of an editor - we should invalidate buffer
	 * when changes are detected through the resource spy on buffers not
	 * associated with an editor - or see if there is a way to have
	 * a better connection between a buffer and its associated resource.
	 */
	public void clearBuffers () {
		for (Entry<String, AdaDocumentBuffer> bufferEntry : fBuffers.entrySet()) {
			AdaDocumentBuffer buffer = bufferEntry.getValue();

			getDocumentProvider().disconnect(buffer.getFile().getInput());
			buffer.dispose();
		}

		fBuffers.clear();
	}

	public MessageFactory getMessageFactory () {
		return fMessageFactory;
	}

	/**
	 * Adds an error to the Eclipse log. If message is null, then a default
	 * message "Unexpected Exception" will be given. If message and exception
	 * are null, then no log message will get added.
	 *
	 * @param message
	 * @param exception
	 */
	public void logError (String message, Throwable exception) {
		if (message == null && exception == null) {
			return;
		}

		if (message == null) {
			message = "UnexpectedException";
		}

		getLog().log(
				new Status(IStatus.ERROR, GNATbenchCorePlugin.getId(),
						IStatus.OK, message, exception));
	}
}
