/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.cdt.internal.wizards.newelement;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.dialogs.WizardNewFileCreationPage;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.wizards.newresource.BasicNewFileResourceWizard;

import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.gpswrappers.GPRProject;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistryFactory;
import com.adacore.gnatbench.core.internal.utils.ProjectUtils;
import com.adacore.gnatbench.ui.internal.adaeditor.AdaEditor;
import com.adacore.gnatbench.ui.internal.adaeditor.AdaEditorPreferences;
import com.adacore.gnatbench.ui.internal.GNATbenchUIPlugin;
import com.adacore.gnatbench.ui.internal.utils.DisplayUtils;

public class NewAdaSourceFileWizard
	extends BasicNewFileResourceWizard
	implements INewWizard
{

	private ImageDescriptor logo = ImageDescriptor.createFromImage(
			GNATbenchUIPlugin
				.getDefault()
				.getImage(GNATbenchUIPlugin.ADACORE_LOGO));


	protected WizardNewFileCreationPage page;

	public void addPages() {
		page = new WizardNewFileCreationPage(
				"somePageName",
				getSelection());

		page.setTitle("Create New Ada Source File");
		page.setDescription("Create a new Ada source file with header " +
				"text automatically inserted.");
		page.setImageDescriptor(logo);
		addPage(page);
	} // addPages


	public boolean canFinish() {
    	if (!page.isPageComplete()) {
			return false;
		} // if

    	// additional test to prevent checking too soon
    	if (page.getFileName() == null || page.getFileName().equals("")) {
			return false;
		} // if

		String projectName = page.getContainerFullPath().uptoSegment(1).makeRelative().toString();
		GPRProject gpr = ((GNATProjectRegistryFactory) GNATbenchSession
				.getDefault().getProjectRegistryFactory())
				.getGlobalProjectFromName(projectName);
		IProject project = gpr.getEclipseProject();
        if (!ProjectUtils.isAdaSourceFileName(page.getFileName(), project)) {
        	page.setErrorMessage("Invalid source file name (extension) for this project");
        	return false;
		} // if

        return true;
	} // canFinish


	public boolean performFinish() {
        IFile file = page.createNewFile();
        if (file == null) {
			return false;
		} // if

        selectAndReveal(file);

        IWorkbenchWindow dw = getWorkbench().getActiveWorkbenchWindow();
        try {
            if (dw != null) {
                IWorkbenchPage page = dw.getActivePage();
                if (page != null) {
                    IEditorPart part = IDE.openEditor(page, file, true);
                    AdaEditor editor = (AdaEditor) part;
                    insertHeader(editor, file);
            		skipToHeaderEnd(editor);
            		editor.doSave(new NullProgressMonitor ());
                } // if
            } // if
        } catch (PartInitException e) {
        	GNATbenchCorePlugin.getDefault().logError(null, e);
        	DisplayUtils.displayError("Create New Ada Source File",
                    e.getMessage());
        } // try

        return true;
    } // performFinish


	protected void insertHeader(final AdaEditor editor, final IFile file) {
		if (!isFileHeaderEnabled()) {
			return;
		} // if

		final IProject project = file.getProject();
		final String fileName = file.getName();

        String headerFileName = null;

		if (fileName.endsWith(ProjectUtils.specSuffixAda(project))) {
			headerFileName = specsHeaderFileName();
			if (headerFileName.trim().equals("")) {
	        	DisplayUtils.displayWarning(
	        			"Create New Ada Source File",
	        			"Inserting header text is enabled but the " +
	        			"filename preference for spec headers is " +
	        			"empty.  See Ada/Editor preferences.");
				return;
			} // if
		} else {
			headerFileName = bodiesHeaderFileName();
			if (headerFileName.trim().equals("")) {
	        	DisplayUtils.displayWarning(
	        			"Create New Ada Source File",
	        			"Inserting header text is enabled but the " +
	        			"filename preference for body headers is " +
	        			"empty.  See Ada/Editor preferences.");
				return;
			} // if
		} // if

        insertText(editor, headerFileName);
	} // insertHeader


	protected void insertText(final AdaEditor editor, final String headerFileName) {
        final IDocument document = editor.getDocumentBuffer().getDocument();
        try {
			document.replace(0, 0, fileContent(headerFileName));
		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
        	DisplayUtils.displayError("Create New Ada Source File",
        			"Could not insert header file text from '" + headerFileName + "'");
        	return;
		}
	} // insertText


	protected String fileContent(final String fileName) {
		FileInputStream fileInput = null;

		try {
			fileInput = new FileInputStream(fileName);
		} catch (FileNotFoundException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
        	DisplayUtils.displayError("Create New Ada Source File",
        			"Could not find header file '" + fileName + "'");
        	return null;
		} // try

		String text = "";
		byte[] byteBuffer = new byte[2048];

		int numberRead = 0;

		try {
			while ((numberRead = fileInput.read(byteBuffer)) > 0) {
				text += new String(byteBuffer, 0, numberRead);
			} // while
		} catch (IOException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
        	DisplayUtils.displayError("Create New Ada Source File",
        			"Could not read header file '" + fileName + "'");
        	return null;
		} // try

		return text;
	} // fileContent


	protected void skipToHeaderEnd(final AdaEditor editor) {
        IDocument document = editor.getDocumentBuffer().getDocument();
        int length = document.getLength();
		editor.selectAndReveal(length+1, 0);
	} // skipToHeaderEnd


    protected boolean isFileHeaderEnabled() {
    	return GNATbenchCorePlugin
    		.getDefault()
    		.getPreferenceStore()
    		.getBoolean(
    			AdaEditorPreferences.INSERT_HEADER_FILE);
    } // isHeaderFileEnabled


    protected String specsHeaderFileName() {
    	return GNATbenchCorePlugin
			.getDefault()
			.getPreferenceStore()
			.getString(
			AdaEditorPreferences.SPEC_HEADER_FILE_NAME);
    } // specsHeaderFileName


    protected String bodiesHeaderFileName() {
    	return GNATbenchCorePlugin
			.getDefault()
			.getPreferenceStore()
			.getString(
			AdaEditorPreferences.BODY_HEADER_FILE_NAME);
    } // bodiesHeaderFileName


} // NewAdaSourceFolderWizard
