/*******************************************************************************
 * Copyright (c) 2006 AdaCore.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.cdt.internal.wizards.newProject;

import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

import com.adacore.gnatbench.ui.internal.GNATbenchUIPlugin;

public class AdaBuilderPage extends WizardPage {
	
	private String builderName = ""; 
	private Button fUseAda2005;

	// the following are enabled if the user does not select the builder
	// from the pull-down list, such that they are entering a foreign builder
	// and we won't know what commands to use automatically.
	
	private Text   buildCommandWidget;
	private String foreignBuildCommand = "";
        
	private Text   cleanCommandWidget;
	private String foreignCleanCommand = "";

	private Text   analyzeCommandWidget;
	private String foreignAnalyzeCommand = "";

	private Text   autoBuildCommandWidget;
	private String foreignAutoBuildCommand = "";
	
	
	private ImageDescriptor logo = ImageDescriptor.createFromImage(
			GNATbenchUIPlugin
				.getDefault()
				.getImage(GNATbenchUIPlugin.ADACORE_LOGO));
	

	public AdaBuilderPage(String pageName) {
		super (pageName);
		this.setTitle(pageName);
		this.setDescription("Identify the GNAT builder or, for other " +
				"builders, enter both the name and the builder commands.");
		this.setImageDescriptor(logo);
	} // AdaBuilderPage
	
	
	public String builderName() {
		return builderName;
	} // builderName	
	
	
	public boolean useAda2005() {
		return fUseAda2005.getSelection();
	} // useAda2005

	
	public boolean isForeignBuilder() {
		return !isGNATProBuilder(builderName);
	} // isForeignBuilder

	
	public String getForeignBuildCommand() {
		return foreignBuildCommand;
	} // getForeignBuildCommand


	public String getForeignCleanCommand() {
		return foreignCleanCommand;
	} // getForeignCleanCommand


	public String getForeignAnalyzeCommand() {
		return foreignAnalyzeCommand;
	} // getForeignAnalyzeCommand


	public String getForeignAutoBuildCommand() {
		return foreignAutoBuildCommand;
	} // getForeignAutoBuildCommand
	
	
    public boolean isPageComplete() {
    	return (!builderName.equals("")
    			// buildername is a GNATname OR all the manual fields are entered
				&& (isGNATProBuilder(builderName) || allForeignCommandsEntered()));
    } // isPageComplete
	

    public void setVisible(boolean visible) {
        super.setVisible(visible);
        if (visible) {
        	fUseAda2005.setFocus();
        } // if
    } // SetVisible

    
	public void createControl(Composite parent) {
		Composite baseComp = new Composite(parent, SWT.NONE);
		setControl(baseComp);

		GridLayout topLayout = new GridLayout();
		baseComp.setLayout(topLayout);

		Composite mainProgComp = new Composite(baseComp, SWT.NONE);
		GridLayout locationLayout = new GridLayout();
		locationLayout.numColumns = 2;
		locationLayout.marginHeight = 1;
		locationLayout.marginWidth = 1;
		mainProgComp.setLayout(locationLayout);
		
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		mainProgComp.setLayoutData(gd);
				
		// do they want to use Ada 2005?
		
		fUseAda2005 = new Button (mainProgComp, SWT.CHECK);
		fUseAda2005.setText("Build using Ada 2005");
		fUseAda2005.setSelection(false);
		gd = new GridData();
		gd.horizontalSpan = 2;
		gd.verticalIndent = 5;
		fUseAda2005.setLayoutData(gd);		
			
		// the combo for the pull-down list providing the GNAT builder names
		// the section for getting the builder name 
		
		Label toolNameLabel = new Label(mainProgComp, SWT.NONE);
		toolNameLabel.setText("Name of the Ada toolchain:");
		gd = new GridData();
		gd.horizontalSpan = 2;
		gd.verticalIndent = 5;
		toolNameLabel.setLayoutData(gd);
		
		// the entry of the compiler name
		
		final Combo toolName = new Combo(mainProgComp, SWT.SINGLE);

		// well yes, this is pretty ugly
		// TODO: come up with a better scheme (file driven??)
		toolName.add("gnatmake");
		toolName.add("gnaampmake");
		toolName.add("erc32-elf-gnatmake");
		toolName.add("leon-elf-gnatmake");		
		toolName.add("powerpc-elf-gnatmake");
		toolName.add("powerpc-elf-lynxos-gnatmake");
		toolName.add("powerpc-xcoff-lynxos-gnatmake");

		toolName.setText("select or enter a builder name (e.g., gnatmake)");
		
		// handle case when the user enters the name manually
		toolName.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				builderName = new String (toolName.getText()).trim();				
				if ((!builderName.equals("")) && !isGNATProBuilder(builderName)) {
					enableForeignCommandEntries();					
				} else {
					// they manually entered a valid GNAT builder name
					disableForeignCommandEntries();					
				} // if				
				if (isCurrentPage()) {
					getContainer().updateButtons();
				} // if
			}});
		
		// handle the case in which the pull-down list is used
		toolName.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {		    	  
				builderName = new String (toolName.getText());
				disableForeignCommandEntries();									
				if (isCurrentPage()) {
					getContainer().updateButtons();
				} // if
			}
		});

		toolName.setLayoutData(new GridData(GridData.GRAB_HORIZONTAL
				| GridData.HORIZONTAL_ALIGN_FILL));

		toolName.setEnabled(true);
		
		// when the user specifies a non-gnat builder, we enable these text entries 
		// in order to get the build, compile, clean, and analyze commands
		
		// full build command

		Label buildCommandLabel = new Label(mainProgComp, SWT.NONE);
		buildCommandLabel.setText("The complete command to do a full build:");
		gd = new GridData();
		gd.horizontalSpan = 2;
		gd.verticalIndent = 35;
		buildCommandLabel.setLayoutData(gd);

		buildCommandWidget = new Text(mainProgComp, SWT.SINGLE | SWT.BORDER);
		
		buildCommandWidget.addModifyListener(new ModifyListener () {

			public void modifyText(ModifyEvent e) {
				foreignBuildCommand = new String (((Text) e.getSource()).getText().trim());
		        if (isCurrentPage()) {
		            getContainer().updateButtons();
		        } // if
			}});
		
		gd = new GridData(GridData.FILL_HORIZONTAL);
		buildCommandWidget.setLayoutData(gd);		
		buildCommandWidget.setEnabled(false);
		
		// clean command		

		Label cleanCommandLabel = new Label(baseComp, SWT.NONE);
		cleanCommandLabel.setText("The complete command to clean a project:");
		gd = new GridData();
		gd.horizontalSpan = 2;
		buildCommandLabel.setLayoutData(gd);

		cleanCommandWidget = new Text(baseComp, SWT.SINGLE | SWT.BORDER);
		
		cleanCommandWidget.addModifyListener(new ModifyListener () {

			public void modifyText(ModifyEvent e) {
				foreignCleanCommand = new String (((Text) e.getSource()).getText().trim());
		        if (isCurrentPage()) {
		            getContainer().updateButtons();
		        } // if
			}});
		
		gd = new GridData(GridData.FILL_HORIZONTAL);
		cleanCommandWidget.setLayoutData(gd);		
		cleanCommandWidget.setEnabled(false);		
		
		// analyze command		

		Label analyzeCommandLabel = new Label(baseComp, SWT.NONE);
		analyzeCommandLabel.setText("The complete command to analyze all sources:");
		gd = new GridData();
		gd.horizontalSpan = 2;
		analyzeCommandLabel.setLayoutData(gd);

		analyzeCommandWidget = new Text(baseComp, SWT.SINGLE | SWT.BORDER);
		
		analyzeCommandWidget.addModifyListener(new ModifyListener () {

			public void modifyText(ModifyEvent e) {
				foreignAnalyzeCommand = new String (((Text) e.getSource()).getText().trim());
		        if (isCurrentPage()) {
		            getContainer().updateButtons();
		        } // if
			}});
		
		gd = new GridData(GridData.FILL_HORIZONTAL);
		analyzeCommandWidget.setLayoutData(gd);		
		analyzeCommandWidget.setEnabled(false);		
		
		// autobuild command to analyze only those that have changed		

		Label autoBuildCommandLabel = new Label(baseComp, SWT.NONE);
		autoBuildCommandLabel.setText("The complete command to analyze changed sources:");
		gd = new GridData();
		gd.horizontalSpan = 2;
		autoBuildCommandLabel.setLayoutData(gd);

		autoBuildCommandWidget = new Text(baseComp, SWT.SINGLE | SWT.BORDER);
		
		autoBuildCommandWidget.addModifyListener(new ModifyListener () {

			public void modifyText(ModifyEvent e) {
				foreignAutoBuildCommand = new String (((Text) e.getSource()).getText().trim());
		        if (isCurrentPage()) {
		            getContainer().updateButtons();
		        } // if
			}});
		
		gd = new GridData(GridData.FILL_HORIZONTAL);
		autoBuildCommandWidget.setLayoutData(gd);		
		autoBuildCommandWidget.setEnabled(false);
	} // createControl
		
	
	protected void enableForeignCommandEntries() {
		buildCommandWidget.setEnabled(true);
		cleanCommandWidget.setEnabled(true);
		analyzeCommandWidget.setEnabled(true);
		autoBuildCommandWidget.setEnabled(true);			
	} // enableForeignCommandEntries
	
	
	protected void disableForeignCommandEntries() {
		buildCommandWidget.setEnabled(false);
		cleanCommandWidget.setEnabled(false);
		analyzeCommandWidget.setEnabled(false);
		autoBuildCommandWidget.setEnabled(false);		
	} // disableForeignBuilderEntries
	
	
	protected boolean isGNATProBuilder(String input) {
		if (input.toLowerCase().endsWith("gnatmake")) {
			return true;
		} // if
//		if (input.equalsIgnoreCase("gprmake")) {
//			return true;
//		} // if
		if (input.toLowerCase().endsWith("gnaampmake")) {
			return true;
		} // if
		return false;
	} // isGNATProBuilder
	
		
	protected boolean allForeignCommandsEntered() {
		if (foreignBuildCommand.equals("")) {
			return false;			
		} // if
		if (foreignCleanCommand.equals("")) {
			return false;			
		} // if
		if (foreignAnalyzeCommand.equals("")) {
			return false;			
		} // if
		if (foreignAutoBuildCommand.equals("")) {
			return false;			
		} // if
		// otherwise...
		return true;
	} // allForeignCommandsEntered
	

} // AdaBuilderPage
