/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.cdt.internal.wizards.newProject;

import java.io.ByteArrayInputStream;
import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.dialogs.WizardNewProjectCreationPage;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.GNATbenchProjectProperties;
import com.adacore.gnatbench.core.internal.builder.GNATbenchIncrementalBuilder;
import com.adacore.gnatbench.core.internal.builder.GNATbenchProjectNature;
import com.adacore.gnatbench.core.internal.make.IMakefile;
import com.adacore.gnatbench.core.internal.utils.SyntaxUtils;
import com.adacore.gnatbench.ui.internal.codingstyle.AdaCodingStylePreferencesProvider;
import com.adacore.gnatbench.ui.internal.GNATbenchUIPlugin;

public abstract class AbstractNewAdaProject extends Wizard implements INewWizard {

	protected AdaMainPage mainPage;
	protected AdaBuilderPage builderPage;

	protected IProject newProject;

	protected ImageDescriptor logo = ImageDescriptor.createFromImage(
			GNATbenchUIPlugin
			.getDefault()
			.getImage(GNATbenchUIPlugin.ADACORE_LOGO));


	// The following subclass ensures that project names do not contain blanks and
	// other problematic characters that make GNATbench unhappy.  That said,
	// having names conform to Ada identifier rules is probably too restrictive...
	public class RestrictedWizardNewProjectCreationPage extends WizardNewProjectCreationPage {

		@Override
		public String getProjectName() {
			return super.getProjectName().toLowerCase();
		}

		public RestrictedWizardNewProjectCreationPage(String pageName) {
			super(pageName);
		} // ctor

		@Override
		protected boolean validatePage() {
			super.validatePage();
			if (!SyntaxUtils.isAdaIdentifier(this.getProjectName())) {
				this.setErrorMessage("Project names must conform to Ada identifier rules");
				return false;
			} // if
			return true;
		} // validatePage

	} // RestrictedWizardNewProjectCreationPage


	protected RestrictedWizardNewProjectCreationPage newProjPage;



	public AbstractNewAdaProject() {
		super();
	} // AbstractNewAdaProject


	public void addPages() {
		newProjPage = new RestrictedWizardNewProjectCreationPage("NewGNATProPage");
		newProjPage.setImageDescriptor(logo);
		// subclasses will set specific descriptions for newprojPage
		// and then add it.
		// subclasses will allocate the mainPage and builderPage and add
		// them in an order appropriate for their other pages.
	} // addPages


	public IProject getNewProject() {
		return newProject;
	} // getNewProject


	public void init(IWorkbench workbench, IStructuredSelection selection) {
	} // Init


	protected abstract boolean setupProject(
			final IProject project,
			final String mainUnitName,
			final String builderName);
	// for example, create a dir for the Object_Dir attribute


	protected abstract boolean createGPRfile(
			final String projectName,
			final String mainUnitName,
			final String builderName);


	protected boolean allPagesComplete() {
		if (!mainPage.isPageComplete()) {
			return false;
		} // if
		if (!builderPage.isPageComplete()) {
			return false;
		} // if
		return true;
	}


	public boolean performFinish() {
		if (!allPagesComplete()) {
			return false;
		} // if

		WorkspaceModifyOperation op = new WorkspaceModifyOperation() {
			protected void execute(IProgressMonitor monitor)
			throws CoreException {
				// create the new project operation
				// call this first!
				if (!createNewAdaProject()) {
					return;
				} // if

				// now that the project exists, we can write the gpr file with
				// the same name
				// as the project itself (hence we MUST create the new project
				// first).

				if (!createGPRfile(newProject.getName(), mainPage.unitName(),
						builderPage.builderName())) {
					return;
				} // if

				if (!createMakefile(newProject,
						builderPage.isForeignBuilder(),
						builderPage.getForeignBuildCommand(),
						builderPage.getForeignCleanCommand(),
						builderPage.getForeignAnalyzeCommand(),
						builderPage.getForeignAutoBuildCommand()))
				{
					return;
				} // if

				if (!setupProject(newProject, mainPage.unitName(), builderPage.builderName())) {
					return;
				} // if

				// now we create the ada main program file with the specified
				// unit name unless
				// they don't want us to generate one, in which case they will
				// uncheck the box

				if (mainPage.shouldCreateMainFile()) {
					if (!createMainFile(mainPage.unitName(), mainPage.requestedHelloWorld())) {
						return;
					} // if
				} // if
			}
		};

		// run the project mod operation
		try {
			getContainer().run(false, true, op);
		} catch (InterruptedException e) {
			return false;
		} catch (InvocationTargetException e) {
			displayError("Create New Ada Project", "Project creation failed");
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return false;
		} // try

		return true;
	} // performFinish


	protected boolean createMakefile(final IProject project,
			final boolean isForeignBuilder,
			final String foreignBuildCommand,
			final String foreignCleanCommand,
			final String foreignAnalyzeCommand,
			final String foreignAutoBuildCommand)
	{
		final IFile file = project.getFile(new Path ("/" + IMakefile.MAKEFILE_NAME));

		final String makeFile;

		if (isForeignBuilder) {
			makeFile = GNATbenchIncrementalBuilder.makefileFromStrings(
					foreignBuildCommand,
					foreignCleanCommand,
             		"# not defined", // the "clean entire tree" command; a no-op
             		foreignAnalyzeCommand,
             		foreignAutoBuildCommand);
		} else {
			// this is a GNAT compiler so it gets the builder name etc
			// from the GNAT project file
			makeFile = GNATbenchIncrementalBuilder.makefileForGPR();
		} // if

		try {
			file.create(new ByteArrayInputStream(makeFile.getBytes()), true,
					null);
		} catch (CoreException e) {
			displayError("Create New Ada Project", "Inserting makefile failed");
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return false;
		} // try

		return true;
	} // createMakefile


	protected boolean createMainFile(final String mainUnitName, final boolean requestedHelloWorld) {
		final String mainFileName = new String (mainUnitName + ".adb");

		final IFile file = newProject.getFile(new Path (mainFileName));

		final String newMainContent;

		if (requestedHelloWorld) {
			newMainContent =
				"with GNAT.IO;  use GNAT.IO;\n" +
				"procedure " + mainUnitName + " is\n" +
				"begin\n" +
			    "   Put_Line (\"Hello World!\");\n" +
				"end " + mainUnitName + ";\n";
		} else {
			newMainContent =
				"procedure " + mainUnitName + " is\n" +
				"begin\n" +
			    "   null; -- remove this null statement and enter your code here\n" +
				"end " + mainUnitName + ";\n";
		} // if

		if (!file.exists()) {
			try {
				file.create(
						new ByteArrayInputStream(newMainContent.getBytes()),
						true, null);
			} catch (CoreException e) {
				displayError("Create New Ada Project",
						"Inserting main program file '" + mainFileName +
						"' to project " + newProject.getName() + " failed");
				GNATbenchCorePlugin.getDefault().logError(null, e);
				return false;
			} // try
		} else {
			displayError("Create New Ada Project",
					"A file named '" + mainFileName +
					"' in project " + newProject.getName() + " already exists");
			return false;
		} // if

		return true;
	} // createMainfile


	protected boolean createNewAdaProject() {
		final IProject newProjectHandle = newProjPage.getProjectHandle();

		IPath defaultPath = Platform.getLocation();
		IPath newPath = newProjPage.getLocationPath();
		if (defaultPath.equals(newPath)) {
			newPath = null;
		} // if

		IWorkspace workspace = ResourcesPlugin.getWorkspace();

		final IProjectDescription description = workspace
				.newProjectDescription(newProjectHandle.getName());

		description.setLocation(newPath);

		try {
			createProject(description, newProjectHandle, new NullProgressMonitor ());
			GNATbenchProjectNature.addNature(newProjectHandle);
		} catch (OperationCanceledException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return false;
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return false;
		} // try

		GNATbenchProjectProperties props = GNATbenchProjectProperties
				.getPropertiesFor(newProjectHandle);
		props.setRootProject(true);

		String projPath = newProjectHandle.getName() + ".gpr";
		// the gpr file has the same name as the project itself
		props.setGPRPath(projPath);

		props.saveProjectFile();

		newProject = newProjectHandle;

		return true;
	} // createNewAdaProject


	protected void createProject(IProjectDescription description,
			IProject projectHandle, IProgressMonitor monitor)
			throws CoreException, OperationCanceledException {
		try {
			monitor.beginTask("", 2000);

			projectHandle.create(description, new SubProgressMonitor(monitor,1000));

			if (monitor.isCanceled())
				throw new OperationCanceledException();

			projectHandle.open(new SubProgressMonitor(monitor, 1000));

		} finally {
			monitor.done();
		} // try
	} // createProject


	protected void displayError(String title, String msg) {
		MessageDialog.openError(getShell(), title, msg);
	} // displayError


	protected String tab(final int level) {
		if (level < 1) {
			return "";
		} // if
		final int indentPref = GNATbenchCorePlugin
			.getDefault()
			.getPreferenceStore()
			.getInt(AdaCodingStylePreferencesProvider.PREF_INDENT_LEVEL);
		StringBuffer buffer = new StringBuffer(indentPref);
		//  essentially we want the Ada equivalent of: ' ' * indentPref
		for (int k = 0; k < (indentPref * level); k++) {
			buffer.append (' ');
		} // for
		return buffer.toString();
	}

} // AbstractNewAdaProject
