/*******************************************************************************
 * Copyright (C) 2006-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.cdt.internal.wizards.conversion;

import java.io.ByteArrayInputStream;
import java.util.Iterator;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.ui.IObjectActionDelegate;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.GNATbenchProjectProperties;
import com.adacore.gnatbench.core.internal.builder.GNATbenchIncrementalBuilder;
import com.adacore.gnatbench.core.internal.builder.GNATbenchProjectNature;
import com.adacore.gnatbench.core.internal.make.IMakefile;
import com.adacore.gnatbench.core.internal.utils.FileUtils;
import com.adacore.gnatbench.ui.internal.utils.DisplayUtils;

/**
 * This class converts an Eclipse project into a GNATbench project.
 * If the Eclipse project already has the necessary .gpr and
 * makefile files we only add the nature and set the properties.
 * This could be the case, for example, when an existing project
 * is checked out of the configuration management system.
 *
 * If there is only one existing .gpr file and it has the same name
 * as the project itself we will use it.
 * If there is only one .gpr file but with a different name we will
 * ask the user if that is the one to use.
 * If there are multiple .gpr files we open a file selection
 * dialog so the user can choose which one to use.
 * If we are using none of the existing .gpr files, or if none exist,
 * we invoke the wizard to create the .gpr and makefile files.
 */
public class GNATbenchProjectConvertor implements IObjectActionDelegate {

	final String dialogTitle = "Convert to GNATbench Project";

	StructuredSelection fSelection;

	public void setActivePart(IAction action, IWorkbenchPart targetPart) {
		// TODO Auto-generated method stub
	} // setActivePart


	public void run(IAction action) {
		for (Iterator <IProject> iter = fSelection.iterator(); iter.hasNext();) {

			String completionMsg = "Conversion complete.";

			IProject project = iter.next();

			final String standardName = project.getName() + ".gpr";

			String projectFileName = standardName;

			final IWorkbench workbench = PlatformUI.getWorkbench();

		    final IWorkbenchWindow window = workbench.getActiveWorkbenchWindow();

			ISelection selection = window.getSelectionService().getSelection();
			// TODO clean this up and just pass selection...
			IStructuredSelection selectionToPass = StructuredSelection.EMPTY;
			if (selection instanceof IStructuredSelection) {
				selectionToPass = (IStructuredSelection) selection;
			} // if

			boolean useExistingGPR = false;

			final IResource [] gprFiles = FileUtils.rootFilesWithExtension("gpr", project);

			if (gprFiles.length == 1) {
				if (gprFiles[0].getName().equalsIgnoreCase(standardName)) {
					projectFileName = gprFiles[0].getName();
					useExistingGPR = true;
				} else {
					useExistingGPR = MessageDialog.openQuestion(
							window.getShell(),
							dialogTitle,
							"Should we reuse the existing project file named '" + gprFiles[0].getName() + "'?");
					if (useExistingGPR) {
						projectFileName = gprFiles[0].getName();
					} // if
				} // if
			} else if (gprFiles.length > 1) {
		        final String selectedGPR = selectedGPRFile(window, project);
		        if (selectedGPR == null) {
		        	// the user cancelled the dialog
		        	useExistingGPR = false;
		        } else {
		        	projectFileName = selectedGPR;
		        	useExistingGPR = true;
		        } // if
			} // if

			final IFile makefile = project.getFile(new Path (IMakefile.MAKEFILE_NAME));
			final boolean existingMakefile = makefile.exists();

			if (existingMakefile && !useExistingGPR) {
				// we cannot rely on the existing makefile to match the new gpr file
				// that will be created by the wizard (even though it is based on
				// user input), so we save a copy of the existing makefile
				try {
					makefile.copy(new Path (IMakefile.MAKEFILE_NAME + "_original"), true, null);
					makefile.delete(true, null);  // no rename???
					completionMsg = completionMsg +
						"\nNote the existing Makefile was renamed to Makefile_original.";
				} catch (CoreException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
					DisplayUtils.displayError(dialogTitle,
							"Could not copy existing Makefile to Makefile_original.\n" +
							"Rename Makefile_original and try again.");
					return;
				} // try
			} // if

			/// invoke the wizard if we need to generate a new makefile and .gpr file
			if (!useExistingGPR) {
				ConvertedAdaProjectWizard wizard = new ConvertedAdaProjectWizard();

				wizard.init(workbench, selectionToPass);
				wizard.setForcePreviousAndNextButtons(true);

				WizardDialog dialog = new WizardDialog(null, wizard);
				dialog.create();
				dialog.setBlockOnOpen(true);

				int result = dialog.open();

				if (result != Window.OK) {
					// the user cancelled out of the wizard
					return;
				} // if

				projectFileName = standardName;
			} // if

			// add our nature
			try {
				GNATbenchProjectNature.addNature(project);
			} catch (CoreException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
				DisplayUtils.displayError(dialogTitle,
					"Could not add GNATbench nature");
				// TODO: remove makefile and project file
				return;
			} // try

			// set the project file property and that it is a root project
			GNATbenchProjectProperties props = GNATbenchProjectProperties.getPropertiesFor(project);
			props.setRootProject(true);
			props.setGPRPath(projectFileName);
			props.saveProjectFile();

			// if there was no makefile but they want to reuse an existing gpr file,
			// we must generate a new makefile (instead of the wizard doing it).
			// NB: createMakefile uses the value set by setGPRPath to get the builder
			// name from the .gpr file, so the call to setGPRPath must come first.
			if (!existingMakefile && useExistingGPR) {
				if (!createMakefile(project)) {
					DisplayUtils.displayError(dialogTitle,
					"Could not create makefile");
					return;
				} // if
			} // if

			MessageDialog.openInformation(
					window.getShell(),
					dialogTitle,
					completionMsg);

		} // for
	} // run


	protected String selectedGPRFile(final IWorkbenchWindow window, final IProject project) {
		FileDialog fd = new FileDialog(window.getShell(), SWT.OPEN);
        fd.setText("Select An Existing GNAT Project File");
        fd.setFilterPath(project.getLocation().toString());
        final String[] filterExt = {"*.gpr"};
        fd.setFilterExtensions(filterExt);
        return fd.open();
	} // selectedGPRFile


	protected boolean createMakefile(final IProject project) {
		final IFile file = project.getFile(new Path ("/" + IMakefile.MAKEFILE_NAME));

		final String makeFile = GNATbenchIncrementalBuilder.makefileForGPR();

		try {
			file.create(new ByteArrayInputStream(makeFile.getBytes()), true,
					null);
		} catch (CoreException e) {
			DisplayUtils.displayError("Convert To GNATbench Project",
					"Inserting makefile failed");
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return false;
		} // try

		return true;
	} // createMakefile


	public void selectionChanged(IAction action, ISelection selection) {
		fSelection = (StructuredSelection) selection;
	} // selectionChanged


} // GNATbenchProjectConvertor
