with Csets;
with Namet;            use Namet;
with Snames;
with Prj;              use Prj;
with Prj.Pars;
with Prj.Util;         use Prj.Util;
with Ada.Command_Line; use Ada.Command_Line;
with Ada.Text_IO;      use Ada.Text_IO;
with GNAT.OS_Lib;      use GNAT.OS_Lib;
with GNAT.Expect;      use GNAT.Expect;
with Ada.Exceptions;   use Ada.Exceptions;

with Ada.Containers.Vectors;
pragma Elaborate_All (Ada.Containers.Vectors);

procedure GNATWrapper is

   Version               : constant String := "2.0.1-20071116";
   --  tool version

   Project_Tree          : constant Project_Tree_Ref := new Project_Tree_Data;
   App_Project_Id        : Project_Id;

   --  the name of the source file used as the main procedure
   Selected_Main_Unit    : String_Access;

   --  the name of the project file specified on the command line
   Project_File_Name     : String_Access;

   --  the Workbench destination when copying the GNAT-produced executable file
   Output_File           : String_Access;

   --  the value of the Executable attribute in the project file that
   --  specifies the file name of the executable produced by the compiler
   Executable_Name       : String_Access;

   Gnatmake_Command      : String_Access;
   Gnatmake_Command_Path : String_Access;

   Executable_Suffix     : String_Access;

   Gnatls_Invocation     : constant String_List_Access :=
      new String_List (1 .. 2);

   Make_Result           : Integer;
   Successful_Copy       : Boolean;
   Is_RTP                : Boolean := False;

   GNAT_Version          : String_Access;

   package Ragged_Strings is
      new Ada.Containers.Vectors (Natural, String_Access);
   use Ragged_Strings;

   --  The args that are required for gnatmake, independent of the RTS
   --  indicated by the build-spec.
   Common_Args : Vector;

   --  The args required for the specific build-spec.
   RTS_Args : Vector;

   --  The optional -margs args passed directly to gnatmake
   Optional_Args  : Vector;

   --  The status codes returned via OS_Exit. Except for zero, these are
   --  arbitrary.  None are used by application code because gnatwrapper
   --  is strictly internal to the AdaCore builder.
   Success                : constant := 0;
   No_Toolchain           : constant := 1;
   Unknown_Error          : constant := 2;
   No_Specific_Main       : constant := 3;
   No_Main_Unit           : constant := 4;
   GNATLS_Error           : constant := 5;
   GNAT_Version_Failure   : constant := 6;
   Copy_Failure           : constant := 7;
   Malformed_Project_File : constant := 8;
   Switch_Error           : constant := 127;

   --  Convert a List to a String_List
   function As_String_List (This : Vector)
      return String_List;

   --  Strip all carriage returns from the input string
   function Strip_CR (Text : String) return String;

   --  Correct the extension returned from the project facility.
   --  Necessary since the project facility appends the extension of the
   --  host compiler (if any), instead of the cross compiler.
   procedure Correct_Executable_File_Name
      (File_Name         : in out String_Access;
       Executable_Suffix :        String_Access;
       Is_RTP            :        Boolean);

   --  Parse command-line switches
   procedure Parse_Switches;

   --  Get the name of the executable file and the name of the source file
   --  containing the main program from the project
   procedure Get_Exe_Name
      (PID                  :        Project_Id;
       Executable_File_Name :    out String_Access;
       Selected_Main_Name   : in out String_Access);

   --  Get the value of the executable_suffix attribute if it is specified
   --  in the project file.  Sets Executable_Suffix to null if not specified.
   procedure Get_Exe_Suffix
      (PID               :     Project_Id;
       In_Tree           :     Project_Tree_Ref;
       Executable_Suffix : out String_Access);

   --  Runs "<cross-compiler-name>-gnatls -v" to get the version of GNAT
   --  installed, to determine if we can use the -eS switch.
   --  The first entry in Gnatls_Invocation is the name of "gnatls" that is
   --  specific to the compiler installed: it might be:
   --     powerpc-wrs-vxworks-gnatls
   --  or
   --     i586--wrs-vxworks-gnatls
   --  depending on the build spec pass to gnatwrapper.
   procedure Get_GNAT_Version
     (Gnatls_Invocation :     Argument_List_Access;
      GNAT_Version      : out String_Access);

   ----------------------
   -- Get_GNAT_Version --
   ----------------------

   procedure Get_GNAT_Version
     (Gnatls_Invocation :     Argument_List_Access;
      GNAT_Version      : out String_Access)
   is
      --  see procedure Compute_Predefined_Paths in package Projects.Registry
      --  (the body) located gps\kernel\src_info\projects-registry.adb

      Current : String_Access := new String'("");

      Result  : Expect_Match;
      Success : Boolean;
      Fd      : Process_Descriptor_Access;

   begin
      begin
         Success := True;

         declare
            Gnatls_Path : String_Access :=
               Locate_Exec_On_Path
                  (Gnatls_Invocation (Gnatls_Invocation'First).all);
         begin
            if Gnatls_Path = null then
               Success := False;
            else
               Fd := new Process_Descriptor;
               Non_Blocking_Spawn
                 (Fd.all,
                  Gnatls_Path.all,
                  Gnatls_Invocation (2 .. Gnatls_Invocation'Last),
                  Buffer_Size => 0, Err_To_Out => True);
               Free (Gnatls_Path);
            end if;
         end;
      exception
         when others =>
            Success := False;
      end;

      if not Success then
         Put_Line
           (Standard_Error,
            "Could not invoke the gnatls command");
         GNAT.OS_Lib.OS_Exit (GNATLS_Error);
      end if;

      Expect (Fd.all, Result, "GNATLS .+(\n| )Copyright", Timeout => 5_000);

      declare
         S : constant String := Strip_CR (Expect_Out_Match (Fd.all));
      begin
         GNAT_Version := new String'(S (S'First + 11 .. S'First + 15));
      end;

   exception
      when Process_Died =>
         Free (Current);
         Close (Fd.all);
         Put_Line (Standard_Error, "Could not get GNAT version");
         GNAT.OS_Lib.OS_Exit (GNAT_Version_Failure);
   end Get_GNAT_Version;

   ----------------------------------
   -- Correct_Executable_File_Name --
   ----------------------------------

   procedure Correct_Executable_File_Name
      (File_Name         : in out String_Access;
       Executable_Suffix :        String_Access;
       Is_RTP            :        Boolean)
   is
      --  This routine corrects the file name extension of the executable
      --  from what the project facility reports into what is actually used
      --  by the compiler.  This is necessary because the project facility
      --  indicates the host's executable extension unless the optional
      --  attribute Builder'Executable_Suffix is specified in the user's
      --  project file.

      --  If the attribute *is* specified we need do nothing. Otherwise, the
      --  approach is to strip off the host-oriented extension, if any, and
      --  then append one that depends on the build spec specified as a
      --  parameter to gnatwrapper. We have to be careful to correctly handle
      --  names that include dots that do not precede extensions, and we must
      --  also handle the case in which the host name does not have an
      --  extension.

      --  the index of the right-most '.' in File_Name, if any
      Dot_Pos          : Natural := 0;

      --  These are the known host executable file name extensions to use when
      --  looking for extensions in a candidate executable name.
      Known_Extensions : constant String_List :=
                           (new String'(""),
                            new String'(".exe"));

   begin
      --  if the suffix is defined, the project facility will return the
      --  right file name so there is nothing to correct.
      if Executable_Suffix /= null then
         return;
      end if;

      --  Locate the right-most '.' in the file name, which may or may not be
      --  part of an extension.
      for K in reverse File_Name.all'Range loop
         if File_Name (K) = '.' then
            Dot_Pos := K;
            exit;
         end if;
      end loop;

      if Dot_Pos /= 0 then
         --  Found a possible extension. We test it to see if it is a real,
         --  recognized extension and not just a part of the name itself. If it
         --  is a known extension we remove it because it came from the project
         --  facility.

         declare
            Proposed_Extension : constant String :=
              File_Name (Dot_Pos .. File_Name'Last);
            Basename           : constant String :=
              File_Name (File_Name'First .. Dot_Pos - 1);
         begin
            for K in Known_Extensions'Range loop
               if Proposed_Extension = Known_Extensions (K).all then
                  --  We have found a known extension at the end of the name.
                  --  Remove it.

                  File_Name := new String'(Basename);
                  exit;
               end if;
            end loop;
         end;
      end if;

      --  At this point, File_Name.all has no extension. It may have a dot
      --  inside the name, but not for an extension.  Now we can append the
      --  proper extension based on the build spec.

      if Is_RTP then
         File_Name := new String'(File_Name.all & ".vxe");
      else
         File_Name := new String'(File_Name.all & ".out");
      end if;
   end Correct_Executable_File_Name;

   --------------------
   -- Parse_Switches --
   --------------------

   procedure Parse_Switches is
      Current_Arg : Positive := 1;
   begin
      loop
         exit when Current_Arg > Argument_Count;
         if Argument (Current_Arg) = "-m" then
            Current_Arg := Current_Arg + 1;
            if Current_Arg > Argument_Count then
               Put_Line
                  (Standard_Error,
                   "Error in makefile: no main unit specified with -m switch");
               GNAT.OS_Lib.OS_Exit (Switch_Error);
            end if;
            Selected_Main_Unit := new String'(Argument (Current_Arg));
         elsif Argument (Current_Arg) = "-s" then
            Current_Arg := Current_Arg + 1;
            if Current_Arg > Argument_Count then
               Put_Line
                  (Standard_Error,
                   "Error in makefile: no spec specified with -s switch");
               GNAT.OS_Lib.OS_Exit (Switch_Error);
            end if;
            if Argument (Current_Arg)'Length >= 3 and then
              Argument (Current_Arg) (1 .. 3) = "PPC"
            then
               Gnatmake_Command := new String'("powerpc-wrs-vxworks-gnatmake");
               Gnatls_Invocation (1) :=
                  new String'("powerpc-wrs-vxworks-gnatls");
            elsif (Argument (Current_Arg)'Length >= 10 and then
                   Argument (Current_Arg) (1 .. 10) = "SIMPENTIUM")
              or else
                (Argument (Current_Arg)'Length >= 5 and then
                 Argument (Current_Arg) (1 .. 5) = "SIMNT")
            then
               Gnatmake_Command := new String'("i586-wrs-vxworks-gnatmake");
               Gnatls_Invocation (1) := new String'("i586-wrs-vxworks-gnatls");
            else
               Gnatmake_Command := new String'("gnatmake");
               Gnatls_Invocation (1) := new String'("gnatls");
            end if;

            if Argument (Current_Arg)'Length >= 4 and then
              Argument (Current_Arg)
              (Argument (Current_Arg)'Last - 3 .. Argument (Current_Arg)'Last)
              = "_RTP"
            then
               Is_RTP := True;
            end if;
         elsif Argument (Current_Arg) = "-o" then
            Current_Arg := Current_Arg + 1;
            if Current_Arg > Argument_Count then
               Put_Line
                 (Standard_Error,
                 "Error in makefile: no output file specified with -o switch");
               GNAT.OS_Lib.OS_Exit (Switch_Error);
            end if;
            Output_File := new String'(Argument (Current_Arg));
         elsif Argument (Current_Arg) = "-margs" then
            while Current_Arg <= Argument_Count loop
               Optional_Args.append (new String'(Argument(Current_Arg)));
               Current_Arg := Current_Arg + 1;
            end loop;
            --  These are the last args, by definition, so we are finished
            --  parsing the switches because Current_Arg controls the loop.
         elsif Argument (Current_Arg) = "-v" then
            Ada.Text_IO.Put_Line (Version);
            GNAT.OS_Lib.OS_Exit (Success);
         else
            if Project_File_Name = null then
               Project_File_Name := new String'(Argument (Current_Arg));
            else
               Put_Line
                 (Standard_Error,
                 "Error in makefile: invalid switch " &
                 Argument (Current_Arg));
            end if;
         end if;
         Current_Arg := Current_Arg + 1;
      end loop;
   end Parse_Switches;

   ------------------
   -- Get_Exe_Name --
   ------------------

   procedure Get_Exe_Name
      (PID                  :        Project_Id;
       Executable_File_Name :    out String_Access;
       Selected_Main_Name   : in out String_Access)
   is
      Value    : String_List_Id := Project_Tree.Projects.Table (PID).Mains;
      Exec_Dir : constant String  := Get_Name_String
        (Project_Tree.Projects.Table (PID).Exec_Directory);
   begin
      if Value = Prj.Nil_String then
         Put_Line (Standard_Error, "no main unit specified in project file");
         GNAT.OS_Lib.OS_Exit (No_Main_Unit);
      end if;

      while Value /= Prj.Nil_String loop
         declare
            Current_Main_Unit  : constant String := Namet.Get_Name_String
              (Project_Tree.String_Elements.Table (Value).Value);
            Current_Executable : constant String := Namet.Get_Name_String
              (Executable_Of
                 (PID,
                  Project_Tree,
                  File_Name_Type
                    (Project_Tree.String_Elements.Table (Value).Value),
                  0));
         begin
            Value := Project_Tree.String_Elements.Table (Value).Next;
            if (Selected_Main_Name /= null and then
                  Selected_Main_Name.all = Current_Main_Unit) or else
              (Selected_Main_Name = null and then Value = Prj.Nil_String)
            then
               if GNAT.OS_Lib.Is_Absolute_Path (Current_Executable) then
                  Executable_File_Name := new String'(Current_Executable);
               else
                  Executable_File_Name := new String'
                    (Exec_Dir & "/" & Current_Executable);
               end if;
               if Selected_Main_Name = null then
                  Selected_Main_Name := new String'(Current_Main_Unit);
               end if;
               exit;
            end if;

            if Selected_Main_Name = null then
               Put_Line
                  (Standard_Error, "this project contains several mains");
               Put_Line
                 (Standard_Error,
                  "specify one using the MAIN attribute" &
                  " in your GNAT project properties");
               GNAT.OS_Lib.OS_Exit (No_Specific_Main);
            end if;
         end;
      end loop;

      if Executable_File_Name = null then
         Put_Line
          (Standard_Error, "no main unit named " & Selected_Main_Name.all);
         GNAT.OS_Lib.OS_Exit (No_Main_Unit);
      end if;
   end Get_Exe_Name;

   --------------------
   -- Get_Exe_Suffix --
   --------------------

   procedure Get_Exe_Suffix
      (PID               :     Project_Id;
       In_Tree           :     Project_Tree_Ref;
       Executable_Suffix : out String_Access)
   is
      use Snames;

      The_Packages : constant Package_Id :=
                       In_Tree.Projects.Table (PID).Decl.Packages;

      Builder_Package : constant Prj.Package_Id :=
                          Prj.Util.Value_Of
                            (Name_Builder,
                             The_Packages,
                             In_Tree);

      Result : Variable_Value := Nil_Variable_Value;

   begin
      if Builder_Package /= No_Package then
         Result := Prj.Util.Value_Of
           (Name_Executable_Suffix,
            In_Tree.Packages.Table (Builder_Package).Decl.Attributes,
            In_Tree);

         if Result.Default then
            Executable_Suffix := null;
         else
            Executable_Suffix := new String'(Get_Name_String (Result.Value));
         end if;

      else
         Executable_Suffix := null;
      end if;
   end Get_Exe_Suffix;

   --------------------
   -- As_String_List --
   --------------------

   function As_String_List (This : Vector)
      return String_List
   is
      Result : String_List (1 .. Integer (This.Length));
      C : Cursor := This.First;
   begin
      for K in 1 .. Result'Length loop
         Result (K) := Element (C);
         Next (C);
      end loop;
      return Result;
   end As_String_List;

   --------------
   -- Strip_CR --
   --------------

   function Strip_CR (Text : String) return String is
      To       : String (1 .. Text'Length);
      Index_To : Positive := 1;
   begin
      for Index in Text'Range loop
         if Text (Index) /= ASCII.CR then
            To (Index_To) := Text (Index);
            Index_To := Index_To + 1;
         end if;
      end loop;

      return To (1 .. Index_To - 1);
   end Strip_CR;

begin
   Parse_Switches;
   --  Note that the program may exit (normally or in error) based on the
   --  switches parsed. For example, the switch "-v" will cause the version
   --  number to be printed and then the program exits.

   if Project_File_Name = null then
      Put_Line
        (Standard_Error, "Error in makefile: no project file specified");
      GNAT.OS_Lib.OS_Exit (Switch_Error);
   end if;

   if Output_File = null then
      Put_Line (Standard_Error,
      "Error in makefile: no output file specified with -o switch");
      GNAT.OS_Lib.OS_Exit (Switch_Error);
   end if;

   if Gnatmake_Command = null then
      Put_Line
        (Standard_Error,
         "Error in makefile: no build spec specified with -s switch");
      GNAT.OS_Lib.OS_Exit (Switch_Error);
   end if;

   Gnatmake_Command_Path := Locate_Exec_On_Path (Gnatmake_Command.all);
   if Gnatmake_Command_Path = null then
      Put_Line
        (Standard_Error,
         "GNAT Pro toolchain is not in your path (cannot find " &
         Gnatmake_Command.all & " command)");
      GNAT.OS_Lib.OS_Exit (No_Toolchain);
   end if;

   Gnatls_Invocation (2) := new String'("-v");
   Get_GNAT_Version (Gnatls_Invocation, GNAT_Version);

   if GNAT_Version(GNAT_Version'First) >= '6' then
      --  The arg "-eS" causes gnatmake to write to stdout instead of
      --  stderr, so that Workbench does not display normal output as errors.
      Common_Args.Append (new String'("-eS"));
   end if;

   Common_Args.Append (new String'("-P"));
   Common_Args.Append (new String'(Project_File_Name.all));

   if Is_RTP then
      RTS_Args.Append (new String'("-mrtp"));
      RTS_Args.Append (new String'("--RTS=rtp"));
      RTS_Args.Append (new String'("-largs"));
      RTS_Args.Append (new String'("-r"));
      RTS_Args.Append (new String'("-nostdlib"));
   else
      RTS_Args.Append (new String'("--RTS=kernel"));
   end if;

   --  Initialize project facilities
   Csets.Initialize;
   Namet.Initialize;
   Snames.Initialize;
   Prj.Initialize (Project_Tree);

   begin
      --  Parse the GNAT project file
      Prj.Pars.Parse
        (Project => App_Project_Id,
         In_Tree => Project_Tree,
         Project_File_Name => Project_File_Name.all);

      --  Get the executable name
      Get_Exe_Name (App_Project_Id, Executable_Name, Selected_Main_Unit);

      --  Get the suffix, if specified, to use when correcting the name
      Get_Exe_Suffix (App_Project_Id, Project_Tree, Executable_Suffix);
   exception
      when others =>
         Put_Line (Standard_Error, "Could not get parse the project file");
         GNAT.OS_Lib.OS_Exit (Malformed_Project_File);
   end;

   Common_Args.Append (Selected_Main_Unit);

   Make_Result := GNAT.OS_Lib.Spawn
     (Gnatmake_Command_Path.all,
      As_String_List (Common_Args & RTS_Args & Optional_Args));

   if Make_Result /= 0 then
      GNAT.OS_Lib.OS_Exit (Make_Result);
   end if;

   --  Now that GNAT has produced the executable we can correct the file name.
   --  We need to correct the extension in the name returned by the project
   --  facility because the project facility is, by default, appending the
   --  extension of the host rather than what the cross compiler actually used.
   --  If Executable_Suffix *is* defined in the project file the project
   --  facility will use it, but that attribute is optional.

   Correct_Executable_File_Name (Executable_Name, Executable_Suffix, Is_RTP);

   --  Copy the executable to the location expected by Workbench
   Put_Line ("copying " & Executable_Name.all & " to " & Output_File.all);
   Copy_File
     (Executable_Name.all,
      Output_File.all,
      Successful_Copy,
      Overwrite);

   if not Successful_Copy then
      Put_Line
        (Standard_Error,
         "copy of Ada object to Workbench object directory failed");
      GNAT.OS_Lib.OS_Exit (Copy_Failure);
   end if;
exception
   when E : others =>
      Put_Line
        (Standard_Error,
         "Unexpected exception in GNATwrapper! Please report the following:");
      Put_Line
        (Standard_Error,
         Exception_Information(E));
      GNAT.OS_Lib.OS_Exit (Unknown_Error);
         --  we still use OS_Exit so the makefile will stop executing
end GNATWrapper;
