------------------------------------------------------------------------------
--                                                                          --
--                  GNAT RUN-TIME LIBRARY (GNARL) COMPONENTS                --
--                                                                          --
--              S Y S T E M . B B . C P U _ P R I M I T I V E S .           --
--                       M U L T I P R O C E S S O R S                      --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                       Copyright (C) 2014, AdaCore                        --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.                                     --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
-- You should have received a copy of the GNU General Public License and    --
-- a copy of the GCC Runtime Library Exception along with this program;     --
-- see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see    --
-- <http://www.gnu.org/licenses/>.                                          --
--                                                                          --
------------------------------------------------------------------------------

pragma Restrictions (No_Elaboration_Code);

with Interfaces; use Interfaces;
with System.Multiprocessors; use System.Multiprocessors;
with System.BB.Interrupts; use System.BB.Interrupts;
with System.BB.Protection;

package body System.BB.CPU_Primitives.Multiprocessors is

   procedure Poke_Handler (Interrupt : Interrupts.Interrupt_ID);

   -----------------------------------------------------
   -- The PCR register in the Memory Coherency Module --
   -----------------------------------------------------

   MCM_PCR : Unsigned_32;
   for MCM_PCR'Address use System'To_Address (16#f800_1010#);
   pragma Import (Ada, MCM_PCR);

   -----------------
   -- Current_CPU --
   -----------------

   function Current_CPU return System.Multiprocessors.CPU is
      Cpu_Id : Unsigned_32;
      for Cpu_Id'Address use System'To_Address (16#f804_0090#);
      pragma Import (Ada, Cpu_Id);
   begin
      return CPU (Cpu_Id) + CPU'First;
   end Current_CPU;

   --------------
   -- Poke_CPU --
   --------------

   procedure Poke_CPU (CPU_Id : System.Multiprocessors.CPU) is
      Val : Unsigned_32;
      IPI : Unsigned_32;
      for IPI'Address use System'To_Address (16#f804_0040#);
      pragma Import (Ada, IPI);
   begin
      Val := Shift_Left (1, CPU'Pos (CPU_Id - CPU'First));
      IPI := IPI or Val;
   end Poke_CPU;

   ---------------
   -- Start_CPU --
   ---------------

   procedure Start_CPU (CPU_Id : System.Multiprocessors.CPU) is
      Val : Unsigned_32;
   begin
      Val := Shift_Left (1, 24 + CPU'Pos (CPU_Id - CPU'First));
      MCM_PCR := MCM_PCR or Val;
   end Start_CPU;

   ------------------
   -- Poke_Handler --
   ------------------

   procedure Poke_Handler (Interrupt : Interrupts.Interrupt_ID) is
      Bogus : Unsigned_32;
      pragma Unreferenced (Bogus);
      IACK  : Unsigned_32;
      for IACK'Address use System'To_Address (16#f804_00a0#);
      pragma Import (Ada, IACK);
      EOI   : Unsigned_32;
      for EOI'Address use System'To_Address (16#f804_00b0#);
      pragma Import (Ada, EOI);
   begin
      --  Make sure we are handling the right interrupt
      pragma Assert (Interrupt = 1);

      --  Acknowledge the interrupt by reading the OpenPIC IACK register
      Bogus := IACK;

      --  Just so that we can call a Leave_Kernel
      Protection.Enter_Kernel;

      --  Leave_Kernel will handle served entries and wake up their calling
      --  threads
      Protection.Leave_Kernel;

      --  Disable interrupts in order to write to EOI register
      CPU_Primitives.Disable_Interrupts;
      EOI := 0;

   end Poke_Handler;

   --------------------
   -- Start_All_CPUs --
   --------------------

   procedure Start_All_CPUs is
      Val : Unsigned_32;
      IPIVPR : Unsigned_32;
      for IPIVPR'Address use System'To_Address (16#f804_10a0#);
      pragma Import (Ada, IPIVPR);
      PCTP : Unsigned_32;
      for PCTP'Address use System'To_Address (16#f806_1080#);
      pragma Import (Ada, PCTP);
   begin
      Attach_Handler (Poke_Handler'Access, 1, Interrupt_Priority'First);
      PCTP := 0;
      IPIVPR := (IPIVPR or 16#000f_0000#) and (not 16#8000_0000#);

      Val := Shift_Left (Shift_Left (1, CPU'Pos (Number_Of_CPUs)) - 1, 24);
      MCM_PCR := MCM_PCR or Val;
   end Start_All_CPUs;

end System.BB.CPU_Primitives.Multiprocessors;
