------------------------------------------------------------------------------
--                                                                          --
--                          GNAT SYSTEM UTILITIES                           --
--                                                                          --
--                               X L E A P S                                --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--          Copyright (C) 1992-2012, Free Software Foundation, Inc.         --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.                                     --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- Extensive contributions were provided by Ada Core Technologies Inc.      --
--                                                                          --
------------------------------------------------------------------------------

--  This utility is used to generate the hard time values of all existing
--  leap second occurences. It uses several target dependent values taken
--  from a-calend.ads and a-calend-vms.ads to perform its computations.
--  The algorithm behind this utility calculates the total number of days
--  upto a leap second occurence from a target dependent origin of time.
--  The program then converts the resulting days into target dependent
--  time units used to create the aggregate.

--    Usage:

--      xleaps [vms]

--    Output files:

--      xleaps.txt

--  When a new leap second is introduced, the following steps must be carried
--  out:

--     1) Increment Leap_Seconds_Count in a-calend.adb and a-calend-vms.adb
--        by one
--     2) Increment LS_Count in xleaps.adb by one
--     3) Add the new date to the aggregate of array LS_Dates in xleaps.adb
--     4) Compile and execute xleaps for common targets and VMS
--     5) Replace the values of array Leap_Second_Times in a-calend.adb and
--        a-calend-vms.adb with the aggregate generated by xleaps
--     6) Update the following tests under 0005-144:
--          cy90006.adb
--          cy90006_ls.adb
--          cy90010.adb
--          cy90010_ls.adb
--          c790016.adb
--          cy90016_ls.adb

with Ada.Calendar;     use Ada.Calendar;
with Ada.Command_Line; use Ada.Command_Line;
with Ada.Text_IO;      use Ada.Text_IO;

procedure XLeaps is

   --  A general type used for calculating the hard time values of a leap
   --  second occurence. The type is explicitly declared to be large enough
   --  and system independent.

   type Units is range -2 ** 63 .. +2 ** 63 - 1;

   --  Target specific units. VMS units are 100 nanoseconds or "milis", other
   --  targets use nanoseconds.

   Mili         : constant := 10_000_000;
   Milis_In_Day : constant := 864_000_000_000;
   Nano         : constant := 1_000_000_000;
   Nanos_In_Day : constant := 86_400_000_000_000;

   --  Origins of different time systems relative to a zero value. See
   --  a-calend and a-calend-vms for more details.

   Origin_Common : constant := -(61 * 366 + 188 * 365);
   Origin_VMS    : constant :=   10 * 366 +  32 * 365 + 45;

   --  Various constants

   Ada_Min_Year       : constant := 1901;
   Days_In_Four_Years : constant := 365 * 3 + 366;
   Secs_In_Day        : constant := 86_400;

   Cumulative_Days_Before_Month :
     constant array (Month_Number) of Units :=
       (0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334);

   type LS_Date is record
      Year  : Year_Number;
      Month : Month_Number;
      Day   : Day_Number;
   end record;

   LS_Count : constant := 25;
   LS_Dates : constant array (1 .. LS_Count) of LS_Date :=
     ((1972,  6, 30), (1972, 12, 31), (1973, 12, 31), (1974, 12, 31),
      (1975, 12, 31), (1976, 12, 31), (1977, 12, 31), (1978, 12, 31),
      (1979, 12, 31), (1981,  6, 30), (1982,  6, 30), (1983,  6, 30),
      (1985,  6, 30), (1987, 12, 31), (1989, 12, 31), (1990, 12, 31),
      (1992,  6, 30), (1993,  6, 30), (1994,  6, 30), (1995, 12, 31),
      (1997,  6, 30), (1998, 12, 31), (2005, 12, 31), (2008, 12, 31),
      (2012,  6, 30));

   Days       : Units;
   Leap       : LS_Date;
   Origin     : Units;
   Result     : Units;
   Time_Units : Units;
   Years      : Natural;

   --  All hard time values will be written to this file

   F : File_Type;

   function Is_Leap (Year : Year_Number) return Boolean;
   --  Determine whether a given year is leap

   procedure Usage;
   --  Generate screen of usage information

   -------------
   -- Is_Leap --
   -------------

   function Is_Leap (Year : Year_Number) return Boolean is
   begin
      --  Leap centenial years

      if Year mod 400 = 0 then
         return True;

      --  Non-leap centenial years

      elsif Year mod 100 = 0 then
         return False;

      --  Regular years

      else
         return Year mod 4 = 0;
      end if;
   end Is_Leap;

   -----------
   -- Usage --
   -----------

   procedure Usage is
   begin
      Put_Line ("Usage : xleaps [vms]");
      Put_Line ("Output: xleaps.txt");
   end Usage;

--  Start of processing for XLeaps

begin
   --  Determine the target and set the appropriate time origin and units

   if Argument_Count = 0 then
      Origin := Origin_Common;
      Time_Units := Nano;

   elsif Argument_Count = 1
     and then Argument (1) = "vms"
   then
      Origin := Origin_VMS;
      Time_Units := Mili;

   else
      Usage;
   end if;

   Create (F, Out_File, "xleaps.txt");

   for Index in 1 .. LS_Count loop
      Leap := LS_Dates (Index);

      --  Add the segments of four years expressed as days until the current
      --  leap second occurence. Note that non-leap centenial years are not
      --  accounted for since there are no leap seconds after 2100.

      Years := Leap.Year - Ada_Min_Year;
      Days  := Units (Years / 4) * Days_In_Four_Years;
      Years := Years mod 4;

      --  Add the remaining years as days

      if Years = 1 then
         Days := Days + 365;

      elsif Years = 2 then
         Days := Days + 365 * 2;

      elsif Years = 3 then
         Days := Days + 365 * 3;
      end if;

      --  Add all the days to the beginning of the current month. Handle
      --  February 29 during leap years.

      Days := Days + Cumulative_Days_Before_Month (Leap.Month);

      if Is_Leap (Leap.Year)
        and then Leap.Month > 2
      then
         Days := Days + 1;
      end if;

      --  Add all the days into the current month and shift the origin to the
      --  target time system.

      Days := Days + Units (Leap.Day) + Origin;

      --  Finally convert all days calculated above into seconds and add all
      --  leap seconds occured upto the current one (Index - 1). The seconds
      --  are then converted to target dependent units.

      Result := (Days * Secs_In_Day + Units (Index - 1)) * Time_Units;

      --  First line of the output file

      if Index = 1 then
         Put_Line (F, "     (" & Result'Img & ",");

      --  Last line of the output file

      elsif Index = LS_Count then
         Put_Line (F, "      " & Result'Img & ");");

      --  In between lines

      else
         Put_Line (F, "      " & Result'Img & ",");
      end if;
   end loop;

   Close (F);
end XLeaps;
