------------------------------------------------------------------------------
--                                                                          --
--                  GNAT RUN-TIME LIBRARY (GNARL) COMPONENTS                --
--                                                                          --
--                         S Y S T E M . B B . T I M E                      --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--        Copyright (C) 1999-2002 Universidad Politecnica de Madrid         --
--             Copyright (C) 2003-2005 The European Space Agency            --
--                     Copyright (C) 2003-2011, AdaCore                     --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.                                     --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
-- You should have received a copy of the GNU General Public License and    --
-- a copy of the GCC Runtime Library Exception along with this program;     --
-- see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see    --
-- <http://www.gnu.org/licenses/>.                                          --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University.       --
-- Extensive contributions were provided by Ada Core Technologies, Inc.     --
--                                                                          --
-- The porting of GNARL to bare board  targets was initially  developed  by --
-- the Real-Time Systems Group at the Technical University of Madrid.       --
--                                                                          --
------------------------------------------------------------------------------

pragma Restrictions (No_Elaboration_Code);

with System.BB.Interrupts;
with System.BB.Peripherals;
with System.BB.Protection;
with System.BB.Parameters;
with System.BB.Threads.Queues;
with System.BB.Timing_Events;
with System.BB.CPU_Primitives.Multiprocessors;
with System.Multiprocessors.Fair_Locks;
with System.Multiprocessors.Spin_Locks;

package body System.BB.Time is

   use Peripherals;
   use System.BB.CPU_Primitives.Multiprocessors;
   use System.Multiprocessors;
   use System.Multiprocessors.Fair_Locks;
   use System.Multiprocessors.Spin_Locks;

   --  We use two timers with the same frequency:
   --     A Periodic Timer for the clock
   --     An Alarm Timer for delays

   -----------------------
   -- Local definitions --
   -----------------------

   Alarm_Lock : Fair_Lock := (Spinning => (others => False),
                              Lock     => (Flag   => Unlocked));
   --  Used to protect access to shared alarm resources
   --  (Timer configuration and Pending_Alarm variable)

   type Clock_Periods is mod 2 ** 32;
   for Clock_Periods'Size use 32;
   --  Values of this type represent number of times that the clock finishes
   --  its countdown. This type should allow atomic reads and updates.

   Period_Counter : Clock_Periods := 0;
   pragma Atomic (Period_Counter);
   --  The Period_Counter determines the most significant part of the clock,
   --  which can be retrieved by the MSP_Clock function. The least significant
   --  part of the clock is held in the hardware clock register. The range of
   --  this register is in Parameters.Timer_Interval'Range, but may be less.
   --  Compute the actual time by adding the MSP_Clock and the hardware clock.

   --  We need to make Period_Counter atomic, because it is updated by the
   --  Clock_Handler. If we would read the Clock_Periods in two parts, it might
   --  be updated between both reads, leading to a result that may be either
   --  too high or too low.

   Pending_Alarm : Time := Time'Last;
   --  Time of the current alarm handled by the timer. Used to determine if a
   --  given alarm is before the current one, and so needs to re-configure the
   --  timer.

   -----------------------
   -- Local subprograms --
   -----------------------

   procedure Alarm_Handler (Interrupt : Interrupts.Interrupt_ID);
   --  Handler for the alarm interrupt

   procedure Clock_Handler (Interrupt : Interrupts.Interrupt_ID);
   --  Handler for the clock interrupt

   function MSP_Clock return Time;
   pragma Inline (MSP_Clock);
   --  Compute the most significant part of Clock based on the Period_Counter
   --  and Max_Timer_Interval;

   -------------------
   -- Alarm_Handler --
   -------------------

   procedure Alarm_Handler (Interrupt : Interrupts.Interrupt_ID) is
      Now                : Time;
      Alarm_Time         : Time;
      Next_Alarm_Overall : Time := Time'Last;

      use System.BB.Threads;

   begin
      --  Make sure we are handling the right interrupt and there is an alarm
      --  pending.

      pragma Assert
        (Pending_Alarm /= Time'Last
          and then Interrupt = Peripherals.Alarm_Interrupt_ID);

      Peripherals.Clear_Alarm_Interrupt;

      --  The access to the queues must be protected

      Protection.Enter_Kernel;

      Now := Clock;

      --  Multiprocessor case special processing

      if System.BB.Parameters.Multiprocessor then

         --  Pending_Alarm is not Atomic, so we need to protect access

         Lock (Alarm_Lock);
         Pending_Alarm := Time'Last;
         Unlock (Alarm_Lock);

         --  This is the alarm CPU, we have to wake up the others CPUs with
         --  expired alarms.

         for CPU_Id in CPU loop

            if CPU_Id /= Current_CPU then
               Alarm_Time := Get_Next_Timeout (CPU_Id);

               if Alarm_Time <= Now then

                  --  Alarm expired, wake up the CPU

                  Poke_CPU (CPU_Id);

               else
                  --  Check if this is the next non-expired alarm
                  --  of the overall system.

                  if Alarm_Time < Next_Alarm_Overall then
                     Next_Alarm_Overall := Alarm_Time;
                  end if;
               end if;
            end if;
         end loop;

         if Next_Alarm_Overall /= Time'Last then
            Update_Alarm (Next_Alarm_Overall);
         end if;

      --  Monoprocessor case

      else
         Pending_Alarm := Time'Last;
      end if;

      --  Execute expired events of the current CPU

      Timing_Events.Execute_Expired_Timing_Events (Now);

      --  Wake up our alarms

      System.BB.Threads.Queues.Wakeup_Expired_Alarms;

      Protection.Leave_Kernel;
   end Alarm_Handler;

   -----------
   -- Clock --
   -----------

   function Clock return Time is
      Before : Time;
      Result : Time;
      After  : Time;

   begin
      --  If MSP_Clock gets updated during the following three statements,
      --  Result may be the sum of the old MSP and wrapped LSP, which would
      --  result in a result that precedes the time of entry of this function.
      --  Before and After will be different. Since the most significant part
      --  of the counter is both read and updated atomically, we know that
      --  After represents the time of the clock interrupt.

      Before := MSP_Clock;
      Result := Before + Time (Read_Clock);
      After  := MSP_Clock;

      return (if After > Result then After else Result);
   end Clock;

   -------------------
   -- Clock_Handler --
   -------------------

   procedure Clock_Handler (Interrupt : Interrupts.Interrupt_ID) is
      Next_Alarm         : Time;
      Next_Alarm_Overall : Time := Time'Last;

   begin
      --  Check that we are in the right handler

      pragma Assert (Interrupt = Clock_Interrupt_ID);

      Clear_Clock_Interrupt;

      --  The access to the queues must be protected

      Protection.Enter_Kernel;

      --  The clock timer has finished counting Max_Timer_Interval + 1 hardware
      --  clock ticks, so we increase the Most Significant Part of the Clock
      --  which is kept in memory.

      Period_Counter := Period_Counter + 1;

      if Pending_Alarm = Time'Last then

         --  No pending alarm

         for CPU_Id in CPU loop
            Next_Alarm := Get_Next_Timeout (CPU_Id);

            if Next_Alarm < Next_Alarm_Overall then
               Next_Alarm_Overall := Next_Alarm;
            end if;
         end loop;

         Update_Alarm (Next_Alarm_Overall);
      end if;

      Protection.Leave_Kernel;
   end Clock_Handler;

   -----------------
   -- Delay_Until --
   -----------------

   procedure Delay_Until (T : Time) is
      Now               : Time;
      Self              : Threads.Thread_Id;
      Inserted_As_First : Boolean;

   begin
      Now := Clock;

      Protection.Enter_Kernel;

      Self := Threads.Thread_Self;

      --  Test if the alarm time is in the future

      if T > Now then

         --  Extract the thread from the ready queue. When a thread wants to
         --  wait for an alarm it becomes blocked.

         Self.State := Threads.Delayed;

         Threads.Queues.Extract (Self);

         --  Insert Thread_Id in the alarm queue (ordered by time) and if it
         --  was inserted at head then check if Alarm Time is closer than the
         --  next clock interrupt.

         Threads.Queues.Insert_Alarm (T, Self, Inserted_As_First);

         if Inserted_As_First then
            Update_Alarm (T);
         end if;

      else
         --  If alarm time is not in the future, the thread must yield the CPU

         Threads.Queues.Yield (Self);
      end if;

      Protection.Leave_Kernel;
   end Delay_Until;

   ----------------------
   -- Get_Next_Timeout --
   ----------------------

   function Get_Next_Timeout (CPU_Id : CPU) return Time is
      Alarm_Time : constant Time :=
                     Threads.Queues.Get_Next_Alarm_Time (CPU_Id);
      Event_Time : constant Time := Timing_Events.Get_Next_Timeout (CPU_Id);
   begin
      if Alarm_Time <= Event_Time then
         return Alarm_Time;
      else
         return Event_Time;
      end if;
   end Get_Next_Timeout;

   -----------------------
   -- Initialize_Timers --
   -----------------------

   procedure Initialize_Timers is
   begin
      --  Install clock handler

      Interrupts.Attach_Handler (Clock_Handler'Access, Clock_Interrupt_ID);

      --  Install timer handler

      Interrupts.Attach_Handler (Alarm_Handler'Access, Alarm_Interrupt_ID);
   end Initialize_Timers;

   ---------------
   -- MSP_Clock --
   ---------------

   function MSP_Clock return Time is
   begin
      --  Note: the following multiplication will typically be a shift

      return Time (Period_Counter) * (Time (Max_Timer_Interval) + 1);
   end MSP_Clock;

   -------------------
   --  Update_Alarm --
   -------------------

   procedure Update_Alarm (Alarm : Time) is
      Now             : constant Time := Clock;
      Time_Difference : Time;

   begin
      if Alarm <= Now then

         --  If alarm is in the past, set the minimum timer value so the
         --  interrupt will be triggered as soon as possible.

         Time_Difference := 1;

      else
         Time_Difference := Alarm - Now;
      end if;

      if Time_Difference <= Time (Max_Timer_Interval) then

         --  If next alarm time is closer than a clock period then we need to
         --  program the alarm.

         if System.BB.Parameters.Multiprocessor then

            --  Only for multiprocessor

            Lock (Alarm_Lock);
         end if;

         if Pending_Alarm = Time'Last then
            Peripherals.Set_Alarm (Timer_Interval (Time_Difference));
            Pending_Alarm := Alarm;
         else
            if Alarm < Pending_Alarm then
               Peripherals.Cancel_And_Set_Alarm
                 (Timer_Interval (Time_Difference));
               Pending_Alarm := Alarm;
            end if;
         end if;

         if System.BB.Parameters.Multiprocessor then

            --  Only for multiprocessor

            Unlock (Alarm_Lock);
         end if;
      end if;
   end Update_Alarm;

end System.BB.Time;
