------------------------------------------------------------------------------
--                                                                          --
--                        GNAAMP COMPILER COMPONENTS                        --
--                                                                          --
--                             A D A B K E N D                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                     Copyright (C) 2001-2006, AdaCore                     --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the  Free Software Foundation,  51  Franklin  Street,  Fifth  Floor, --
-- Boston, MA 02110-1301, USA.                                              --
--                                                                          --
-- GNAAMP - The GNAT Ada 95 tool chain for the Rockwell-Collins AAMP family --
--          of microprocessors is maintained by Ada Core Technologies, Inc. --
--          - http://www.gnat.com                                           --
------------------------------------------------------------------------------

--  This is the version of the Back_End package for back ends written in Ada

with Debug;
with Lib;
with Opt;      use Opt;
with Output;   use Output;
with Osint;    use Osint;
with Osint.C;  use Osint.C;
with Switch.C; use Switch.C;
with Types;    use Types;

package body Adabkend is

   use Switch;

   Argument_Count : constant Integer := Arg_Count - 1;
   --  Number of arguments (excluding program name)

   -------------------
   -- Call_Back_End --
   -------------------

   procedure Call_Back_End is
   begin
      if (Opt.Verbose_Mode or Opt.Full_List)
        and then (not Debug.Debug_Flag_7)
      then
         Write_Eol;
         Write_Str (Product_Name);
         Write_Str (", Copyright ");
         Write_Str (Copyright_Years);
         Write_Str (" Ada Core Technologies, Inc.");
         Write_Str (" (http://www.gnat.com)");
         Write_Eol;
         Write_Eol;
      end if;

      Driver (Lib.Cunit (Types.Main_Unit));
   end Call_Back_End;

   ------------------------
   -- Scan_Compiler_Args --
   ------------------------

   procedure Scan_Compiler_Arguments is
      Output_File_Name_Seen : Boolean := False;
      --  Set to True after having scanned the file_name for switch
      --  "-gnatO file_name"

      Next_Arg : Positive := 1;

      procedure Scan_Back_End_Switches (Switch_Chars : String);
      --  Procedure to scan out switches stored in Switch_Chars. The first
      --  character is known to be a valid switch character, and there are no
      --  blanks or other switch terminator characters in the string, so the
      --  entire string should consist of valid switch characters, except that
      --  an optional terminating NUL character is allowed.
      --
      --  If the switch is not valid, control will not return. The switches
      --  must still be scanned to skip the arguments of the "-o" or the
      --  (undocumented) "-dumpbase" switch, by incrementing the Next_Arg
      --  variable. The "-dumpbase" switch is used to set the basename for
      --  GCC dumpfiles, and ignored for GNAAMP.

      ----------------------------
      -- Scan_Back_End_Switches --
      ----------------------------

      procedure Scan_Back_End_Switches (Switch_Chars : String) is
         First : constant Positive := Switch_Chars'First + 1;
         Last  : Natural := Switch_Chars'Last;

      begin
         if Last >= First
           and then Switch_Chars (Last) = ASCII.NUL
         then
            Last := Last - 1;
         end if;

         --  Process any back end switches, returning if the switch does not
         --  affect code generation or falling through if it does, so the
         --  switch will get stored.

         if Switch_Chars (First .. Last) = "o"
            or else Switch_Chars (First .. Last) = "dumpbase"

         then
            Next_Arg := Next_Arg + 1;
            return; -- ignore this switch

         elsif Switch_Chars (First .. Last) = "quiet" then
            return; -- ignore this switch

         elsif Switch_Chars (First .. Last) = "c" then
            return; -- ignore this switch

         --  The -x switch and its language name argument will generally be
         --  ignored by non-gcc back ends (e.g. the GNAAMP back end). In any
         --  case, we save the switch and argument in the compilation switches.

         elsif Switch_Chars (First .. Last) = "x" then
            Lib.Store_Compilation_Switch (Switch_Chars);
            Next_Arg := Next_Arg + 1;

            declare
               Argv : String (1 .. Len_Arg (Next_Arg));

            begin
               Fill_Arg (Argv'Address, Next_Arg);

               if Is_Switch (Argv) then
                  Fail ("language name missing after -x");
               else
                  Lib.Store_Compilation_Switch (Switch_Chars);
               end if;
            end;

            return;

         elsif Is_Back_End_Switch (Switch_Chars) then
            null;

         else
            Fail ("invalid switch: """, Switch_Chars, """");
         end if;

         --  Store any other GCC switches

         Lib.Store_Compilation_Switch (Switch_Chars);
      end Scan_Back_End_Switches;

   --  Start of processing for Scan_Compiler_Args

   begin
      --  Loop through command line arguments, storing them for later access

      while Next_Arg <= Argument_Count loop
         Look_At_Arg : declare
            Argv : String (1 .. Len_Arg (Next_Arg));

         begin
            Fill_Arg (Argv'Address, Next_Arg);

            if Argv'Length = 0 then
               Fail ("Empty argument");
            end if;

            --  If the previous switch has set the Output_File_Name_Present
            --  flag (that is we have seen a -gnatO), then the next argument
            --  is the name of the output object file.

            if Opt.Output_File_Name_Present
              and then not Output_File_Name_Seen
            then
               if Is_Switch (Argv) then
                  Fail ("Object file name missing after -gnatO");
               else
                  Set_Output_Object_File_Name (Argv);
                  Output_File_Name_Seen := True;
               end if;

               --  If the previous switch has set the Search_Directory_Present
               --  flag (that is if we have just seen -I), then the next
               --  argument is a search directory path.

            elsif Search_Directory_Present then
               if Is_Switch (Argv) then
                  Fail ("search directory missing after -I");
               else
                  Add_Src_Search_Dir (Argv);
                  Search_Directory_Present := False;
               end if;

            elsif not Is_Switch (Argv) then -- must be a file name
               Add_File (Argv);

            elsif Is_Front_End_Switch (Argv) then
               Scan_Front_End_Switches (Argv);

            --  All non-front-end switches are back-end switches

            else
               Scan_Back_End_Switches (Argv);
            end if;
         end Look_At_Arg;

         Next_Arg := Next_Arg + 1;
      end loop;
   end Scan_Compiler_Arguments;

end Adabkend;
