------------------------------------------------------------------------------
--                                                                          --
--                 GNAT RUN-TIME LIBRARY (GNARL) COMPONENTS                 --
--                                                                          --
--                   S Y S T E M . O S _ I N T E R F A C E                  --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                     Copyright (C) 2001-2010, AdaCore                     --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.                                     --
--                                                                          --
-- As a special exception under Section 7 of GPL version 3, you are granted --
-- additional permissions described in the GCC Runtime Library Exception,   --
-- version 3.1, as published by the Free Software Foundation.               --
--                                                                          --
-- In particular,  you can freely  distribute your programs  built with the --
-- GNAT Pro compiler, including any required library run-time units,  using --
-- any licensing terms  of your choosing.  See the AdaCore Software License --
-- for full details.                                                        --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University.       --
-- Extensive contributions were provided by Ada Core Technologies, Inc.     --
--                                                                          --
------------------------------------------------------------------------------

--  This is a LynxOS (POSIX Threads) version of this package

pragma Polling (Off);
--  Turn off polling, we do not want ATC polling to take place during
--  tasking operations. It causes infinite loops and other problems.

package body System.OS_Interface is

   use Interfaces.C;

   -----------------
   -- To_Duration --
   -----------------

   function To_Duration (TS : timespec) return Duration is
   begin
      return Duration (TS.tv_sec) + Duration (TS.tv_nsec) / 10#1#E9;
   end To_Duration;

   ------------------------
   -- To_Target_Priority --
   ------------------------

   function To_Target_Priority
     (Prio : System.Any_Priority) return Interfaces.C.int
   is
   begin
      return Interfaces.C.int (Prio);
   end To_Target_Priority;

   -----------------
   -- To_Timespec --
   -----------------

   function To_Timespec (D : Duration) return timespec is
      S : time_t;
      F : Duration;

   begin
      S := time_t (Long_Long_Integer (D));
      F := D - Duration (S);

      --  If F has negative value due to a round-up, adjust for positive F
      --  value.

      if F < 0.0 then
         S := S - 1;
         F := F + 1.0;
      end if;

      return timespec'(tv_sec => S,
                       tv_nsec => long (Long_Long_Integer (F * 10#1#E9)));
   end To_Timespec;

   -------------
   -- sigwait --
   -------------

   function sigwait
     (set :  access sigset_t;
      sig :  access Signal)
      return int
   is
      function sigwaitinfo
        (set   : access sigset_t;
         info  : System.Address) return Signal;
      pragma Import (C, sigwaitinfo, "sigwaitinfo");

   begin
      sig.all := sigwaitinfo (set, Null_Address);

      if sig.all = -1 then
         return errno;
      end if;

      return 0;
   end sigwait;

   --------------------
   -- Get_Stack_Base --
   --------------------

   function Get_Stack_Base (thread : pthread_t) return Address is
      pragma Warnings (Off, thread);

   begin
      return Null_Address;
   end Get_Stack_Base;

   ------------------
   -- pthread_init --
   ------------------

   procedure pthread_init is
   begin
      null;
   end pthread_init;

   -----------------
   -- sigaltstack --
   -----------------

   function sigaltstack
     (ss  : not null access stack_t;
      oss : access stack_t) return int
   is
   begin
      raise Program_Error;
      return -1;
   end sigaltstack;

   -------------------------
   -- pthread_setspecific --
   -------------------------

   function pthread_setspecific
     (key   : pthread_key_t;
      value : System.Address)
     return int
   is
      function st_setspecific
        (key : pthread_key_t;
         value : System.Address)
        return int;
      pragma Import (C, st_setspecific, "st_setspecific");

   begin
      return st_setspecific (key, value);
   end pthread_setspecific;

   -------------------------
   -- pthread_getspecific --
   -------------------------

   function pthread_getspecific
     (key : pthread_key_t) return System.Address
   is
      function st_getspecific
        (key    : pthread_key_t;
         retval : access System.Address)
        return int;
      pragma Import (C, st_getspecific, "st_getspecific");

      retval : aliased System.Address;
      Result : int;

   begin
      Result := st_getspecific (key, retval'Access);
      pragma Assert (Result = 0);
      return retval;
   end pthread_getspecific;

   ------------------------
   -- pthread_key_create --
   ------------------------

   function pthread_key_create
     (key        : access pthread_key_t;
      destructor : destructor_pointer)
     return int
   is
      function st_keycreate
        (destructor : destructor_pointer;
         key        : access pthread_key_t)
        return int;
      pragma Import (C, st_keycreate, "st_keycreate");

   begin
      return st_keycreate (destructor, key);
   end pthread_key_create;

end System.OS_Interface;
