------------------------------------------------------------------------------
--                                                                          --
--                  GNAT RUN-TIME LIBRARY (GNARL) COMPONENTS                --
--                                                                          --
--              S Y S T E M . B B . E X E C U T I O N _ T I M E             --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                     Copyright (C) 2011-2012, AdaCore                     --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.                                     --
--                                                                          --
-- As a special exception under Section 7 of GPL version 3, you are granted --
-- additional permissions described in the GCC Runtime Library Exception,   --
-- version 3.1, as published by the Free Software Foundation.               --
--                                                                          --
-- In particular,  you can freely  distribute your programs  built with the --
-- GNAT Pro compiler, including any required library run-time units,  using --
-- any licensing terms  of your choosing.  See the AdaCore Software License --
-- for full details.                                                        --
--                                                                          --
------------------------------------------------------------------------------

--  Ravenscar version of this package for XtratuM

with System.BB.Parameters;
with System.BB.Board_Support;
with System.BB.Threads.Queues;

with System.OS_Interface;
with Interfaces.C;

package body System.BB.Execution_Time is

   use type System.BB.Time.Time;
   use System.BB.Interrupts;
   use System.BB.Threads;

   -----------------------
   -- Local Definitions --
   -----------------------

   Interrupts_Execution_Time : array (Interrupt_ID) of System.BB.Time.Time :=
     (others => System.BB.Time.Time'First);
   --  Time counter for each interrupt

   CPU_Clock : System.BB.Time.Time := System.BB.Time.Time'First;
   --  Date of the last Interrupt

   Disabled : Boolean := True;
   --  If the CPU clock is disabled the next elapsed time will be discarded (to
   --  handle CPU idle time).

   function Read_Execution_Clock return System.BB.Time.Time;
   --  Get the execution clock which counts only the time when the partition is
   --  active. This is different from Clock when we are on top of a partitioned
   --  system (like in this case).

   ----------------------------
   -- Disable_Execution_Time --
   ----------------------------

   procedure Disable_Execution_Time is
   begin
      Disabled := True;
   end Disable_Execution_Time;

   ----------------------------
   -- Global_Interrupt_Clock --
   ----------------------------

   function Global_Interrupt_Clock return System.BB.Time.Time is
      Sum : System.BB.Time.Time := System.BB.Time.Time'First;

   begin
      for Interrupt in Interrupt_ID loop
         Sum := Sum + Interrupts_Execution_Time (Interrupt);
      end loop;

      return Sum;
   end Global_Interrupt_Clock;

   ---------------------
   -- Interrupt_Clock --
   ---------------------

   function Interrupt_Clock
     (Interrupt : Interrupt_ID) return System.BB.Time.Time is
   begin
      return Interrupts_Execution_Time (Interrupt);
   end Interrupt_Clock;

   ----------------------
   -- Scheduling_Event --
   ----------------------

   procedure Scheduling_Event is
      Now            : constant System.BB.Time.Time := Read_Execution_Clock;
      Last_CPU_Clock : constant System.BB.Time.Time := CPU_Clock;
      Elapsed_Time   : System.BB.Time.Time;

   begin
      pragma Assert (Now >= Last_CPU_Clock);

      Elapsed_Time := Now - Last_CPU_Clock;

      --  Reset the clock

      CPU_Clock := Now;

      if Disabled then
         --  Discard the elapsed time and re-enable the clock

         Disabled := False;
         return;
      end if;

      if Current_Interrupt /= No_Interrupt
        and then Thread_Self.Active_Priority =
                   Board_Support.Priority_Of_Interrupt (Current_Interrupt)
      then
         --  This CPU currently executes an interrupt

         Interrupts_Execution_Time (Current_Interrupt) :=
           Interrupts_Execution_Time (Current_Interrupt) + Elapsed_Time;

      --  This CPU currently executes a task

      else
         Thread_Self.Execution_Time :=
           Thread_Self.Execution_Time + Elapsed_Time;
      end if;
   end Scheduling_Event;

   ------------------
   -- Thread_Clock --
   ------------------

   function Thread_Clock
     (Th : System.BB.Threads.Thread_Id) return System.BB.Time.Time
   is
   begin
      pragma Assert (Th /= Null_Thread_Id);

      --  If the thread Th is running, we need to add the elapsed time between
      --  the last scheduling and now.

      --  The thread Th is running if it is the current one and:
      --    * no interrupt is executed
      --    * or the interrupt triggered this thread.

      if Th = Thread_Self and then
        (Current_Interrupt = No_Interrupt
          or else Thread_Self.Active_Priority /=
                    Board_Support.Priority_Of_Interrupt (Current_Interrupt))
      then
         declare
            Now          : constant BB.Time.Time := Read_Execution_Clock;
            Elapsed_Time : constant BB.Time.Time := Now - CPU_Clock;
         begin
            return Th.Execution_Time + Elapsed_Time;
         end;

      else
         return Th.Execution_Time;
      end if;
   end Thread_Clock;

   --------------------------
   -- Read_Execution_Clock --
   --------------------------

   function Read_Execution_Clock return System.BB.Time.Time is
      XM_EXEC_CLOCK : constant := 1;
      --  Execution-time clock

      type XM_Time_T is range -2 ** 63 .. 2 ** 63 - 1;
      for XM_Time_T'Size use 64;
      --  Time in XtratuM

      procedure Get_Time
         (Clock_Id : Interfaces.C.unsigned; Time : access XM_Time_T);
      pragma Import (C, Get_Time, "XM_get_time");
      --  Read clock

      XtratuM_Time : aliased XM_Time_T;

   begin
      --  Get the execution time and not the wall time because we need to take
      --  into account only the time when the partition is active.

      Get_Time (XM_EXEC_CLOCK, XtratuM_Time'Access);

      return System.BB.Time.Time (XtratuM_Time);
   end Read_Execution_Clock;

end System.BB.Execution_Time;
