------------------------------------------------------------------------------
--                                                                          --
--                  GNAT RUN-TIME LIBRARY (GNARL) COMPONENTS                --
--                                                                          --
--     S Y S T E M . T A S K _ P R I M I T I V E S . O P E R A T I O N S    --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                     Copyright (C) 2001-2009, AdaCore                     --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
------------------------------------------------------------------------------

--  This is the VxWorks Ravenscar Cert version of this package

--  This package contains all the GNULL primitives that interface directly
--  with the underlying OS.

with System.Init;

with System.OS_Interface;

with System.Tasking.Debug;

with System.VxWorks.Ext;

with Interfaces.C;

with Unchecked_Conversion;

package body System.Task_Primitives.Operations is

   use System.Tasking;
   use System.OS_Interface;
   use System.Parameters;
   use type System.VxWorks.Ext.t_id;
   use type Interfaces.C.int;

   ----------------
   -- Local Data --
   ----------------

   Low_Priority : constant := 255;
   --  VxWorks native (default) lowest scheduling priority

   type Set_Stack_Limit_Proc_Acc is access procedure;
   pragma Convention (C, Set_Stack_Limit_Proc_Acc);

   Set_Stack_Limit_Hook : Set_Stack_Limit_Proc_Acc;
   pragma Import (C, Set_Stack_Limit_Hook, "__gnat_set_stack_limit_hook");
   --  Procedure to be called when a task is created to set stack limit if
   --  limit checking is used.

   --------------------
   -- Local Packages --
   --------------------

   package Specific is

      procedure Initialize;
      pragma Inline (Initialize);
      --  Initialize task specific data

      function Is_Valid_Task return Boolean;
      pragma Inline (Is_Valid_Task);
      --  Does executing thread have a TCB?

      procedure Set (Self_Id : Task_Id);
      pragma Inline (Set);
      --  Set the self id for the current task

      procedure Delete;
      pragma Inline (Delete);
      --  Delete the task specific data associated with the current task

      function Self return Task_Id;
      pragma Inline (Self);
      --  Return a pointer to the Ada Task Control Block of the calling task

   private
      pragma Unreferenced (Initialize, Is_Valid_Task, Delete);
   end Specific;

   package body Specific is separate;
   --  The body of this package is target specific

   function To_VxWorks_Priority
     (Priority : System.OS_Interface.int) return System.OS_Interface.int;
   pragma Inline (To_VxWorks_Priority);
   --  Convert between VxWorks and Ada priority

   function To_Ada_Priority
     (Priority : System.OS_Interface.int) return System.Any_Priority;
   pragma Inline (To_Ada_Priority);
   --  Convert between Ada priority and VxWorks priority

   -------------------------
   -- To_VxWorks_Priority --
   -------------------------

   function To_VxWorks_Priority
     (Priority : System.OS_Interface.int) return System.OS_Interface.int is
   begin
      return Low_Priority - Priority;
   end To_VxWorks_Priority;

   ---------------------
   -- To_Ada_Priority --
   ---------------------

   function To_Ada_Priority
     (Priority : System.OS_Interface.int) return System.Any_Priority is
   begin
      return System.Any_Priority (Low_Priority - Priority);
   end To_Ada_Priority;

   -----------
   -- Sleep --
   -----------

   procedure Sleep (Self_ID : Task_Id; Reason : System.Tasking.Task_States) is
      pragma Warnings (Off, Reason);

      Result : System.OS_Interface.int;

   begin
      --  Perform a blocking operation to take the CV semaphore

      Result := semTake (Self_ID.Common.LL.CV, WAIT_FOREVER);
      pragma Assert (Result = 0);
   end Sleep;

   -----------------
   -- Delay_Until --
   -----------------

   procedure Delay_Until (Abs_Time : Time) is
      Self_ID  : constant Task_Id := Specific.Self;
      Ticks    : int;
      Timedout : Boolean := False;
      Result   : int;
      Wakeup   : Boolean := False;

   begin
      Ticks    := To_Clock_Ticks (Abs_Time - Monotonic_Clock);

      if Ticks > 0 then
         loop
            --  Perform a blocking operation to take the CV semaphore. Note
            --  that a blocking operation in VxWorks will reenable task
            --  scheduling. When we are no longer blocked and control is
            --  returned, task scheduling will again be disabled.

            Result := semTake (Self_ID.Common.LL.CV, Ticks);

            if Result = 0 then

               --  Somebody may have called Wakeup for us

               Wakeup := True;

            else
               if errno /= S_objLib_OBJ_TIMEOUT then
                  Wakeup := True;

               else
                  --  If Ticks = int'last, it was most probably truncated so
                  --  let's make another round after recomputing Ticks from
                  --  the absolute time.

                  if Ticks /= int'Last then
                     Timedout := True;

                  else
                     Ticks := To_Clock_Ticks (Abs_Time - Monotonic_Clock);

                     if Ticks < 0 then
                        Timedout := True;
                     end if;
                  end if;
               end if;
            end if;

            exit when Timedout or else Wakeup;
         end loop;

      else
         Timedout := True;

         --  Should never hold a lock while yielding

         taskDelay (0);
      end if;
   end Delay_Until;

   ---------------------
   -- Monotonic_Clock --
   ---------------------

   function Monotonic_Clock return Time is
      TS     : aliased timespec;
      Result : int;
   begin
      Result := clock_gettime (CLOCK_REALTIME, TS'Unchecked_Access);
      pragma Assert (Result = 0);
      return Time (To_Duration (TS));
   end Monotonic_Clock;

   -------------------
   -- RT_Resolution --
   -------------------

   function RT_Resolution return Time is
      use Interfaces;

      Ticks_Per_Second : constant Unsigned_64
        := Unsigned_64 (sysClkRateGet);

      function To_Duration is new Unchecked_Conversion
        (Unsigned_64, Duration);

   begin
      return Time (1.0 / (To_Duration (Ticks_Per_Second)));
   end RT_Resolution;

   ------------
   -- Wakeup --
   ------------

   procedure Wakeup (T : Task_Id; Reason : System.Tasking.Task_States) is
      pragma Warnings (Off, Reason);
      Result : System.OS_Interface.int;
   begin
      Result := semGive (T.Common.LL.CV);
      pragma Assert (Result = 0);
   end Wakeup;

   ------------------
   -- Set_Priority --
   ------------------

   procedure Set_Priority
     (T    : Task_Id;
      Prio : System.Any_Priority)
   is
      Result : System.OS_Interface.int;
   begin
      Result := taskPrioritySet
        (T.Common.LL.Thread, To_VxWorks_Priority
           (System.OS_Interface.int (Prio)));
      pragma Assert (Result = 0);
   end Set_Priority;

   ------------------
   -- Get_Priority --
   ------------------

   function Get_Priority (T : Task_Id) return System.Any_Priority is
      Result           : System.OS_Interface.int;
      VxWorks_Priority : aliased System.OS_Interface.int;
   begin
      Result := taskPriorityGet (T.Common.LL.Thread, VxWorks_Priority'Access);
      pragma Assert (Result = 0);
      return To_Ada_Priority (VxWorks_Priority);
   end Get_Priority;

   ----------------
   -- Enter_Task --
   ----------------

   procedure Enter_Task (Self_ID : Task_Id) is
      procedure Init_Float;
      pragma Import (C, Init_Float, "__gnat_init_float");
      --  Properly initializes the FPU for PPC systems

      Result : System.OS_Interface.int;
      pragma Unreferenced (Result);
   begin
      Self_ID.Common.LL.Thread := taskIdSelf;
      Specific.Set (Self_ID);
      Init_Float;
      System.Init.Install_Handler;

      --  Register the task to System.Tasking.Debug

      System.Tasking.Debug.Add_Task_Id (Self_ID);

      --  If stack checking is enabled and limit checking is used, set the
      --  stack limit for this task.

      if Set_Stack_Limit_Hook /= null then
         Set_Stack_Limit_Hook.all;
      end if;
   end Enter_Task;

   --------------------
   -- Initialize_TCB --
   --------------------

   procedure Initialize_TCB (Self_ID : Task_Id; Succeeded : out Boolean) is
   begin
      Self_ID.Common.LL.CV := semBCreate (SEM_Q_PRIORITY, SEM_EMPTY);
      Succeeded := (if Self_ID.Common.LL.CV = 0 then False else True);
   end Initialize_TCB;

   -----------------
   -- Create_Task --
   -----------------

   procedure Create_Task
     (T          : Task_Id;
      Wrapper    : System.Address;
      Stack_Size : System.Parameters.Size_Type;
      Priority   : System.Any_Priority;
      Succeeded  : out Boolean)
   is
      Adjusted_Stack_Size : System.OS_Interface.size_t;

      function Get_Task_Options return int;
      pragma Import (C, Get_Task_Options, "__gnat_get_task_options");
      --  Function that returns the options to be set for the task that we
      --  are creating. We fetch the options assigned to the current task,
      --  so offering some user level control over the options for a task
      --  hierarchy, and force VX_FP_TASK because it is almost always
      --  required.

   begin
      --  Add ~1/4 to requested stack size for secondary stack
      if Stack_Size = Unspecified_Size then
         Adjusted_Stack_Size :=
           System.OS_Interface.size_t ((Default_Stack_Size * 5) / 4);
      elsif Stack_Size < Minimum_Stack_Size then
         Adjusted_Stack_Size :=
           System.OS_Interface.size_t ((Minimum_Stack_Size * 5) / 4);
      else
         Adjusted_Stack_Size :=
           System.OS_Interface.size_t ((Stack_Size * 5) / 4);
      end if;

      T.Common.LL.Thread := taskSpawn
        (System.Null_Address,
         To_VxWorks_Priority (System.OS_Interface.int (Priority)),
         Get_Task_Options,
         Adjusted_Stack_Size,
         Wrapper,
         To_Address (T));
      Succeeded := T.Common.LL.Thread /= -1;
   end Create_Task;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize (Environment_Task : System.Tasking.Task_Id) is
   begin
      Enter_Task (Environment_Task);
   end Initialize;

   ---------------------
   -- Is_Task_Context --
   ---------------------

   function Is_Task_Context return Boolean is
      function intContext return int;
      --  Binding to the C routine intContext. This function returns 1 only
      --  if the current execution state is in interrupt context.
      pragma Import (C, intContext, "intContext");
   begin
      return intContext /= 1;
   end Is_Task_Context;

   ----------
   -- Self --
   ----------

   function Self return Task_Id renames Specific.Self;

end System.Task_Primitives.Operations;
