------------------------------------------------------------------------------
--                                                                          --
--                  GNAT RUN-TIME LIBRARY (GNARL) COMPONENTS                --
--                                                                          --
--                  S Y S T E M . B B . P E R I P H E R A L S               --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--        Copyright (C) 1999-2002 Universidad Politecnica de Madrid         --
--             Copyright (C) 2003-2006 The European Space Agency            --
--                     Copyright (C) 2003-2009, AdaCore                     --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University.       --
-- Extensive contributions were provided by Ada Core Technologies, Inc.     --
--                                                                          --
-- The porting of GNARL to bare board  targets was initially  developed  by --
-- the Real-Time Systems Group at the Technical University of Madrid.       --
--                                                                          --
------------------------------------------------------------------------------

--  This is the LEON3 version of this package

pragma Restrictions (No_Elaboration_Code);

with System.BB.Peripherals.Registers;

package body System.BB.Peripherals is

   use type Registers.Scaler_12;
   use type Registers.Scaler_10;

   -----------------------
   -- Local Definitions --
   -----------------------

   Prescaler_Min : constant := 4;
   --  In order to obtain the highest granularity of the clock we set the
   --  minimum allowed prescaler division factor, which is 5, corresponding
   --  to a prescaler reload register value of 4.

   Periodic_Count : constant := Timer_Interval'Last - 1;
   --  Value to be loaded in the clock counter to accomplish the
   --  Clock_Interrupt_Period.
   --
   --  One is subtracted from Timer_Interval'Last because the timeout period
   --  will count an extra cycle for reloading the counter.

   subtype UART_Baudrate is Natural;
   --  The range of baudrates supported by the UART depends on the range of
   --  the scaler, which is a 12-bit value ranging from 0 to 2 ** 12 - 1, and
   --  on the system clock frequency.

   type UART_Parity is (None, Even, Odd);
   --  Parity values

   ----------------------
   -- Local Procedures --
   ----------------------

   procedure Initialize_Memory;
   pragma Inline (Initialize_Memory);
   --  Initialize the memory on the board

   procedure Initialize_Clock;
   --  Perform all the initialization related to the clock

   procedure Initialize_UART
     (Baudrate     : UART_Baudrate;
      Parity       : UART_Parity);
   --  Procedure to initialize the UART port

   --------------------------
   -- Cancel_And_Set_Alarm --
   --------------------------

   procedure Cancel_And_Set_Alarm (Ticks : Timer_Interval) renames Set_Alarm;
   --  This procedure cancel a previous alarm and set a new one.
   --  Setting a new alarm cancel the previous one in this target
   --  So Cancel_And_Set_Alarm and Set_Alarm are identical.

   --------------------------
   -- Clear_Alarm_Interupt --
   --------------------------

   procedure Clear_Alarm_Interrupt is
   begin
      --  Interrupts are cleared automatically when they are acknowledged

      null;
   end Clear_Alarm_Interrupt;

   --------------------------
   -- Clear_Clock_Interupt --
   --------------------------

   procedure Clear_Clock_Interrupt is
   begin
      --  Interrupts are cleared automatically when they are acknowledged

      null;
   end Clear_Clock_Interrupt;

   ----------------------
   -- Initialize_Board --
   ----------------------

   procedure Initialize_Board is
   begin
      --  The initialization of the LEON board consists on initializing the
      --  memory, and initializing the clock in order to have the desired
      --  granularity and range.

      Initialize_Memory;
      Initialize_Clock;
   end Initialize_Board;

   ----------------------
   -- Initialize_Clock --
   ----------------------

   procedure Initialize_Clock is
      use Registers;

      Prescaler               : constant Prescaler_Register :=
        (Value => Prescaler_Min, Reserved => (others => False));
      --  Minimum prescaler to be used to achieve best granularity

      Periodic_Mode : constant Timer_Control_Register :=
        (Enable            => True,
         Reload_Counter    => True,
         Load_Counter      => True,
         Interrupt_Enable  => True,
         Interrupt_Pending => False,
         Chain             => False,
         Debug_Halt        => False,
         Reserved => (others => False));

   begin
      --  Set the prescaler value to achieve the required granularity

      Prescaler_Reload := Prescaler;

      --  Load the counter for the real-time clock

      Timer_2_Reload := Periodic_Count;

      --  Program the timer in periodic mode to serve as a clock

      Timer_2_Control := Periodic_Mode;

      --  Enable clock interrupts

      Interrupt_Mask := Interrupt_Mask or 2**General_Purpose_Timer_2;
   end Initialize_Clock;

   ------------------------
   -- Initialize_Console --
   ------------------------

   procedure Initialize_Console is
   begin
      --  Initialize the UART as output console

      Initialize_UART (115200, None);
   end Initialize_Console;

   -----------------------
   -- Initialize_Memory --
   -----------------------

   procedure Initialize_Memory is
   begin
      --  Nothing to be done for LEON3

      null;
   end Initialize_Memory;

   ---------------------
   -- Initialize_UART --
   ---------------------

   procedure Initialize_UART
     (Baudrate     : UART_Baudrate;
      Parity       : UART_Parity)
   is
      Control : Registers.UART_Control_Register;
      Scaler  : Registers.UART_Scaler_Register;

   begin
      --  Read the Control Register

      Control := Registers.UART_Control;

      --  Skip initialization if if the UART has been already initialized
      --  (for example, by the remote monitor).

      if Control.Transmitter_Enable or Control.Receiver_Enable then
         return;
      end if;

      --  Set the UART scaler according to the baudrate given

      Scaler :=
        (UART_Scaler =>
              Registers.Scaler_12
                ((SBP.Clock_Frequency * 10 / (Baudrate * 8) - 5) / 10),
            Reserved    => (others => False));

      --  Enable TX and RX and disable interrupts

      Control :=
        (Receiver_Enable    => True,
         Transmitter_Enable => True,
         Parity_Select      => Registers.Parity_Kind'Val (0),
         Reserved_3         => (others => False),
         others             => False);

      --  Set the requested parity checking

      case Parity is
         when None =>
            null;

         when Even =>
            Control.Parity_Enable := True;
            Control.Parity_Select := Registers.Even;

         when Odd  =>
            Control.Parity_Enable := True;
            Control.Parity_Select := Registers.Odd;
      end case;

      --  Write to the Control and Scaler Registers

      Registers.UART_Control := Control;
      Registers.UART_Scaler  := Scaler;
   end Initialize_UART;

   ---------------------------
   -- Priority_Of_Interrupt --
   ---------------------------

   function Priority_Of_Interrupt
     (Level : SBP.Interrupt_Level) return System.Any_Priority
   is
   begin
      --  Assert that it is a real interrupt

      pragma Assert (Level /= 0);

      return (Any_Priority (Level) + Interrupt_Priority'First - 1);
   end Priority_Of_Interrupt;

   ----------------
   -- Read_Clock --
   ----------------

   function Read_Clock return Timer_Interval is
   begin
      return Periodic_Count - Registers.Timer_2_Counter;
   end Read_Clock;

   ---------------
   -- Set_Alarm --
   ---------------

   procedure Set_Alarm (Ticks : Timer_Interval) is
      use Registers;
      One_Shot_Mode : constant Timer_Control_Register :=
        (Enable            => True,
         Reload_Counter    => False,
         Load_Counter      => True,
         Interrupt_Enable  => True,
         Interrupt_Pending => False,
         Chain             => False,
         Debug_Halt        => False,
         Reserved => (others => False));

   begin
      --  Alarm Clock downcount will reach 0 in Ticks. The granularity of
      --  time intervals is equal to Clock Period.

      --  Set the prescaler: already done in Initialize_Clock

      --  Load the counter

      Timer_1_Reload := Ticks;

      --  Program the timer in one-shot mode

      Timer_1_Control := One_Shot_Mode;

      --   Enable Timer 1 Interrupts

      Interrupt_Mask := Interrupt_Mask or 2**General_Purpose_Timer_1;
   end Set_Alarm;

   ----------------------
   -- Ticks_Per_Second --
   ----------------------

   function Ticks_Per_Second return Natural is
   begin
      --  The prescaler is clocked by the system clock. When it underflows, it
      --  is reloaded from the prescaler reload register and a timer tick is
      --  generated. The effective division rate is therefore equal to the
      --  prescaler reload register value plus 1.

      return System.BB.Parameters.Clock_Frequency / (Prescaler_Min + 1);
   end Ticks_Per_Second;

   ------------------
   -- To_Interrupt --
   ------------------

   function To_Interrupt
     (Vector : SBP.Range_Of_Vector) return SBP.Interrupt_Level
   is
   begin
      --  The range corresponding to asynchronous traps is between
      --  16#11# and 16#1F#.

      pragma Assert (Vector >= 16#11# and then Vector <= 16#1F#);

      return Vector - 16#10#;
   end To_Interrupt;

   ---------------
   -- To_Vector --
   ---------------

   function To_Vector (Level : SBP.Interrupt_Level) return SBP.Range_Of_Vector
   is
   begin
      return Level + 16#10#;
   end To_Vector;

   ------------------
   -- Console_Send --
   ------------------

   procedure Console_Send (Char    : Character)
   is
      UART_Tx         : constant Registers.UART_Data_Register :=
        (FIFO => Char, Reserved => (others => False));

   begin
      --  Send the character through the selected channel by polling
      --  Wait until the port is ready to load data

      loop
         exit when not Registers.UART_Status.Transmitter_FIFO_Full;
      end loop;

      --  Send the character once the UART is ready

      Registers.UART_Data := UART_Tx;

   end Console_Send;

end System.BB.Peripherals;
