------------------------------------------------------------------------------
--                                 Ada2Java                                 --
--                                                                          --
--                     Copyright (C) 2007-2014, AdaCore                     --
--                                                                          --
-- This is free software;  you can redistribute it  and/or modify it  under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  This software is distributed in the hope  that it will be useful, --
-- but WITHOUT ANY WARRANTY;  without even the implied warranty of MERCHAN- --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General  Public  License  distributed  with  this  software;   see  file --
-- COPYING3.  If not, go to http://www.gnu.org/licenses for a complete copy --
-- of the license.                                                          --
------------------------------------------------------------------------------

--  This is a placeholder for the routine that scans tool parameters.

with Ada.Strings.Wide_Unbounded; use Ada.Strings.Wide_Unbounded;
with GNAT.Command_Line;

with ASIS_UL.Options;  use ASIS_UL.Options;

with Ada2Java;         use Ada2Java;
with Ada2Java.Kernel;  use Ada2Java.Kernel;
with Ada2Java.Projects;

with Gnatvsn; use Gnatvsn;

with Switch;

separate (ASIS_UL.Environment)
procedure Scan_Parameters is
   No_More_Args : Boolean := True;

   procedure Print_Version_Info;
   procedure Print_Help;

   procedure Print_Version_Info is
   begin
      Info_No_EOL ("ADA2JAVA (built with ");
      Info_No_EOL (To_String
                   (Asis.Implementation.ASIS_Implementor_Version));
      Info        (")");

      Info_No_EOL ("Copyright 2007-");
      Info_No_EOL (Gnatvsn.Current_Year);
      Info        (", AdaCore.");
   end Print_Version_Info;

   procedure Print_Help is
   begin
      Print_Version_Info;
      Info ("");
      Info ("Usage: ada2java {<switch> | <input-file>} <input-file> " &
            "{<switch> | <input-file>}");
      Info ("");
      Info ("  Each <input-file> must be the name of a source");
      Info ("  file for an Ada package spec (including the extension).");
      Info ("");

      Info ("general options:");
      Switch.Display_Usage_Version_And_Help;
      Info ("  -debug-glue");
      Info ("     add debug comments");
      Info ("");
      Info ("  -verbose=(silent|errors_only|errors_and_warnings|" &
            "limited_verbosity|full_verbosity|debug)");
      Info ("     output verbosity level");
      Info ("  -b name");
      Info ("     Use this name as the base of generated java packages");
      Info ("  -c dir");
      Info ("     Generate the java interface in that directory");
      Info ("  -L name");
      Info ("     Set the native library name, generate build project & " &
            "automatic load");
      Info ("  -M name");
      Info ("     Set the main unit name, generate build project for an & " &
            "Ada main");
      Info ("  -o dir");
      Info ("     Generate the ada glue code in that directory");
      Info ("  -P prj");
      Info ("     Use this input project");

      Info ("");
      Info ("generation configuration options:");
      Info ("  --[no-]assume-escaped");
      Info ("     Set default assumption for access storing");
      Info ("  --bound-package-root=<package root name>");
      Info ("     Set the name of the root glue Ada packages (default is "
            & "JNI_Binding).");
      Info ("  --bound-package-suffix=<suffix name>");
      Info ("     Set the suffix of the glue Ada packages (default is _JNI).");
      Info ("  --[no-]attach-(parameter|access|controlling|ada2005)");
      Info ("     Set attachement policy");
      Info ("  --[no-]java-enum");
      Info ("     Enable or disable use of Java enumerations");
      Info ("  --link-mode=(export|register_natives)");
      Info ("     Set discovery method of native functions by Java");
      Info ("  --library-kind=(dynamic|encapsulated)");
      Info ("     Set library generation method: normal or standalone (used by"
            & " -L)");
      Info ("  --main-class=<java class>");
      Info ("     If the -M switch is used, changes the main java class to" &
            " use");
      Info ("  --no-locking[-finalize] / "
            & "--locking[-finalize]-(check|protect)");
      Info ("     Set default locking for subprograms");
      Info ("  --lock=<lock full name>");
      Info ("     Changes the default lock to the full qualified name.");
      Info ("  --[no-]unaliased-access");
      Info ("     Set access allowance on unaliased data");

      Info ("");
   end Print_Help;
begin
   Initialize_Option_Scan;

   loop
      case GNAT.Command_Line.Getopt
        ("v verbose= h d dd q P: b: c: o: L: M:"
         & " -assume-escaped"
         & " -bound-package-root="
         & " -bound-package-suffix="
         & " -link-mode="
         & " -library-kind="
         & " -no-assume-escaped"
         & " -no-locking"
         & " -locking-check"
         & " -locking-protect"
         & " -no-locking-finalize"
         & " -locking-finalize-check"
         & " -locking-finalize-protect"
         & " -lock="
         & " -attach-parameter"
         & " -attach-access"
         & " -attach-controlling"
         & " -attach-ada2005"
         & " -no-attach-parameter"
         & " -no-attach-access"
         & " -no-attach-controlling"
         & " -no-attach-ada2005"
         & " -java-enum"
         & " -no-java-enum"
         & " -unaliased-access"
         & " -no-unaliased-access"
         & " -main-class="
         & " -trace-checks"
         & " -trace-null-conversions"
         & " -debug-glue"
         & " -version"
         & " -help")
      is

         when ASCII.NUL =>
            exit;

         when 'h' =>
            Print_Help;

         when 'v' =>
            if Full_Switch = "v" then
               Print_Version_Info;
            else
               Verbose_Mode := True;

               if Parameter = "" then
                  null;
               elsif Parameter = "silent" then
                  Current_Trace_Level := Silent;
               elsif Parameter = "errors_only" then
                  Current_Trace_Level := Errors_Only;
               elsif Parameter = "errors_and_warnings" then
                  Current_Trace_Level :=  Errors_And_Warnings;
               elsif Parameter = "limited_verbosity" then
                  Current_Trace_Level := Limited_Verbosity;
               elsif Parameter = "full_verbosity" then
                  Current_Trace_Level := Full_Verbosity;
               elsif Parameter = "debug" then
                  Current_Trace_Level := Ada2Java.Debug;
               end if;
            end if;

         when 'q' =>
            Quiet_Mode := True;

         when 'd' =>
            if Full_Switch = "d" then
               Debug_Mode := True;
            elsif Full_Switch = "dd" then
               Progress_Indicator_Mode := True;
            end if;

         when 'P' =>
            Project_File := new String'(Parameter);
            --  We do not store the '-P' option here, because we need the
            --  project file name with full path, we will do this in
            --  Check_Parameters later on/

         when 'b' =>
            Default_Base_Package := To_Unbounded_Wide_String
              (To_Wide_String (Parameter));

         when 'c' =>
            if Is_Absolute_Path (Parameter) then
               Default_Class_Path := To_Unbounded_Wide_String
                 (To_Wide_String (Parameter));
            else
               Default_Class_Path := To_Unbounded_Wide_String
                 (To_Wide_String
                    (Get_Current_Dir & Directory_Separator & Parameter));
            end if;

         when 'o' =>
            if Is_Absolute_Path (Parameter) then
               Default_Ada_Base_Folder := To_Unbounded_Wide_String
                 (To_Wide_String (Parameter));
            else
               Default_Ada_Base_Folder := To_Unbounded_Wide_String
                 (To_Wide_String
                    (Get_Current_Dir & Directory_Separator & Parameter));
            end if;

         when 'L' =>
            Ada2Java.Native_Library_Name := new String'(Parameter);

         when 'M' =>
            Ada2Java.Main_Unit_Name := new String'(Parameter);
            Ada2Java.Link_Method := Register_Natives;

         when '-' =>
            if Full_Switch = "-assume-escaped" then
               Ada2Java.Default_Assumed_Stored := True;
            elsif Full_Switch = "-no-assume-escaped" then
               Ada2Java.Default_Assumed_Stored := False;
            elsif Full_Switch = "-link-mode" then
               Ada2Java.Link_Method := Link_Method_Type'Value (Parameter);
            elsif Full_Switch = "-library-kind" then
               Ada2Java.Library_Kind := Library_Kind_Type'Value (Parameter);
            elsif Full_Switch = "-no-locking" then
               Ada2Java.Default_Locking_State := Disable;
            elsif Full_Switch = "-locking-check" then
               Ada2Java.Default_Locking_State := Check;
            elsif Full_Switch = "-locking-protect" then
               Ada2Java.Default_Locking_State := Protect;
            elsif Full_Switch = "-no-locking-finalize" then
               Ada2Java.Default_Finalize_Locking_State := Disable;
            elsif Parameter = "-locking-finalize-check" then
               Ada2Java.Default_Finalize_Locking_State := Check;
            elsif Full_Switch = "-locking-finalize-protect" then
               Ada2Java.Default_Finalize_Locking_State := Protect;
            elsif Full_Switch = "-lock" then
               Ada2Java.Default_Lock := new Wide_String'
                 (Ada.Characters.Conversions.To_Wide_String (Parameter));
            elsif Full_Switch = "-attach-parameter" then
               Ada2Java.Default_Attachement := True;
               Ada2Java.Default_Access_Attachement := True;
               Ada2Java.Default_Controlling_Attachement := True;
               Ada2Java.Default_Ada_2005_Attachement := True;
            elsif Full_Switch = "-attach-access" then
               Ada2Java.Default_Access_Attachement := True;
            elsif Full_Switch = "-attach-controlling" then
               Ada2Java.Default_Controlling_Attachement := True;
            elsif Full_Switch = "-attach-ada2005" then
               Ada2Java.Default_Ada_2005_Attachement := True;
            elsif Full_Switch = "-no-attach-parameter" then
               Ada2Java.Default_Attachement := False;
               Ada2Java.Default_Access_Attachement := False;
               Ada2Java.Default_Controlling_Attachement := False;
               Ada2Java.Default_Ada_2005_Attachement := False;
            elsif Full_Switch = "-no-attach-access" then
               Ada2Java.Default_Access_Attachement := False;
            elsif Full_Switch = "-no-attach-controlling" then
               Ada2Java.Default_Controlling_Attachement := False;
            elsif Full_Switch = "-no-attach-ada2005" then
               Ada2Java.Default_Ada_2005_Attachement := False;
            elsif Full_Switch = "-java-enum" then
               Ada2Java.Use_Java_1_5_Enums := True;
            elsif Full_Switch = "-no-java-enum" then
               Ada2Java.Use_Java_1_5_Enums := False;
            elsif Full_Switch = "-unaliased-access" then
               Ada2Java.Allow_Unaliased_Access := True;
            elsif Full_Switch = "-no-unaliased-access" then
               Ada2Java.Allow_Unaliased_Access := False;
            elsif Full_Switch = "-main-class" then
               Ada2Java.Java_Main_Class := new String'(Parameter);
            elsif Full_Switch = "-bound-package-root" then
               Ada2Java.Bound_Package_Root :=
                 new Wide_String'(To_Wide_String (Parameter));
            elsif Full_Switch = "-bound-package-suffix" then
               Ada2Java.Bound_Package_Suffix :=
                 new Wide_String'(To_Wide_String (Parameter));
            elsif Full_Switch = "-trace-checks" then
               Ada2Java.Trace_Checks := True;
            elsif Full_Switch = "-trace-null-conversions" then
               Ada2Java.Trace_Null_Conversions := True;
            elsif Full_Switch = "-debug-glue" then
               Ada2Java.Debug_Glue := True;
            elsif GNAT.Command_Line.Full_Switch = "-version" then
               Print_Version_Info;
               OS_Exit (0);
            elsif GNAT.Command_Line.Full_Switch = "-help" then
               Print_Help;
               OS_Exit (0);
            end if;

         when others =>
            raise Parameter_Error;
      end case;
   end loop;

   Trace ("Scan parameters", Ada2Java.Debug);

   --  --8<--8<--8<--8<--8<--8<--8<--8<--8<--8<--8<--8<
   --  Should be used if the tool may have more then one source file as
   --  its argument
   loop
      Add_Source_To_Process
        (Get_Argument, No_More_Args, Ada2Java.Projects.Ada2Java_Prj);
      exit when No_More_Args;
   end loop;
   --  -->8-->8-->8-->8-->8-->8-->8-->8-->8-->8-->8-->8

   Process_cargs_Section;

exception
   when GNAT.Command_Line.Invalid_Switch =>
      Error ("invalid switch : " & Full_Switch);
      raise Parameter_Error;

   when GNAT.Command_Line.Invalid_Parameter =>
      Error ("parameter missed for : -" & Full_Switch);
      raise Parameter_Error;

end Scan_Parameters;
